#include <infra/netmon/probe_slice_maintainer.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/library/boxes.h>
#include <infra/netmon/library/settings.h>
#include <infra/netmon/settings.h>

#include <infra/netmon/library/clickhouse/client.h>
#include <infra/netmon/library/clickhouse/translate.h>

#include <library/cpp/consistent_hashing/consistent_hashing.h>

#include <util/datetime/cputimer.h>
#include <util/stream/output.h>
#include <util/string/builder.h>
#include <util/string/subst.h>
#include <util/system/info.h>

namespace NNetmon {
    namespace {
        const char* DATETIME_FORMAT = "%Y-%m-%d %H:%M:%S";

        const char* PROBE_FETCH_QUERY = R"(
SELECT
    SourceFqdn,
    TargetFqdn,
    SourceAddress,
    TargetAddress,
    SourcePort,
    TargetPort,
    Network,
    Protocol,
    toUInt64(Generated),
    Score,
    Rtt
FROM {tableName}
WHERE (
    Inserted >= toDateTime('{insertedSince}')
    AND InsertedDate >= toDate('{insertedSince}')
    AND Inserted <= toDateTime('{insertedUntil}')
    AND InsertedDate <= toDate('{insertedUntil}')
    AND Score >= 0
)
)";

        const std::size_t SHARD_COUNT = Max(TLibrarySettings::Get()->GetCpuGuarantee() / 2UL, 2UL);

        class TProbeInserter: public TOffloadedTask {
        public:
            class TExecutor: public TCustomThreadExecutor<TExecutor> {
            public:
                TExecutor()
                    : TCustomThreadExecutor("ProbeInserter", SHARD_COUNT)
                {
                }
            };

            void operator()() override {
                TUnistatTimer timer{TUnistat::Instance(), ENetmonSignals::ProbeInsertionTime};
                Slice.InsertProbes(std::move(Probes), TInstant::Now());
            }

            static inline TThreadPool::TFuture Enqueue(TProbeSlice& slice, TProbe::TRefVector&& probes) {
                return TExecutor::Get()->Add(THolder<TProbeInserter>(new TProbeInserter(slice, std::move(probes))));
            }

        private:
            TProbeInserter(TProbeSlice& slice, TProbe::TRefVector&& probes)
                : Slice(slice)
                , Probes(std::move(probes))
            {
            }

            TProbeSlice& Slice;
            TProbe::TRefVector Probes;
        };

        inline TProbeSliceKey CreateSliceKey(ui64 network, ui64 protocol, ui64 shardIndex) {
            return TProbeSliceKey{
                static_cast<ENetworkType>(network),
                static_cast<EProtocolType>(protocol),
                shardIndex};
        }
    }

    class TProbeSliceMaintainer::TImpl : public TScheduledTask {
    public:
        // SourceFqdn, TargetFqdn, SourceAddress, TargetAddress,
        // SourcePort, TargetPort, Network, Protocol,
        // Generated, Score, Rtt
        using TRow = std::tuple<
            ui64, ui64, TString, TString,
            ui16, ui16, ui8, ui8,
            ui64, double, double>;

        class TSeenHostsRefresher : public TScheduledTask {
        public:
            TSeenHostsRefresher(const TImpl* parent)
                : TScheduledTask(TSettings::Get()->GetSeenHostsInterval())
                , Parent(parent)
            {
            }

            TThreadPool::TFuture Run() override {
                return TThreadPool::Get()->Add([this]() {
                    auto seenHosts(TCountedHostStorage::TState::Make(*SeenHosts.Own()));
                    TSimpleTimer timer;
                    for (const auto& slice : Parent->GetSliceList()) {
                        const auto changeSet(slice->GetProbeStorage()->GetSeenHosts().ResetChangeSet());
                        seenHosts->Apply(changeSet);
                    }
                    DEBUG_LOG << "Seen hosts refreshing took " << timer.Get() << Endl;
                    SeenHosts.Swap(seenHosts);
                });
            }

            inline TTopologyStorage::THostIdSet Get() const {
                return SeenHosts.Own()->GetHosts();
            }

        private:
            const TImpl* Parent;
            TCountedHostStorage::TState::TBox SeenHosts;
        };

        TImpl(const TTopologyStorage& topologyStorage,
              const TSeenHostsUpdater& seenHostsUpdater,
              const TTerminatedHostsMaintainer& terminatedHostsMaintainer,
              const TWalleUpdater& walleUpdater,
              bool schedule)
            : TScheduledTask(TSettings::Get()->GetProbeFetchingInterval())
            , TopologyStorage(topologyStorage)
            , SeenHostsUpdater(seenHostsUpdater)
            , TerminatedHostsMaintainer(terminatedHostsMaintainer)
            , WalleUpdater(walleUpdater)
            , Scheduled(schedule)
            , IntervalStart(TInstant::Now() - TSettings::Get()->GetProbeFetchingForwardWindow())
            , IntervalEnd(IntervalStart)
            , SeenHostsRefresher(this)
            , SeenHostsRefresherGuard(SeenHostsRefresher.Schedule())
        {
        }

        ~TImpl() {
            auto guard = TGuard<TAdaptiveLock>(SliceLock);
            while (!SliceTree.Empty()) {
                THolder<TProbeSlice> aggregator(&(*SliceTree.Begin()));
            }
        }

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                TUnistatTimer timer{TUnistat::Instance(), ENetmonSignals::ProbeReadingTime};
                MaintainForwardWindow();
                if (IntervalStart > TInstant::Now() - TSettings::Get()->GetSwitchAggregationWindow()) {
                    MaintainBackwardWindow();
                }
            });
        }

        TProbeSlice::TRefVector GetSliceList() const {
            auto guard = TGuard<TAdaptiveLock>(SliceLock);
            TProbeSlice::TRefVector sliceList;
            TProbeSlice::TTree::TConstIterator it(SliceTree.Begin());
            for (; it != SliceTree.End(); ++it) {
                sliceList.push_back(&(*it));
            }
            return sliceList;
        }

        TProbeSlice::TRefVector GetSliceList(const TProbeSectionKey& section) const {
            auto guard = TGuard<TAdaptiveLock>(SliceLock);
            TProbeSlice::TRefVector sliceList;
            TProbeSlice::TTree::TConstIterator it(SliceTree.LowerBound(section));
            for (; it != SliceTree.End() && it->SameSection(section); ++it) {
                sliceList.push_back(&(*it));
            }
            return sliceList;
        }

        TProbeGatherer::TRef CreateProbeGatherer(const TProbeAggregatorKey& key) const {
            const TProbeSectionKey section(key.GetSectionKey());
            return TProbeGatherer::Make(GetSliceList(section), key.GetExpressionId(), TopologyStorage);
        }

        TTopologyStorage::THostIdSet GetPartialSeenHosts() const {
            return SeenHostsRefresher.Get();
        }

        bool IsReady() const {
            if (!Scheduled) {
                return true;
            }

            auto now(TInstant::Now());
            return (
                IntervalEnd > now - TSettings::Get()->GetProbeFetchingForwardWindow() * 2
                && IntervalStart <= now - TSettings::Get()->GetSwitchAggregationWindow());
        }

    private:
        using TProbeMap = THashMap<TProbeSliceKey, TProbe::TRefVector>;

        TProbeSlice& GetOrCreateSlice(const TProbeSliceKey &key) {
            auto guard = TGuard<TAdaptiveLock>(SliceLock);
            auto* foundSlice = SliceTree.Find(key);
            if (foundSlice != nullptr) {
                return *foundSlice;
            }
            THolder<TProbeSlice> slice(
                MakeHolder<TProbeSlice>(key, TopologyStorage, SeenHostsUpdater, TerminatedHostsMaintainer, WalleUpdater));
            SliceTree.Insert(slice.Get());
            return *slice.Release();
        }

        void MaintainForwardWindow() {
            auto now(TInstant::Now());
            auto sinceTime = now - TSettings::Get()->GetProbeFetchingForwardWindow();
            sinceTime = Max(Min(IntervalEnd, sinceTime), TInstant::Now() - TSettings::Get()->GetProbeFetchingBackwardWindow());
            LoadProbes(sinceTime, now);
            IntervalEnd = now;
        }

        void MaintainBackwardWindow() {
            auto untilTime = Max(IntervalStart, TInstant::Now() - TSettings::Get()->GetSwitchAggregationWindow());
            auto sinceTime = untilTime - TSettings::Get()->GetProbeFetchingBackwardWindow();
            LoadProbes(sinceTime, untilTime);
            IntervalStart = sinceTime;
        }

        void LoadProbes(const TInstant& sinceTime, const TInstant& untilTime) {
            TSimpleTimer timer;

            struct tm sinceTs, untilTs;
            sinceTime.LocalTime(&sinceTs);
            untilTime.LocalTime(&untilTs);

            TString query(PROBE_FETCH_QUERY);
            SubstGlobal(query, "{tableName}", "probes");
            SubstGlobal(query, "{insertedSince}", Strftime(DATETIME_FORMAT, &sinceTs));
            SubstGlobal(query, "{insertedUntil}", Strftime(DATETIME_FORMAT, &untilTs));

            auto probeMap(MakeAtomicShared<TProbeMap>());
            TClickhouseClient::TQueryOptions options(query);
            Futures.emplace_back(TClickhouseClient::Get()->Select(options, [this, probeMap] (const NClickHouse::TBlock& block) {
                const TVector<TRow> rows(TranslateClickhouseBlock<TRow>(block));
                GroupProbes(rows, *probeMap);
            }));

            // wait for probe insertion from previous iteration
            try {
                NThreading::WaitExceptionOrAll(Futures).GetValue(TDuration::Max());
            } catch (...) {
                ERROR_LOG << CurrentExceptionMessage() << Endl;
            }
            Futures.clear();

            for (auto& pair : *probeMap) {
                Futures.emplace_back(TProbeInserter::Enqueue(GetOrCreateSlice(pair.first), std::move(pair.second)));
            }

            INFO_LOG << "Probe reading from " << sinceTime << " to " << untilTime << " took " << timer.Get() << Endl;
        }

        void GroupProbes(const TVector<TRow>& rows, TProbeMap& probeMap) {
            for (const auto& row : rows) {
                const auto sourceIface = TopologyStorage.FindHostInterface(std::get<0>(row));
                const auto targetIface = TopologyStorage.FindHostInterface(std::get<1>(row));
                if (!sourceIface || !targetIface) {
                    continue;
                }

                ui64 shardIndex = ConsistentHashing(
                    MultiHash(
                        sourceIface->GetSwitch().GetHash(),
                        targetIface->GetSwitch().GetHash()
                    ),
                    SHARD_COUNT
                );
                const TProbeSliceKey sliceKey(CreateSliceKey(std::get<6>(row), std::get<7>(row), shardIndex));

                const TIpAddress sourceAddress(std::get<2>(row));
                const TIpAddress targetAddress(std::get<3>(row));

                probeMap[sliceKey].emplace_back(TProbe::Make(
                    sourceIface,
                    targetIface,
                    sourceAddress,
                    targetAddress,
                    std::get<4>(row),  // source port
                    std::get<5>(row),  // target port
                    TInstant::Seconds(std::get<8>(row)),  // generated
                    std::get<9>(row),  // score
                    std::get<10>(row) > 0.0 ? std::get<10>(row) * 1000.0 : -1.0  // rtt
                ));
            }
        }

        const TTopologyStorage& TopologyStorage;
        const TSeenHostsUpdater& SeenHostsUpdater;
        const TTerminatedHostsMaintainer& TerminatedHostsMaintainer;
        const TWalleUpdater& WalleUpdater;
        bool Scheduled;

        TInstant IntervalStart;
        TInstant IntervalEnd;

        TProbeSlice::TTree SliceTree;
        TAdaptiveLock SliceLock;
        TVector<TThreadPool::TFuture> Futures;

        TSeenHostsRefresher SeenHostsRefresher;
        TScheduledTask::TTaskGuard SeenHostsRefresherGuard;
    };

    TProbeSliceMaintainer::TProbeSliceMaintainer(const TTopologyStorage& topologyStorage,
                                                 const TSeenHostsUpdater& seenHostsUpdater,
                                                 const TTerminatedHostsMaintainer& terminatedHostsMaintainer,
                                                 const TWalleUpdater& walleUpdater)
        : TProbeSliceMaintainer(topologyStorage,
                                seenHostsUpdater,
                                terminatedHostsMaintainer,
                                walleUpdater,
                                !TLibrarySettings::Get()->GetClickHouseShards().empty())
    {
    }

    TProbeSliceMaintainer::TProbeSliceMaintainer(const TTopologyStorage& topologyStorage,
                                                 const TSeenHostsUpdater& seenHostsUpdater,
                                                 const TTerminatedHostsMaintainer& terminatedHostsMaintainer,
                                                 const TWalleUpdater& walleUpdater,
                                                 bool schedule)
        : Impl(MakeHolder<TImpl>(topologyStorage, seenHostsUpdater, terminatedHostsMaintainer, walleUpdater, schedule))
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TProbeSliceMaintainer::~TProbeSliceMaintainer() {
    }

    TProbeSlice::TRefVector TProbeSliceMaintainer::GetSliceList() const {
        return Impl->GetSliceList();
    }

    TProbeSlice::TRefVector TProbeSliceMaintainer::GetSliceList(const TProbeSectionKey& section) const {
        return Impl->GetSliceList(section);
    }

    TProbeGatherer::TRef TProbeSliceMaintainer::CreateProbeGatherer(const TProbeAggregatorKey& key) const {
        return Impl->CreateProbeGatherer(key);
    }

    TTopologyStorage::THostIdSet TProbeSliceMaintainer::GetPartialSeenHosts() const {
        return Impl->GetPartialSeenHosts();
    }

    bool TProbeSliceMaintainer::IsReady() const {
        return Impl->IsReady();
    }
}
