#include <infra/netmon/probe_slice_merger.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/settings.h>

namespace NNetmon {
    namespace {
        class TSwitchConfig : public TNonCopyable {
        public:
            using TIndexMap = TSwitchIndexMap;
            using TIndexEventHub = TEventHub<TIndexMap::TIndex::TRef>;

            TSwitchConfig()
                : Executor("SwitchSliceCollapser", 1)
                , EventHub(TIndexEventHub::Make())
            {
            }

            inline TDuration GetInterval() const {
                return TSettings::Get()->GetSwitchAggregationInterval();
            }

            inline const TIndexMap::TRef ExtractIndexMap(const TProbeSlice& slice) const {
                return slice.GetSwitchLevelIndex();
            }

            inline TBaseThreadExecutor& GetExecutor() const {
                return Executor;
            }

            inline const TIndexEventHub& OnUpdated() const {
                return *EventHub;
            }

        private:
            mutable TBaseThreadExecutor Executor;
            TIndexEventHub::TRef EventHub;
        };

        class TLineConfig : public TNonCopyable {
        public:
            using TIndexMap = TLineIndexMap;
            using TIndexEventHub = TEventHub<TIndexMap::TIndex::TRef>;

            TLineConfig()
                : Executor("LineSliceCollapser", 1)
                , EventHub(TIndexEventHub::Make())
            {
            }

            inline TDuration GetInterval() const {
                return TSettings::Get()->GetLineAggregationInterval();
            }

            inline const TIndexMap::TRef ExtractIndexMap(const TProbeSlice& slice) const {
                return slice.GetLineLevelIndex();
            }

            inline TBaseThreadExecutor& GetExecutor() const {
                return Executor;
            }

            inline const TIndexEventHub& OnUpdated() const {
                return *EventHub;
            }

        private:
            mutable TBaseThreadExecutor Executor;
            TIndexEventHub::TRef EventHub;
        };

        class TDatacenterConfig : public TNonCopyable {
        public:
            using TIndexMap = TDatacenterIndexMap;
            using TIndexEventHub = TEventHub<TIndexMap::TIndex::TRef>;

            TDatacenterConfig()
                : Executor("DcSliceCollapser", 1)
                , EventHub(TIndexEventHub::Make())
            {
            }

            inline TDuration GetInterval() const {
                return TSettings::Get()->GetDcAggregationInterval();
            }

            inline const TIndexMap::TRef ExtractIndexMap(const TProbeSlice& slice) const {
                return slice.GetDatacenterLevelIndex();
            }

            inline TBaseThreadExecutor& GetExecutor() const {
                return Executor;
            }

            inline const TIndexEventHub& OnUpdated() const {
                return *EventHub;
            }

        private:
            mutable TBaseThreadExecutor Executor;
            TIndexEventHub::TRef EventHub;
        };

        template <class TConfig>
        class TSliceCollapser : public TScheduledTask, public TAtomicRefCount<TSliceCollapser<TConfig>> {
        public:
            using TRef = TIntrusivePtr<TSliceCollapser>;
            using TMapType = THashMap<TProbeAggregatorKey, TRef>;

            using TIndexMap = typename TConfig::TIndexMap;
            using TIndex = typename TIndexMap::TIndex;
            using TState = typename TIndex::TState;
            using TIndexBox = TIntrusiveLockedBox<TIndex>;

            template <typename... Args>
            static inline TRef Make(Args&&... args) {
                return new TSliceCollapser(std::forward<Args>(args)...);
            }

            static inline typename TIndex::TRef Find(const TMapType& collapsers,
                                                     const TProbeAggregatorKey& key) {
                auto it(collapsers.find(key));
                if (it.IsEnd()) {
                    return TIndex::Make(key, TInstant::Now());
                } else {
                    return it->second->Get();
                }
            }

            TThreadPool::TFuture Run() override {
                return Config.GetExecutor().Add([this]() {
                    TUnistatTimer timer{TUnistat::Instance(), ENetmonSignals::MergeTime};
                    Collapse();
                });
            }

            inline typename TIndex::TRef Get() const {
                return Index.Get();
            }

        private:
            TSliceCollapser(const TConfig& config,
                            const IProbeSliceMaintainer& sliceMaintainer,
                            const TProbeAggregatorKey& key)
                : TScheduledTask(config.GetInterval())
                , Config(config)
                , SliceMaintainer(sliceMaintainer)
                , Key(key)
                , Index(TIndex::Make(Key, TInstant::Now()))
                , SchedulerGuard(Schedule())
            {
            }

            void Collapse() {
                auto index(TIndex::Make(Key, TInstant::Now()));
                const auto sliceList = SliceMaintainer.GetSliceList(Key.GetSectionKey());
                for (const auto& slice : sliceList) {
                    const auto partialIndexMap(Config.ExtractIndexMap(*slice));
                    const auto it = partialIndexMap->find(Key.GetExpressionId());
                    if (!it.IsEnd()) {
                        index->MergeFrom(*it->second);
                    }
                }
                Index.Swap(index);
                Config.OnUpdated().Notify(Get());
            }

            const TConfig& Config;
            const IProbeSliceMaintainer& SliceMaintainer;
            const TProbeAggregatorKey Key;

            mutable TIndexBox Index;

            TScheduledTask::TTaskGuard SchedulerGuard;
        };
    }

    class TProbeSliceMerger::TImpl : public TScheduledTask {
    public:
        using TDatacenterCollapser = TSliceCollapser<TDatacenterConfig>;
        using TLineCollapser = TSliceCollapser<TLineConfig>;
        using TSwitchCollapser = TSliceCollapser<TSwitchConfig>;

        TImpl(const IProbeSliceMaintainer& sliceMaintainer)
            : TScheduledTask(TSettings::Get()->GetDcAggregationInterval())
            , SliceMaintainer(sliceMaintainer)
        {
        }

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                DoRun();
            });
        }

        TSwitchPairIndex::TRef GetSwitchLevelIndex(const TProbeAggregatorKey& key) const {
            auto guard = TGuard<TAdaptiveLock>(CollapserLock);
            return TSwitchCollapser::Find(SwitchCollapsers, key);
        }

        TLinePairIndex::TRef GetLineLevelIndex(const TProbeAggregatorKey& key) const {
            auto guard = TGuard<TAdaptiveLock>(CollapserLock);
            return TLineCollapser::Find(LineCollapsers, key);
        }

        TDatacenterPairIndex::TRef GetDatacenterLevelIndex(const TProbeAggregatorKey& key) const {
            auto guard = TGuard<TAdaptiveLock>(CollapserLock);
            return TDatacenterCollapser::Find(DatacenterCollapsers, key);
        }

        TProbeAggregatorKeySet GetSwitchLevelKeys() const {
            return GetKnownKeys(SwitchConfig);
        }

        TProbeAggregatorKeySet GetLineLevelKeys() const {
            return GetKnownKeys(LineConfig);
        }

        TProbeAggregatorKeySet GetDatacenterLevelKeys() const {
            return GetKnownKeys(DatacenterConfig);
        }

        const TEventHub<TSwitchPairIndex::TRef>& OnSwitchUpdated() const {
            return SwitchConfig.OnUpdated();
        }

        const TEventHub<TLinePairIndex::TRef>& OnLineUpdated() const {
            return LineConfig.OnUpdated();
        }

        const TEventHub<TDatacenterPairIndex::TRef>& OnDatacenterUpdated() const {
            return DatacenterConfig.OnUpdated();
        }

        TThreadPool::TFuture WaitSwitchTask() {
            return WaitForTask(SwitchCollapsers);
        }

        TThreadPool::TFuture WaitQueueTask() {
            return WaitForTask(LineCollapsers);
        }

        TThreadPool::TFuture WaitDatacenterTask() {
            return WaitForTask(DatacenterCollapsers);
        }

    private:
        void DoRun() {
            auto guard = TGuard<TAdaptiveLock>(CollapserLock);

            Maintain(DatacenterConfig, DatacenterCollapsers);
            Maintain(LineConfig, LineCollapsers);
            Maintain(SwitchConfig, SwitchCollapsers);
        }

        template <class TConfig>
        TProbeAggregatorKeySet GetKnownKeys(const TConfig& config) const {
            TProbeAggregatorKeySet knownKeys;
            const auto sliceList(SliceMaintainer.GetSliceList());
            for (const auto& slice : sliceList) {
                TProbeSectionKey sectionKey(slice->GetKey().GetSectionKey());
                const auto partialIndexMap(config.ExtractIndexMap(*slice));
                for (const auto& pair : *partialIndexMap) {
                    knownKeys.emplace(TProbeAggregatorKey::FromSectionKey(sectionKey, pair.first));
                }
            }
            return knownKeys;
        }

        template <class TConfig>
        void Maintain(const TConfig& config, typename TSliceCollapser<TConfig>::TMapType& collapsers) {
            const auto knownKeys(GetKnownKeys(DatacenterConfig));
            for (const auto& key : knownKeys) {
                if (collapsers.find(key).IsEnd()) {
                    collapsers.emplace(key, TSliceCollapser<TConfig>::Make(config, SliceMaintainer, key));
                }
            }
            for (auto it(collapsers.begin()); it != collapsers.end();) {
                if (knownKeys.find(it->first) == knownKeys.cend()) {
                    collapsers.erase(it++);
                } else {
                    ++it;
                }
            }
        }

        template <class T>
        TThreadPool::TFuture WaitForTask(const T& collapsers) {
            auto guard = TGuard<TAdaptiveLock>(CollapserLock);
            TVector<TThreadPool::TFuture> futures;
            futures.emplace_back(SpinAndWait());
            for (const auto& pair : collapsers) {
                futures.emplace_back(pair.second->SpinAndWait());
            }
            return NThreading::WaitExceptionOrAll(futures);
        }

        const IProbeSliceMaintainer& SliceMaintainer;

        const TDatacenterConfig DatacenterConfig;
        const TLineConfig LineConfig;
        const TSwitchConfig SwitchConfig;

        TAdaptiveLock CollapserLock;

        TDatacenterCollapser::TMapType DatacenterCollapsers;
        TLineCollapser::TMapType LineCollapsers;
        TSwitchCollapser::TMapType SwitchCollapsers;
    };

    TProbeSliceMerger::TProbeSliceMerger(const IProbeSliceMaintainer& sliceMaintainer)
        : Impl(MakeHolder<TImpl>(sliceMaintainer))
        , SchedulerGuard(Impl->Schedule())
    {
    }

    TProbeSliceMerger::~TProbeSliceMerger() {
    }

    TSwitchPairIndex::TRef TProbeSliceMerger::GetSwitchLevelIndex(const TProbeAggregatorKey& key) const {
        return Impl->GetSwitchLevelIndex(key);
    }

    TLinePairIndex::TRef TProbeSliceMerger::GetLineLevelIndex(const TProbeAggregatorKey& key) const {
        return Impl->GetLineLevelIndex(key);
    }

    TDatacenterPairIndex::TRef TProbeSliceMerger::GetDatacenterLevelIndex(const TProbeAggregatorKey& key) const {
        return Impl->GetDatacenterLevelIndex(key);
    }

    TProbeAggregatorKeySet TProbeSliceMerger::GetSwitchLevelKeys() const {
        return Impl->GetSwitchLevelKeys();
    }

    TProbeAggregatorKeySet TProbeSliceMerger::GetLineLevelKeys() const {
        return Impl->GetLineLevelKeys();
    }

    TProbeAggregatorKeySet TProbeSliceMerger::GetDatacenterLevelKeys() const {
        return Impl->GetDatacenterLevelKeys();
    }

    const TEventHub<TSwitchPairIndex::TRef>& TProbeSliceMerger::OnSwitchUpdated() const {
        return Impl->OnSwitchUpdated();
    }

    const TEventHub<TLinePairIndex::TRef>& TProbeSliceMerger::OnLineUpdated() const {
        return Impl->OnLineUpdated();
    }

    const TEventHub<TDatacenterPairIndex::TRef>& TProbeSliceMerger::OnDatacenterUpdated() const {
        return Impl->OnDatacenterUpdated();
    }

    TThreadPool::TFuture TProbeSliceMerger::WaitSwitchTask() {
        return Impl->WaitSwitchTask();
    }

    TThreadPool::TFuture TProbeSliceMerger::WaitQueueTask() {
        return Impl->WaitQueueTask();
    }

    TThreadPool::TFuture TProbeSliceMerger::WaitDatacenterTask() {
        return Impl->WaitDatacenterTask();
    }
}
