#pragma once

#include <infra/netmon/probe_slice_maintainer.h>
#include <infra/netmon/library/scheduler.h>
#include <infra/netmon/library/event_hub.h>

namespace NNetmon {
    class IProbeSliceMerger {
    public:
        virtual ~IProbeSliceMerger() = default;

        virtual TSwitchPairIndex::TRef GetSwitchLevelIndex(const TProbeAggregatorKey& key) const = 0;
        virtual TLinePairIndex::TRef GetLineLevelIndex(const TProbeAggregatorKey& key) const = 0;
        virtual TDatacenterPairIndex::TRef GetDatacenterLevelIndex(const TProbeAggregatorKey& key) const = 0;

        virtual TProbeAggregatorKeySet GetSwitchLevelKeys() const = 0;
        virtual TProbeAggregatorKeySet GetLineLevelKeys() const = 0;
        virtual TProbeAggregatorKeySet GetDatacenterLevelKeys() const = 0;

        virtual const TEventHub<TSwitchPairIndex::TRef>& OnSwitchUpdated() const = 0;
        virtual const TEventHub<TLinePairIndex::TRef>& OnLineUpdated() const = 0;
        virtual const TEventHub<TDatacenterPairIndex::TRef>& OnDatacenterUpdated() const = 0;
    };

    template<typename TIndex>
    typename TIndex::TRef GetMergerIndex(const IProbeSliceMerger& merger, const TProbeAggregatorKey& key);

    template<>
    inline TSwitchPairIndex::TRef GetMergerIndex<TSwitchPairIndex>(const IProbeSliceMerger& merger, const TProbeAggregatorKey& key) {
        return merger.GetSwitchLevelIndex(key);
    }

    template<>
    inline TLinePairIndex::TRef GetMergerIndex<TLinePairIndex>(const IProbeSliceMerger& merger, const TProbeAggregatorKey& key) {
        return merger.GetLineLevelIndex(key);
    }

    template<>
    inline TDatacenterPairIndex::TRef GetMergerIndex<TDatacenterPairIndex>(const IProbeSliceMerger& merger, const TProbeAggregatorKey& key) {
        return merger.GetDatacenterLevelIndex(key);
    }

    class TProbeSliceMerger: public IProbeSliceMerger, public TNonCopyable {
    public:
        TProbeSliceMerger(const IProbeSliceMaintainer& sliceMaintainer);
        ~TProbeSliceMerger();

        TSwitchPairIndex::TRef GetSwitchLevelIndex(const TProbeAggregatorKey& key) const override;
        TLinePairIndex::TRef GetLineLevelIndex(const TProbeAggregatorKey& key) const override;
        TDatacenterPairIndex::TRef GetDatacenterLevelIndex(const TProbeAggregatorKey& key) const override;

        TProbeAggregatorKeySet GetSwitchLevelKeys() const override;
        TProbeAggregatorKeySet GetLineLevelKeys() const override;
        TProbeAggregatorKeySet GetDatacenterLevelKeys() const override;

        const TEventHub<TSwitchPairIndex::TRef>& OnSwitchUpdated() const override;
        const TEventHub<TLinePairIndex::TRef>& OnLineUpdated() const override;
        const TEventHub<TDatacenterPairIndex::TRef>& OnDatacenterUpdated() const override;

        TThreadPool::TFuture WaitSwitchTask();
        TThreadPool::TFuture WaitQueueTask();
        TThreadPool::TFuture WaitDatacenterTask();

    private:
        class TImpl;
        THolder<TImpl> Impl;

        TScheduledTask::TTaskGuard SchedulerGuard;
    };
}
