#include <infra/netmon/probe_storage.h>
#include <infra/netmon/settings.h>

#include <util/generic/xrange.h>

namespace NNetmon {
    TAtomic SliceStatsDcIndexTotal;
    TAtomic SliceStatsLineIndexTotal;
    TAtomic SliceStatsSwitchIndexTotal;

    namespace {
        const std::size_t PROBE_RELEASE_INTERVAL = 10000;
    }

    TProbeStorage::~TProbeStorage() {
        auto guard = TLightWriteGuard(ProbeLock);
        while (!GeneratedTree.Empty()) {
            TProbe::TRef probe(GeneratedTree.Begin()->Get());
            probe->GetGeneratedItem().UnLink();
            probe->GetSwitchItem().UnLink();
            probe->DecRef();
        }
    }

    TProbeStorage::TProbeStats TProbeStorage::InsertProbes(TVector<TProbe::TRef>&& currentProbes, const TInstant& now) {
        TProbeStats stats;
        TSimpleTimer timer;
        auto insertGuard = TGuard<TAdaptiveLock>(InsertLock);

        {
            auto guard = TLightWriteGuard(ProbeLock);

            for (auto& probe : currentProbes) {
                ++stats.Processed;

                if (probe->GetGeneratedItem().Exists(GeneratedTree)) {
                    ++stats.Duplicates;
                } else {
                    ++stats.Created;

                    GeneratedTree.Insert(&probe->GetGeneratedItem());

                    // maintain switch pairs on each level
                    DatacenterLevelIndex.LinkProbe(*probe);

                    probe->Ref();
                }

                if (stats.Processed % PROBE_RELEASE_INTERVAL == 0) {
                    // allow reads for some time
                    auto unguard = Unguard(guard);
                    SpinLockPause();
                }
            }
        }

        currentProbes.clear();

        stats.InsertTime = timer.Get();
        timer.Reset();

        {
            auto guard = TLightWriteGuard(ProbeLock);
            stats.Unlinked += DatacenterLevelIndex.UnlinkExpiredProbes(now);
            stats.Unlinked += QueueLevelIndex.UnlinkExpiredProbes(now);
            stats.Unlinked += SwitchLevelIndex.UnlinkExpiredProbes(now);

            stats.Deleted += RemoveUnlinkedProbes();

            AtomicSet(SliceStatsDcIndexTotal, DatacenterLevelIndex.GetProbeCount());
            AtomicSet(SliceStatsLineIndexTotal, QueueLevelIndex.GetProbeCount());
            AtomicSet(SliceStatsSwitchIndexTotal, SwitchLevelIndex.GetProbeCount());
        }

        stats.DeleteTime = timer.Get();

        return stats;
    }

    std::size_t TProbeStorage::RemoveUnlinkedProbes() {
        // remove old probes from memory
        std::size_t deleted = 0;
        while (!GeneratedTree.Empty()) {
            TProbe::TRef probe(GeneratedTree.Begin()->Get());
            if (probe->GetSwitchItem().Linked()) {
                break;
            }

            probe->GetGeneratedItem().UnLink();
            probe->DecRef();

            ++deleted;
        }
        return deleted;
    }

    TProbeStorage::TProbeStorage(const TProbeSliceKey& sliceKey, const TTopologyStorage& topologyStorage)
        : SliceKey(sliceKey)
        , TopologyStorage(topologyStorage)
        , SwitchLevelIndex(GeneratedTree, TSettings::Get()->GetSwitchAggregationWindow())
        , QueueLevelIndex(GeneratedTree, TSettings::Get()->GetLineAggregationWindow(), &SeenHosts)
        , DatacenterLevelIndex(GeneratedTree, TSettings::Get()->GetDcAggregationWindow(), &SeenHosts)
    {
        DatacenterLevelIndex.SetUnlinkTree(&QueueLevelIndex);
        QueueLevelIndex.SetUnlinkTree(&SwitchLevelIndex);
        SwitchLevelIndex.SetLookupTree(&QueueLevelIndex);
        QueueLevelIndex.SetLookupTree(&DatacenterLevelIndex);
    }
}
