#pragma once

#include <infra/netmon/probe_index.h>
#include <infra/netmon/state_index.h>

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/stream/buffer.h>
#include <util/system/yield.h>
#include <util/generic/algorithm.h>

namespace NNetmon {
    class TProbeStorage: public TNonCopyable, public TAtomicRefCount<TProbeStorage> {
    public:
        using TRef = TIntrusivePtr<TProbeStorage>;

        template <typename... Args>
        static inline TRef Make(Args&&... args) {
            return new TProbeStorage(std::forward<Args>(args)...);
        }

        ~TProbeStorage();

        struct TProbeStats {
            ui64 Processed = 0;
            ui64 Created = 0;
            ui64 Duplicates = 0;
            ui64 Unlinked = 0;
            ui64 Deleted = 0;
            TDuration InsertTime;
            TDuration DeleteTime;
        };
        TProbeStats InsertProbes(TVector<TProbe::TRef>&& currentProbes, const TInstant& now);

        inline std::size_t GetSwitchProbeCount() const {
            return SwitchLevelIndex.GetProbeCount();
        }
        inline std::size_t GetLineProbeCount() const {
            return QueueLevelIndex.GetProbeCount();
        }
        inline std::size_t GetDatacenterProbeCount() const {
            return DatacenterLevelIndex.GetProbeCount();
        }

        struct TSwitchIndexAccessor {
            static inline const TProbeIndex& Extract(const TProbeStorage& group) noexcept {
                return group.SwitchLevelIndex;
            }
        };

        struct TLineIndexAccessor {
            static inline const TProbeIndex& Extract(const TProbeStorage& group) noexcept {
                return group.QueueLevelIndex;
            }
        };

        struct TDatacenterIndexAccessor {
            static inline const TProbeIndex& Extract(const TProbeStorage& group) noexcept {
                return group.DatacenterLevelIndex;
            }
        };

        template <class TAccessor>
        TSwitchIndexMap::TRef CreateSwitchIndexMap(
                const TTopologyStorage::THostSetBox::TConstValueRef seenHosts,
                const TTopologyStorage::THostSetBox::TConstValueRef terminatedHosts,
                const TTopologyStorage::THostSetBox::TConstValueRef deadHosts) const {

            auto selector(TopologyStorage.GetTopologySelector());
            TSwitchIndexMap::TRef switchIndexMap(MakeAtomicShared<TSwitchIndexMap>(SliceKey));
            TExpressionIdList expressionIds;

            const auto now(TInstant::Now());
            auto guard = TLightReadGuard(ProbeLock);
            TAccessor::Extract(*this).ForEachProbe([&](TProbe* probe) {
                if (!probe) {
                    auto unguard(Unguard(guard));
                    SpinLockPause();
                    return;
                }

                // find tags that exists on source and target hosts
                expressionIds.clear();
                probe->IntersectExpressionIds(*selector, expressionIds);

                // let's insert probe to appropriate indexes
                for (const auto expressionId : expressionIds) {
                    TSwitchPairIndex* switchIndex = nullptr;
                    const auto it = switchIndexMap->find(expressionId);
                    if (it.IsEnd()) {
                        // let's create new one
                        const auto key(TProbeAggregatorKey::FromSliceKey(SliceKey, expressionId));
                        const auto newSwitchIndex = TSwitchPairIndex::Make(key, now);
                        switchIndexMap->emplace(expressionId, newSwitchIndex);
                        switchIndex = newSwitchIndex.Get();
                    } else {
                        switchIndex = it->second.Get();
                    }
                    Y_ASSERT(switchIndex != nullptr);

                    if (probe->IsDead(*seenHosts, *terminatedHosts, *deadHosts)) {
                        switchIndex->AppendDeadProbe(*probe);
                    } else {
                        switchIndex->AppendAliveProbe(*probe, now);
                    }
                }
            });

            // compute availability when all states are ready
            for (auto& pair : *switchIndexMap) {
                pair.second->UpdateAvailability(*selector);
            }

            return switchIndexMap;
        }

        template <class TAccessor, class TKey, class F>
        TProbe::TRefVector FindProbes(const TKey& key, F&& filter) {
            auto guard = TLightReadGuard(ProbeLock);

            TProbe::TRefVector probeVector;
            TAccessor::Extract(*this).FindProbes(key, [&](TProbe* probe) noexcept {
                if (filter(probe)) {
                    probeVector.emplace_back(probe);
                }
            });

            return probeVector;
        }

        template <class TAccessor, class F>
        TProbe::TRefVector FindProbes(F&& filter) {
            auto guard = TLightReadGuard(ProbeLock);

            TProbe::TRefVector probeVector;
            TAccessor::Extract(*this).ForEachProbe([&](TProbe* probe) noexcept {
                if (!probe) {
                    auto unguard(Unguard(guard));
                    SpinLockPause();
                } else if (filter(probe)) {
                    probeVector.emplace_back(probe);
                }
            });

            return probeVector;
        }

        inline const TCountedHostStorage& GetSeenHosts() const {
            return SeenHosts;
        }

    private:
        TProbeStorage(const TProbeSliceKey& sliceKey, const TTopologyStorage& topologyStorage);

        std::size_t RemoveUnlinkedProbes();

        TProbeSliceKey SliceKey;
        const TTopologyStorage& TopologyStorage;

        TProbe::TGeneratedTree GeneratedTree;

        TCountedHostStorage SeenHosts;

        TProbeIndex SwitchLevelIndex;
        TProbeIndex QueueLevelIndex;
        TProbeIndex DatacenterLevelIndex;

        TAdaptiveLock InsertLock;
        TLightRWLock ProbeLock;
    };
}
