#include <infra/netmon/probe_storage.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/common_ut.h>
#include <infra/netmon/topology/common_ut.h>
#include <infra/netmon/topology/settings.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/xrange.h>

#include <cfloat>

using namespace NNetmon;

auto TRUE_PREDICATE = [](const TProbe*) {
    return true;
};

class TProbeStorageTest: public TTestBase {
    UNIT_TEST_SUITE(TProbeStorageTest);
    UNIT_TEST(TestOldProbeInsertion)
    UNIT_TEST(TestProbeDeduplication)
    UNIT_TEST(TestProbeLifetime)
    UNIT_TEST(TestSeenHosts)
    UNIT_TEST(TestFindProbes)
    UNIT_TEST_SUITE_END();

public:
    TProbeStorageTest()
        : TopologyStorage(TGlobalTopology::GetTopologyStorage())
        , TopologySelector(TopologyStorage.GetTopologySelector())
        , SliceKey{BACKBONE6, ICMP, 0}
    {
    }

private:
    TTopology::THostInterfaceRef SafeFindIface(const TString& name, TTopologyStorage& topologyStorage) {
        auto iface(topologyStorage.FindHostInterface(name));
        UNIT_ASSERT(iface);
        return iface;
    }

    inline void TestOldProbeInsertion() {
        TProbeStorage::TRef probeStorage(TProbeStorage::Make(SliceKey, TopologyStorage));

        const auto now(TInstant::Now());
        const auto probe(CreateProbe(
            "man1-7127.search.yandex.net",
            "sas1-1234.search.yandex.net",
            now - TSettings::Get()->GetSwitchAggregationWindow()
        ));

        {
            // insert new probe
            const auto stats(probeStorage->InsertProbes(TVector<TProbe::TRef>{probe}, now));
            UNIT_ASSERT(stats.Created);
            UNIT_ASSERT(!stats.Deleted);
            UNIT_ASSERT_EQUAL(probeStorage->GetSwitchProbeCount(), 1);
            UNIT_ASSERT_EQUAL(probeStorage->GetLineProbeCount(), 0);
            UNIT_ASSERT_EQUAL(probeStorage->GetDatacenterProbeCount(), 0);
        }

        {
            // check that nothing changed
            const auto stats(probeStorage->InsertProbes(TVector<TProbe::TRef>{}, now));
            UNIT_ASSERT(!stats.Created);
            UNIT_ASSERT(!stats.Deleted);
        }

        {
            // check that probe will be deleted
            const auto stats(probeStorage->InsertProbes(TVector<TProbe::TRef>{}, now + TDuration::Seconds(1)));
            UNIT_ASSERT(!stats.Created);
            UNIT_ASSERT(stats.Deleted);
            UNIT_ASSERT_EQUAL(probeStorage->GetSwitchProbeCount(), 0);
        }
    }

    inline void TestProbeDeduplication() {
        TProbeStorage::TRef probeStorage(TProbeStorage::Make(SliceKey, TopologyStorage));

        const auto now(TInstant::Now());
        const auto oneProbe(CreateProbe(
            "man1-7127.search.yandex.net",
            "sas1-1234.search.yandex.net",
            now
        ));
        const auto anotherProbe(CreateProbe(
            "man1-7127.search.yandex.net",
            "sas1-1234.search.yandex.net",
            now - TDuration::Seconds(1)
        ));

        {
            // insert new probe
            const auto stats(probeStorage->InsertProbes(TVector<TProbe::TRef>{oneProbe}, now));
            UNIT_ASSERT(stats.Created);
            UNIT_ASSERT(!stats.Duplicates);
        }

        {
            // insert same probe
            const auto stats(probeStorage->InsertProbes(TVector<TProbe::TRef>{oneProbe}, now));
            UNIT_ASSERT(!stats.Created);
            UNIT_ASSERT(stats.Duplicates);
        }

        {
            // insert another probe
            const auto stats(probeStorage->InsertProbes(TVector<TProbe::TRef>{anotherProbe}, now));
            UNIT_ASSERT(stats.Created);
            UNIT_ASSERT(!stats.Duplicates);
        }

        UNIT_ASSERT_EQUAL(probeStorage->GetSwitchProbeCount(), 2);
        UNIT_ASSERT_EQUAL(probeStorage->GetLineProbeCount(), 2);
        UNIT_ASSERT_EQUAL(probeStorage->GetDatacenterProbeCount(), 2);
    }

    inline void TestSeenHosts() {
        const auto now(TInstant::Now());
        TProbeStorage::TRef probeStorage(TProbeStorage::Make(SliceKey, TopologyStorage));
        probeStorage->InsertProbes(TVector<TProbe::TRef>{
            CreateProbe("man1-7127.search.yandex.net", "sas1-1234.search.yandex.net", now)
        }, now);

        const auto seenHosts(probeStorage->GetSeenHosts().GetHosts());

        UNIT_ASSERT_EQUAL(seenHosts, TTopologyStorage::THostIdSet{
            FindHost("man1-7127.search.yandex.net")->GetReducedId()
        });
    }

    inline void TestFindProbes() {
        TestFindProbes(true);
        TestFindProbes(false);
    }

    inline void TestFindProbes(bool podAsQueue) {
        TTopologySettings::Get()->SetUsePodAsQueue(podAsQueue);

        TTopologyUpdater topologyUpdater(false);
        TExpressionStorage expressionStorage(false);
        TGroupStorage groupStorage(false);
        auto topologyStorage_ = podAsQueue ? MakeAtomicShared<TTopologyStorage>(topologyUpdater, expressionStorage, groupStorage, false) : nullptr;

        // use local topology if podAsQueue == true, else use global topology
        TTopologyStorage& topologyStorage = podAsQueue ? *topologyStorage_ : TopologyStorage;

        const auto now(TInstant::Now());

        const auto first(CreateProbe(
            "man1-7127.search.yandex.net",
            "sas1-1234.search.yandex.net",
            now - TSettings::Get()->GetSwitchAggregationWindow(),
            &topologyStorage
        ));
        const auto second(CreateProbe(
            "man1-7127.search.yandex.net",
            "sas1-1235.search.yandex.net",
            now - TSettings::Get()->GetLineAggregationWindow(),
            &topologyStorage
        ));
        const TProbe::TRefVector reference{first, second};

        TProbeStorage::TRef probeStorage(TProbeStorage::Make(SliceKey, topologyStorage));
        probeStorage->InsertProbes(TVector<TProbe::TRef>{first, second}, now);
        const auto found(probeStorage->FindProbes<TProbeStorage::TSwitchIndexAccessor>(
            TLinePairKey(
                topologyStorage.FindQueue((podAsQueue ? "sas1" : "sas"), (podAsQueue ? "sas-1" : "sas1.4")),
                topologyStorage.FindQueue("man", (podAsQueue ? "man-3" : "man3"))
            ),
            TRUE_PREDICATE
        ));

        UNIT_ASSERT_EQUAL(probeStorage->GetSwitchProbeCount(), 2);
        UNIT_ASSERT_EQUAL(probeStorage->GetLineProbeCount(), 1);
        UNIT_ASSERT_EQUAL(probeStorage->GetDatacenterProbeCount(), 1);
        UNIT_ASSERT_EQUAL(found, reference);
        TTopologySettings::Get()->SetUsePodAsQueue(false);
    }

    inline void TestProbeLifetime() {
        TestProbeLifetime(true);
        TestProbeLifetime(false);
    }

    inline void TestProbeLifetime(bool podAsQueue) {
        TTopologySettings::Get()->SetUsePodAsQueue(podAsQueue);

        TTopologyUpdater topologyUpdater(false);
        TExpressionStorage expressionStorage(false);
        TGroupStorage groupStorage(false);
        auto topologyStorage_ = podAsQueue ? MakeAtomicShared<TTopologyStorage>(topologyUpdater, expressionStorage, groupStorage, false) : nullptr;

        // use local topology if podAsQueue == true, else use global topology
        TTopologyStorage& topologyStorage = podAsQueue ? *topologyStorage_ : TopologyStorage;
        auto topologySelector = topologyStorage.GetTopologySelector();

        const auto now(TInstant::Now());
        TProbeStorage::TRef probeStorage(TProbeStorage::Make(SliceKey, topologyStorage));

        TTopology::TSwitchRef sourceSwitch(topologyStorage.FindSwitch("man", (podAsQueue ? "man-3" : "man3"), "man1-3s165"));
        TTopology::TSwitchRef targetSwitch(topologyStorage.FindSwitch((podAsQueue ? "sas1" : "sas"), (podAsQueue ? "sas-1" : "sas1.4"), "sas1-s930"));
        UNIT_ASSERT(sourceSwitch);
        UNIT_ASSERT(targetSwitch);

        // insert probes
        probeStorage->InsertProbes(TVector<TProbe::TRef>{
            // man1-s136 -> sas1-s796
            CreateProbe("man1-7127.search.yandex.net", "sas3-5500.search.yandex.net", now, &topologyStorage),
            // man1-s136 -> sas1-s930
            CreateProbe("man1-7127.search.yandex.net", "sas1-1234.search.yandex.net", now, &topologyStorage),
            // man1-s136 -> sas1-s726
            CreateProbe("man1-7127.search.yandex.net", "sas2-5330.search.yandex.net", now, &topologyStorage)
        }, now);
        UNIT_ASSERT_EQUAL(probeStorage->GetSwitchProbeCount(), 3);
        UNIT_ASSERT_EQUAL(probeStorage->GetLineProbeCount(), 3);
        UNIT_ASSERT_EQUAL(probeStorage->GetDatacenterProbeCount(), 3);

        // let's find probes
        auto foundProbes = probeStorage->FindProbes<TProbeStorage::TSwitchIndexAccessor>(
            TSwitchPairKey(targetSwitch, sourceSwitch), TRUE_PREDICATE
        );
        UNIT_ASSERT_EQUAL(foundProbes.size(), 1);
        UNIT_ASSERT_EQUAL(*foundProbes.front()->GetSourceIface(), *SafeFindIface("man1-7127.search.yandex.net", topologyStorage));
        UNIT_ASSERT_EQUAL(*foundProbes.front()->GetTargetIface(), *SafeFindIface("sas1-1234.search.yandex.net", topologyStorage));

        auto missingProbes = probeStorage->FindProbes<TProbeStorage::TSwitchIndexAccessor>(
            TSwitchPairKey(sourceSwitch, targetSwitch), TRUE_PREDICATE
        );
        UNIT_ASSERT_EQUAL(missingProbes.size(), 0);

        // try to compute switch state from our probes
        TExpressionId defaultExpression = topologyStorage.DefaultExpressionId();
        const TTopologyStorage::THostSetBox seenHosts;
        const TTopologyStorage::THostSetBox terminatedHosts;
        const TTopologyStorage::THostSetBox deadHosts;
        auto switchIndexMap(
            probeStorage->CreateSwitchIndexMap<TProbeStorage::TSwitchIndexAccessor>(
                seenHosts.Get(), terminatedHosts.Get(), deadHosts.Get()));
        UNIT_ASSERT(!switchIndexMap->find(defaultExpression).IsEnd());

        TSwitchPairIndex::TRef switchIndex(switchIndexMap->find(defaultExpression)->second);
        UNIT_ASSERT_EQUAL(switchIndex->GetPairCount(), 3);
        for (auto it(switchIndex->GetTree().Begin()); it != switchIndex->GetTree().End(); ++it) {
            UNIT_ASSERT_EQUAL(it->GetProbeCount(), 1);
            auto connectivity = it->GetConnectivityHistogram().GetValues();
            UNIT_ASSERT(connectivity.Defined());
            for (const auto idx : xrange(connectivity->size())) {
                UNIT_ASSERT_DOUBLES_EQUAL(connectivity->at(idx), 0.0, FLT_EPSILON);
            }
        }

        // try to compute queue state from switch state
        TLinePairIndex::TRef queueIndex(TLinePairIndex::Make(
                *topologySelector, *switchIndex, TProbeAggregatorKey::FromSliceKey(SliceKey, defaultExpression)));
        UNIT_ASSERT_EQUAL(queueIndex->GetPairCount(), 1);
        for (auto it(queueIndex->GetTree().Begin()); it != queueIndex->GetTree().End(); ++it) {
            UNIT_ASSERT_EQUAL(it->GetProbeCount(), 3);
            auto connectivity = it->GetConnectivityHistogram().GetValues();
            UNIT_ASSERT(connectivity.Defined());
            for (const auto idx : xrange(connectivity->size())) {
                UNIT_ASSERT_DOUBLES_EQUAL(connectivity->at(idx), 0.0, FLT_EPSILON);
            }
        }

        // try to compute datacenter state from switch state
        TDatacenterPairIndex::TRef datacenterIndex(TDatacenterPairIndex::Make(
                *topologySelector, *switchIndex, TProbeAggregatorKey::FromSliceKey(SliceKey, defaultExpression)));
        UNIT_ASSERT_EQUAL(datacenterIndex->GetPairCount(), 1);
        for (auto it(datacenterIndex->GetTree().Begin()); it != datacenterIndex->GetTree().End(); ++it) {
            UNIT_ASSERT_EQUAL(it->GetProbeCount(), 3);
            auto connectivity = it->GetConnectivityHistogram().GetValues();
            UNIT_ASSERT(connectivity.Defined());
            for (const auto idx : xrange(connectivity->size())) {
                UNIT_ASSERT_DOUBLES_EQUAL(connectivity->at(idx), 0.0, FLT_EPSILON);
            }
        }

        // drop indexes
        datacenterIndex.Drop();
        queueIndex.Drop();
        switchIndex.Drop();

        // let's cleanup old probes
        probeStorage->InsertProbes(TVector<TProbe::TRef>{}, TSettings::Get()->GetDcAggregationWindow().ToDeadLine());
        UNIT_ASSERT_EQUAL(probeStorage->GetSwitchProbeCount(), 3);
        UNIT_ASSERT_EQUAL(probeStorage->GetLineProbeCount(), 0);
        UNIT_ASSERT_EQUAL(probeStorage->GetDatacenterProbeCount(), 0);

        probeStorage->InsertProbes(TVector<TProbe::TRef>{}, TSettings::Get()->GetLineAggregationWindow().ToDeadLine());
        UNIT_ASSERT_EQUAL(probeStorage->GetSwitchProbeCount(), 3);
        UNIT_ASSERT_EQUAL(probeStorage->GetLineProbeCount(), 0);

        probeStorage->InsertProbes(TVector<TProbe::TRef>{}, TSettings::Get()->GetSwitchAggregationWindow().ToDeadLine());
        UNIT_ASSERT_EQUAL(probeStorage->GetSwitchProbeCount(), 0);

        // check that found probes are still alive
        UNIT_ASSERT_EQUAL(foundProbes.front()->GetSourceIface(), topologyStorage.FindHostInterface("man1-7127.search.yandex.net"));
        UNIT_ASSERT_EQUAL(foundProbes.front()->RefCount(), 1);
        TTopologySettings::Get()->SetUsePodAsQueue(false);
    }

    TTopologyStorage& TopologyStorage;
    TTopologySelector::TBox::TConstValueRef TopologySelector;
    TProbeSliceKey SliceKey;
};

UNIT_TEST_SUITE_REGISTRATION(TProbeStorageTest);
