#include <infra/netmon/probe.h>
#include <infra/netmon/common_ut.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

#include <cfloat>

using namespace NNetmon;

class TProbeTest: public TTestBase {
    UNIT_TEST_SUITE(TProbeTest);
    UNIT_TEST(TestExpressions)
    UNIT_TEST(TestSerialization)
    UNIT_TEST_SUITE_END();

public:
    TProbeTest()
        : TopologyStorage(TGlobalTopology::GetTopologyStorage())
    {
    }

private:
    inline void TestExpressions() {
        const auto bareProbe(CreateProbe("jmon-test.search.yandex.net", "man1-0015.search.yandex.net"));
        const auto virtualProbe(CreateProbe("jmon-test.search.yandex.net", "noc-duty.yndx.net"));

        const auto selector(GetTopologySelector());
        UNIT_ASSERT(bareProbe->HasExpressionId(*selector, selector->DefaultExpressionId()));
        UNIT_ASSERT(!virtualProbe->HasExpressionId(*selector, selector->DefaultExpressionId()));
    }

    inline void TestSerialization() {
        auto source = TopologyStorage.FindHostInterface("jmon-test.search.yandex.net");
        auto target = TopologyStorage.FindHostInterface("sas1-0002.media.yandex.net");
        UNIT_ASSERT(source && target);
        TProbe::TRef probe(TProbe::Make(
            source, target,
            EmptyAddress, EmptyAddress, 0, 0,
            TInstant::Now(), 0.5, 1.0
        ));

        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(probe->ToProto(builder));
        const NProbe::TProbe& serializedProbe(
            *flatbuffers::GetRoot<NProbe::TProbe>(builder.GetBufferPointer()));

        UNIT_ASSERT_EQUAL(
            TopologyStorage.FindHostInterface(*serializedProbe.SourceIface()),
            probe->GetSourceIface());
        UNIT_ASSERT_EQUAL(
            TopologyStorage.FindHostInterface(*serializedProbe.TargetIface()),
            probe->GetTargetIface());

        UNIT_ASSERT_EQUAL(serializedProbe.Generated(), probe->GetGenerated().Seconds());

        UNIT_ASSERT_DOUBLES_EQUAL(
            serializedProbe.Score(), probe->GetScore(), FLT_EPSILON);
        UNIT_ASSERT_DOUBLES_EQUAL(
            serializedProbe.RoundTripTime(), probe->GetRoundTripTime(), FLT_EPSILON);
    }

    TTopologyStorage& TopologyStorage;
    const TIpAddress EmptyAddress;
};

UNIT_TEST_SUITE_REGISTRATION(TProbeTest);
