#include <infra/netmon/racktables.h>
#include <infra/netmon/library/boxes.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/topology/settings.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/json/json_reader.h>

namespace NNetmon {
    namespace {
        const TString REQUEST_DATA = "{\"query\": \"query {object(code:\\\"{Netmon Disabled} and ([Датацентры] and \
                            {серверный свитч} and not ({магистральный свитч} or {свитч агрегации} or {пиринговый свитч}))\\\"){name}}\"}";
    }

    class TRtUpdater::TImpl: public TScheduledTask {
    public:
        TImpl(const TTopologyStorage& topologyStorage)
            : TScheduledTask(TSettings::Get()->GetRtUpdateInterval())
            , TopologyStorage(topologyStorage)
        {
        }

        TThreadPool::TFuture Run() override {
            TString url = TSettings::Get()->GetRtUrl();
            return THttpRequester::Get()->MakeRequest(
                url,
                {},
                NHttp::TFetchOptions()
                    .SetOAuthToken(TSettings::Get()->GetRtAuthToken())
                    .SetContentType("application/json")
                    .SetPostData(REQUEST_DATA)
            ).Subscribe([this, url](const THttpRequester::TFuture& future) {
                try {
                    ProcessRtResponse(future.GetValue());
                } catch (...) {
                    ERROR_LOG << "Request to " << url << " failed: " << CurrentExceptionMessage() << Endl;
                }
            }).IgnoreResult();
        }

        inline TTopologyStorage::THostSetBox::TConstValueRef GetHosts() const {
            return DisabledHosts.Get();
        }

    private:
        void ProcessRtResponse(const NHttpFetcher::TResultRef& response) {
            auto hosts = MakeAtomicShared<TTopologyStorage::THostSet>();
            std::size_t switchCount = 0;
            try {
                NJson::TJsonValue root;
                NJson::ReadJsonFastTree(response->Data, &root, true);
                auto names = root["data"]["object"].GetArraySafe();
                for (const NJson::TJsonValue& name : names) {
                    TTopology::TSwitchRef switch_ = TopologyStorage.FindSwitch(name["name"].GetStringSafe());
                    if (switch_) {
                        switchCount++;
                        for (auto host: switch_->GetRealHosts()) {
                            hosts->insert(TTopology::THostRef(host));
                        }
                    }
                }
                DisabledHosts.Swap(hosts);
            } catch (...) {
                ERROR_LOG << "Response processsing failed: " << CurrentExceptionMessage() << Endl;
            }

            INFO_LOG << switchCount << " disabled switches with " << DisabledHosts.Get()->size() << " hosts received from racktables" << Endl;
        }

        const TTopologyStorage& TopologyStorage;
        TTopologyStorage::THostSetBox DisabledHosts;
    };

    TRtUpdater::TRtUpdater(const TTopologyStorage& topologyStorage)
        : TRtUpdater(topologyStorage, !TSettings::Get()->GetRtAuthToken().empty())
    {
    }

    TRtUpdater::TRtUpdater(const TTopologyStorage& topologyStorage, bool schedule)
        : Impl(MakeHolder<TImpl>(topologyStorage))
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TRtUpdater::~TRtUpdater() = default;

    TTopologyStorage::THostSetBox::TConstValueRef TRtUpdater::GetHosts() const {
        return Impl->GetHosts();
    }
}
