#pragma once

#include <infra/netmon/library/api_client_helpers.h>
#include <infra/netmon/library/thread_pool.h>
#include <infra/netmon/settings.h>

namespace NNetmon {
    template <class TResponse, class TExecutor = TThreadPool>
    class TGatherer : public TNonCopyable {
    public:
        using TResponseList = TVector<TFlatObject<TResponse>>;
        using TFuture = NThreading::TFuture<TResponseList>;
        using TBox = TAtomicLockedBox<TResponseList>;

        TGatherer(const TString& path, const flatbuffers::FlatBufferBuilder& builder)
            : Path(path)
            , Builder(builder)
            , Result_(TBox::Make())
        {
        }

        const typename TBox::TRef Box() const {
            return Result_;
        }

        TThreadPool::TFuture Gather() {
            auto result = Result_;
            return ScatterRequests([result] (const TNehRequester::TFuture& future_) {
                TFlatObject<TResponse> response;
                ReadFlatBuffer(future_.GetValue()->Data, response);
                result->Own()->emplace_back(std::move(response));
            });
        }

        static TFuture Collect(const TString& path, const flatbuffers::FlatBufferBuilder& builder) {
            TGatherer<TResponse, TExecutor> gatherer(path, builder);
            return gatherer.Gather().Apply([box = gatherer.Box()] (const TThreadPool::TFuture& future_){
                future_.GetValue();
                TResponseList result;
                result.swap(*box->Own());
                return std::move(result);
            });
        }

    private:
        template <class TFunc>
        inline NThreading::TFuture<void> ScatterRequests(TFunc&& func) const {
            auto& sliceShards = TSettings::Get()->GetSlicerShards();

            TVector<NThreading::TFuture<void>> futuresToWait;
            futuresToWait.reserve(sliceShards.size());

            for (const auto& shard : sliceShards) {
                try {
                    futuresToWait.emplace_back(MakeRequest(
                        shard, Path, Builder, TExecutor::Get()
                    ).Apply(func));
                } catch (...) {
                    ERROR_LOG << "Can't make request to 'tcp2://" << shard.Host << ":" << shard.Port
                              << Path << "': " << CurrentExceptionMessage() << Endl;
                }
            }

            return WaitExceptionOrAll(futuresToWait);
        }

        const TString Path;
        const flatbuffers::FlatBufferBuilder& Builder;

        typename TBox::TRef Result_;
    };
}
