#include <infra/netmon/schema_maintainer.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/library/settings.h>

#include <infra/netmon/library/clickhouse/tables.h>

#include <util/generic/xrange.h>
#include <util/draft/holder_vector.h>

namespace NNetmon {
    namespace {
        const TStringBuf PROBES_LOCAL_TABLE_TEMPLATE = TStringBuf(R"(
CREATE TABLE {tableName}
(
    SourceFqdn UInt64,
    SourceSwitch UInt64,
    SourceQueue UInt64,
    SourceDc UInt64,
    SourceAddress String,
    SourcePort UInt16,
    TargetFqdn UInt64,
    TargetSwitch UInt64,
    TargetQueue UInt64,
    TargetDc UInt64,
    TargetAddress String,
    TargetPort UInt16,
    Network UInt8,
    Protocol UInt8,
    Score Float64,
    Rtt Float64,
    Generated DateTime DEFAULT now(),
    Inserted DateTime DEFAULT now(),
    InsertedDate Date DEFAULT today()
) ENGINE = MergeTree(
    InsertedDate,
    (
        InsertedDate,
        Inserted,
        Network,
        Protocol,
        SourceDc,
        TargetDc,
        SourceQueue,
        TargetQueue,
        SourceSwitch,
        TargetSwitch,
        SourceFqdn,
        TargetFqdn
    ),
    8192
)
)");

        const TStringBuf PROBES_REPLICATED_TABLE_TEMPLATE = TStringBuf(R"(
CREATE TABLE {tableName}
(
    SourceFqdn UInt64,
    SourceSwitch UInt64,
    SourceQueue UInt64,
    SourceDc UInt64,
    SourceAddress String,
    SourcePort UInt16,
    TargetFqdn UInt64,
    TargetSwitch UInt64,
    TargetQueue UInt64,
    TargetDc UInt64,
    TargetAddress String,
    TargetPort UInt16,
    Network UInt8,
    Protocol UInt8,
    Score Float64,
    Rtt Float64,
    Generated DateTime DEFAULT now(),
    Inserted DateTime DEFAULT now(),
    InsertedDate Date DEFAULT today()
) ENGINE = ReplicatedMergeTree(
    '/clickhouse/tables/{shard}/{tableName}',
    '{replica}',
    InsertedDate,
    (
        InsertedDate,
        Inserted,
        Network,
        Protocol,
        SourceDc,
        TargetDc,
        SourceQueue,
        TargetQueue,
        SourceSwitch,
        TargetSwitch,
        SourceFqdn,
        TargetFqdn
    ),
    8192
)
)");

        const TStringBuf DISTRIBUTED_PROBES_TABLE = TStringBuf(R"(
CREATE TABLE distributed_probes (
    SourceFqdn UInt64,
    SourceSwitch UInt64,
    SourceQueue UInt64,
    SourceDc UInt64,
    SourceAddress String,
    SourcePort UInt16,
    TargetFqdn UInt64,
    TargetSwitch UInt64,
    TargetQueue UInt64,
    TargetDc UInt64,
    TargetAddress String,
    TargetPort UInt16,
    Network UInt8,
    Protocol UInt8,
    Score Float64,
    Rtt Float64,
    Generated DateTime DEFAULT now(),
    Inserted DateTime DEFAULT now(),
    InsertedDate Date DEFAULT today()
) ENGINE = Distributed(netmon, 'default', 'probes');
)");

        const TStringBuf LOCAL_PROBES_TABLE = TStringBuf(R"(
CREATE TABLE probes (
    SourceFqdn UInt64,
    SourceSwitch UInt64,
    SourceQueue UInt64,
    SourceDc UInt64,
    SourceAddress String,
    SourcePort UInt16,
    TargetFqdn UInt64,
    TargetSwitch UInt64,
    TargetQueue UInt64,
    TargetDc UInt64,
    TargetAddress String,
    TargetPort UInt16,
    Network UInt8,
    Protocol UInt8,
    Score Float64,
    Rtt Float64,
    Generated DateTime DEFAULT now(),
    Inserted DateTime DEFAULT now(),
    InsertedDate Date DEFAULT today()
) ENGINE = Merge('default', '^probes_');
)");
    }

    class TSchemaMaintainer::TImpl : public TScheduledTask {
    public:
        TImpl()
            : TScheduledTask(TSettings::Get()->GetSchemaInterval(), true)
        {
        }

        inline TThreadPool::TFuture Run() override {
            Futures.clear();
            Maintainers.Clear();

            auto Maintainer = [&](auto&&... args) {
                TTableFactory::TRef ptr(MakeHolder<TTableMaintainer>(args...));
                Futures.push_back(ptr->Run());
                Maintainers.PushBack(std::move(ptr));
            };

            auto Creator = [&](auto&&... args) {
                TTableFactory::TRef ptr(MakeHolder<TTableCreator>(args...));
                Futures.push_back(ptr->Run());
                Maintainers.PushBack(std::move(ptr));
            };

            TTableCreator::TTableMap tableMap;

            tableMap.emplace(TStringBuf("distributed_probes"), DISTRIBUTED_PROBES_TABLE);
            tableMap.emplace(TStringBuf("probes"), LOCAL_PROBES_TABLE);

            for (auto shardIdx : xrange(TLibrarySettings::Get()->GetClickHouseShards().size())) {
                if (TSettings::Get()->AreClickhouseProbesReplicated()) {
                    Maintainer(shardIdx, PROBES_REPLICATED_TABLE_TEMPLATE, PROBES_TABLE_PREFIX, PROBES_TABLE_FORMAT, TSettings::Get()->GetClickhouseTablesCount());
                } else {
                    Maintainer(shardIdx, PROBES_LOCAL_TABLE_TEMPLATE, PROBES_TABLE_PREFIX, PROBES_TABLE_FORMAT, TSettings::Get()->GetClickhouseTablesCount());
                }

                Creator(shardIdx, tableMap);
            }

            return WaitExceptionOrAll(Futures);
        }

    private:
        TVector<TClickhouseClient::TFuture> Futures;
        THolderVector<TTableFactory> Maintainers;
    };

    TSchemaMaintainer::TSchemaMaintainer()
        : TSchemaMaintainer(!TLibrarySettings::Get()->GetClickHouseShards().empty())
    {
    }

    TSchemaMaintainer::TSchemaMaintainer(bool schedule)
        : Impl(MakeHolder<TImpl>())
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TSchemaMaintainer::~TSchemaMaintainer() {
    }

    TThreadPool::TFuture TSchemaMaintainer::SpinAndWait() noexcept {
        return Impl->SpinAndWait();
    }
}
