#include <infra/netmon/seen_hosts.h>
#include <infra/netmon/response_gatherer.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/idl/api.fbs.h>

#include <util/string/builder.h>

namespace NNetmon {
    class TSeenHostsUpdater::TImpl : public TScheduledTask {
    public:
        using TGatherer = TGatherer<NApi::TSeenHostsResponse>;

        TImpl(const TTopologyStorage& topologyStorage)
            : TScheduledTask(TSettings::Get()->GetSeenHostsInterval(), true)
            , TopologyStorage(topologyStorage)
            , Ready(false)
            , EventHub(TVoidEventHub::Make())
        {
        }

        TThreadPool::TFuture Run() override {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTSeenHostsRequest(builder));
            Timer.Reset();
            return TGatherer::Collect("/slicer/v1/seen_hosts", builder).Apply([this](const NThreading::TFuture<TGatherer::TResponseList>& future) {
                auto newSeenHosts(MakeAtomicShared<TTopologyStorage::THostSet>());
                for (const auto& response : future.GetValue()) {
                    for (const auto& host : *response->Hosts()) {
                        auto foundHost = TopologyStorage.FindHost(*host);
                        if (foundHost && !newSeenHosts->contains(foundHost)) {
                            newSeenHosts->emplace(foundHost);
                        }
                    }
                }
                DEBUG_LOG << "Seen hosts updating took " << Timer.Get() << Endl;
                SeenHosts.Swap(newSeenHosts);
                AtomicSet(Ready, true);
                EventHub->Notify();
            });
        }

        inline TTopologyStorage::THostSetBox::TConstValueRef GetHosts() const {
            return SeenHosts.Get();
        }

        inline bool IsReady() const {
            return AtomicGet(Ready);
        }

        const TVoidEventHub& OnChanged() const {
            return *EventHub;
        }

    private:
        const TTopologyStorage& TopologyStorage;
        TTopologyStorage::THostSetBox SeenHosts;
        TAtomic Ready;
        TVoidEventHub::TRef EventHub;
        TSimpleTimer Timer;
    };

    TSeenHostsUpdater::TSeenHostsUpdater(const TTopologyStorage& topologyStorage)
        : Impl(MakeHolder<TImpl>(topologyStorage))
        , SchedulerGuard(Impl->Schedule())
    {
    }

    TSeenHostsUpdater::~TSeenHostsUpdater() = default;

    TTopologyStorage::THostSetBox::TConstValueRef TSeenHostsUpdater::GetHosts() const {
        return Impl->GetHosts();
    }

    bool TSeenHostsUpdater::IsReady() const {
        return Impl->IsReady();
    }

    const TVoidEventHub& TSeenHostsUpdater::OnChanged() const {
        return Impl->OnChanged();
    }

    TSeenHostsMaintainer::TState::TState(TTopologySelector::TBox::TConstValueRef topologySelector, TExpressionId expressionId)
        : TotalHosts(topologySelector->CountHosts(expressionId))
    {
    }

    class TSeenHostsMaintainer::TImpl : public TNonCopyable {
    public:
        TImpl(const TTopologyStorage& topologyStorage, const TSeenHostsUpdater& seenHostsUpdater)
            : TopologyStorage(topologyStorage)
            , SeenHostsUpdater(seenHostsUpdater)
            , StateMap(TStateMap())
        {
        }

        void OnHostsUpdated() {
            TSimpleTimer Timer;
            auto newStateMap(MakeAtomicShared<TStateMap>());
            const auto topologySelector(TopologyStorage.GetTopologySelector());
            for (const auto& host : *SeenHostsUpdater.GetHosts()) {
                const auto* expressionIds(topologySelector->FindSourceExpressions(host));
                if (expressionIds) {
                    for (const auto expressionId : *expressionIds) {
                        auto it(newStateMap->States.find(expressionId));
                        if (it.IsEnd()) {
                            it = newStateMap->States.emplace(std::piecewise_construct,
                                                                std::forward_as_tuple(expressionId),
                                                                std::forward_as_tuple(topologySelector, expressionId)).first;
                        }
                        AddHost(it->second, host, topologySelector, expressionId);
                    }
                }
            }
            newStateMap->Timestamp = TInstant::Now();
            DEBUG_LOG << "Seen hosts state updating took " << Timer.Get() << Endl;
            StateMap.Swap(newStateMap);
        }

        inline TStateMap::TConstValueRef GetStateMap() const {
            return StateMap.Get();
        }

    private:
        void AddHost(TState& state, const TTopology::THostRef host, TTopologySelector::TRef topologySelector, TExpressionId expressionId) {
            state.Hosts.emplace(host);

            auto dcIt(state.CountByDc.find(host.GetDatacenter()));
            if (dcIt != state.CountByDc.end()) {
                ++dcIt->second.first;
            } else {
                TTopologySelector::THostFilterByDatacenter filter(host.GetDatacenter());
                state.CountByDc[host.GetDatacenter()] = std::make_pair(1, topologySelector->CountHosts(expressionId, filter));
            }

            auto lineIt(state.CountByQueue.find(host.GetLine()));
            if (lineIt != state.CountByQueue.end()) {
                ++lineIt->second.first;
            } else {
                TTopologySelector::THostFilterByLine filter(host.GetLine());
                state.CountByQueue[host.GetLine()] = std::make_pair(1, topologySelector->CountHosts(expressionId, filter));
            }
        }

        const TTopologyStorage& TopologyStorage;
        const TSeenHostsUpdater& SeenHostsUpdater;
        TStateMap::TBox StateMap;
    };

    TSeenHostsMaintainer::TSeenHostsMaintainer(const TTopologyStorage& topologyStorage, const TSeenHostsUpdater& seenHostsUpdater)
        : Impl(MakeHolder<TImpl>(topologyStorage, seenHostsUpdater))
        , SeenHostsSubscription(seenHostsUpdater.OnChanged().Subscribe([this]() {
            Impl->OnHostsUpdated();
        }))
    {
    }

    TSeenHostsMaintainer::~TSeenHostsMaintainer() = default;

    TSeenHostsMaintainer::TStateMap::TConstValueRef TSeenHostsMaintainer::GetStateMap() const {
        return Impl->GetStateMap();
    }
}
