#include <infra/netmon/settings.h>
#include <infra/netmon/topology/topology.h>

#include <util/generic/algorithm.h>
#include <util/generic/xrange.h>
#include <util/random/shuffle.h>
#include <util/stream/input.h>
#include <util/string/builder.h>
#include <util/system/info.h>

namespace NNetmon {
    TSettings::TSettings()
        : ClickhouseProbesReplicated(false)
        , ClickhouseTablesCount(5 * 24)

        , ExpressionsFreezed(false)

        , MtnVlanProbesEnabled(false)

        , ProbeFetchingInterval(TDuration::Seconds(5))
        , ProbeFetchingForwardWindow(TDuration::Seconds(15))
        , ProbeFetchingBackwardWindow(TDuration::Seconds(60))

        , KeyFetchingInterval(TDuration::Seconds(1))

        , DcAggregationInterval(TDuration::Seconds(5))
        , QueueAggregationInterval(TDuration::Seconds(15))
        , SwitchAggregationInterval(TDuration::Seconds(60))

        , DcAggregationWindow(TDuration::Minutes(2))
        , QueueAggregationWindow(TDuration::Minutes(2))
        , SwitchAggregationWindow(TDuration::Minutes(5))

        , SeenHostsInterval(DcAggregationInterval)
        , TerminatedHostsInterval(TDuration::Seconds(2))
        , IndexDumpInterval(TDuration::Seconds(5))
        , MonitorInterval(TDuration::Seconds(5))
        , SchemaInterval(TDuration::Minutes(1))

        , ProbeRescheduleInterval()
        , ProbeScheduleTtl(TDuration::Seconds(60))
        , LinkPollerHostMuteDuration(TDuration::Minutes(60))
        , ProbeScheduleProtocols({UDP})
        , ProbeScheduleTrafficClasses({})
        , ProbeScheduleCrossDcTrafficClasses({})
        , VlanInversionSelector({})
        , ScheduledProbesBetweenTwoSwitches({{"default", 3}})
        , ScheduledCrossDcProbesBetweenTwoPods(300)
        , CrossDcProbeScheduleFullMesh(false)
        , Ipv4IntraDcScheduleEnabled(false)
        , ScheduledProbePacketCount(10)
        , LinkPollerFailsThresholdToHostMute(0.05)
        , InterestedHostsChangeThreshold(0.02)

        , NocSlaSwitchMapCapacity(0)

        , HistorySeriesPath("history/series")
        , HistoryStatePath("history/state")

        , DcHistoryTimeToLive(TDuration::Days(14))
        , QueueHistoryTimeToLive(TDuration::Days(7))
        , SwitchHistoryTimeToLive(TDuration::Days(1))

        , ReportsRateLimiterCapacity(0)
        , ReportsRateLimiterRefillRatePerMs(0)

        , InfraUrl("https://infra-api.yandex-team.ru")
        , InfraEnvironmentId("526") // noc-sla

        , RtAuthToken("")
        , RtUrl("https://ro.racktables.yandex-team.ru/api")
        , RtUpdateInterval(TDuration::Minutes(5))
    {
    }

    bool TSettings::TVlanInversionSelector::Contains(const THost& host) const {
        if (!host.IsRtc()) {
            return false;
        }
        return Names.contains(host.GetDatacenter().GetName())
               || (host.GetPod() && Names.contains(host.GetPod()->GetName()));
    }

    void TSettings::Load(const NConfig::TConfig& config) {
        if (!config["clickhouse"].IsNull()) {
            const auto& clickhouseConfig = config["clickhouse"];

            if (!clickhouseConfig["probes_replicated"].IsNull()) {
                ClickhouseProbesReplicated = clickhouseConfig["probes_replicated"].As<bool>(ClickhouseProbesReplicated);
            }

            if (!clickhouseConfig["tables_count"].IsNull()) {
                ClickhouseTablesCount = clickhouseConfig["tables_count"].As<size_t>(ClickhouseTablesCount);
                Y_VERIFY(ClickhouseTablesCount);
            }
        }

        if (!config["history"].IsNull()) {
            const auto& historyConfig = config["history"];
            if (!historyConfig["state_path"].IsNull()) {
                HistoryStatePath = historyConfig["state_path"].As<TString>(HistoryStatePath);
            }
            if (!historyConfig["series_path"].IsNull()) {
                HistorySeriesPath = historyConfig["series_path"].As<TString>(HistorySeriesPath);
            }
            if (!historyConfig["dc_ttl"].IsNull()) {
                TDuration::TryParse(historyConfig["dc_ttl"].As<TString>(), DcHistoryTimeToLive);
            }
            if (!historyConfig["queue_ttl"].IsNull()) {
                TDuration::TryParse(historyConfig["queue_ttl"].As<TString>(), QueueHistoryTimeToLive);
            }
            if (!historyConfig["switch_ttl"].IsNull()) {
                TDuration::TryParse(historyConfig["switch_ttl"].As<TString>(), SwitchHistoryTimeToLive);
            }
        }

        SlicerShards.clear();
        if (!config["slicer"].IsNull()) {
            TLibrarySettings::TryToParseShards(config["slicer"]["shards"], SlicerShards);
        }

        if (!config["sudoers"].IsNull()) {
            for (const auto& element : config["sudoers"].Get<NConfig::TArray>()) {
                Sudoers.emplace(element.As<TString>());
            }
        }

        if (!config["tasks_users"].IsNull()) {
            for (const auto& element : config["tasks_users"].Get<NConfig::TArray>()) {
                TasksUsers.emplace(element.As<TString>());
            }
        }

        if (!config["whitelisted_expressions"].IsNull()) {
            for (const auto& element : config["whitelisted_expressions"].Get<NConfig::TArray>()) {
                WhitelistedExpressions.emplace(element.As<TString>());
            }
        }

        if (!config["expressions_freezed"].IsNull()) {
            ExpressionsFreezed = config["expressions_freezed"].As<bool>(ExpressionsFreezed);
        }

        if (!config["mtn_vlan_probes_enabled"].IsNull()) {
            MtnVlanProbesEnabled = config["mtn_vlan_probes_enabled"].As<bool>(MtnVlanProbesEnabled);
        }

        if (!config["probe_schedule"].IsNull()) {
            const auto& probeScheduleConfig = config["probe_schedule"];

            if (probeScheduleConfig["dc"].IsA<TString>()) {
                TVector<TString> dcs(1, probeScheduleConfig["dc"].As<TString>());
                SetProbeScheduleDcs(dcs);
            } else {
                TVector<TString> dcs;
                for (const auto& element : probeScheduleConfig["dc"].Get<NConfig::TArray>()) {
                    dcs.emplace_back(element.As<TString>());
                }
                SetProbeScheduleDcs(dcs);
            }

            Y_VERIFY(!ProbeScheduleRawDcs.empty());
            for (const auto& dc : ProbeScheduleRawDcs) {
                Y_VERIFY(!dc.empty());
            }
            Y_VERIFY(!ProbeScheduleDcs.empty());

            TDuration::TryParse(probeScheduleConfig["reschedule_interval"].As<TString>(), ProbeRescheduleInterval);
            TDuration::TryParse(probeScheduleConfig["schedule_ttl"].As<TString>(), ProbeScheduleTtl);
            Y_VERIFY(ProbeRescheduleInterval ? ProbeRescheduleInterval > ProbeScheduleTtl : true);
            TDuration::TryParse(probeScheduleConfig["link_poller_host_mute_duration"].As<TString>(), LinkPollerHostMuteDuration);

            if (!probeScheduleConfig["excluded_switches"].IsNull()) {
                for (const auto& element : probeScheduleConfig["excluded_switches"].Get<NConfig::TArray>()) {
                    ProbeScheduleExcludedSwitches.emplace(element.As<TString>());
                }
            }

            if (!probeScheduleConfig["protocols"].IsNull()) {
                for (const auto& element : probeScheduleConfig["protocols"].Get<NConfig::TArray>()) {
                    EProtocolType protocol;
                    if (::TryFromString<NNetmon::EProtocolType>(element.As<TString>(), protocol)) {
                        ProbeScheduleProtocols.emplace(protocol);
                    }
                }
            }

            if (!probeScheduleConfig["traffic_classes"].IsNull()) {
                for (const auto& element : probeScheduleConfig["traffic_classes"].Get<NConfig::TArray>()) {
                    ETrafficClassType trafficClass;
                    if (::TryFromString<NNetmon::ETrafficClassType>(element.As<TString>(), trafficClass)) {
                        ProbeScheduleTrafficClasses.emplace(trafficClass);
                    }
                }
            }

            if (!probeScheduleConfig["cross_dc_traffic_classes"].IsNull()) {
                for (const auto& element : probeScheduleConfig["cross_dc_traffic_classes"].Get<NConfig::TArray>()) {
                    ETrafficClassType trafficClass;
                    if (::TryFromString<NNetmon::ETrafficClassType>(element.As<TString>(), trafficClass)) {
                        ProbeScheduleCrossDcTrafficClasses.emplace(trafficClass);
                    }
                }
            }

            if (!probeScheduleConfig["vlan_inversion_selector"].IsNull()) {
                // selector should contain a list of datacenter and/or pod names
                THashSet<TString> selector;
                for (const auto& element : probeScheduleConfig["vlan_inversion_selector"].Get<NConfig::TArray>()) {
                    selector.emplace(element.As<TString>());
                }
                VlanInversionSelector = TVlanInversionSelector(selector);
            }

            if (!probeScheduleConfig["probes_between_two_switches"].IsNull()) {
                const auto& cntProbesConfig = probeScheduleConfig["probes_between_two_switches"];

                if (cntProbesConfig.IsA<NConfig::TDict>()) {
                    for (const auto& [k, v] : cntProbesConfig.Get<NConfig::TDict>()) {
                        ScheduledProbesBetweenTwoSwitches[k] = v.As<size_t>();
                    }
                } else {
                    ScheduledProbesBetweenTwoSwitches["default"] = cntProbesConfig.As<size_t>();
                }
            }
            ScheduledCrossDcProbesBetweenTwoPods = probeScheduleConfig["cross_dc_probes_between_two_pods"].As<size_t>(ScheduledCrossDcProbesBetweenTwoPods);
            CrossDcProbeScheduleFullMesh = probeScheduleConfig["cross_dc_full_mesh"].As<bool>(CrossDcProbeScheduleFullMesh);

            Ipv4IntraDcScheduleEnabled = probeScheduleConfig["enable_ipv4"].As<bool>(Ipv4IntraDcScheduleEnabled);

            ScheduledProbePacketCount = probeScheduleConfig["packet_count"].As<size_t>(ScheduledProbePacketCount);
            LinkPollerFailsThresholdToHostMute = probeScheduleConfig["link_poller_fails_threshold_to_host_mute"].As<double>(LinkPollerFailsThresholdToHostMute);
            InterestedHostsChangeThreshold = probeScheduleConfig["interested_hosts_threshold"].As<double>(InterestedHostsChangeThreshold);

            if (!probeScheduleConfig["netmon_urls"].IsNull()) {
                NetmonUrls.clear();
                for (const auto& dcUrls : probeScheduleConfig["netmon_urls"].Get<NConfig::TDict>()) {
                    auto& storedUrls = NetmonUrls[dcUrls.first];
                    for (const auto& url : dcUrls.second.Get<NConfig::TArray>()) {
                        storedUrls.emplace_back(url.As<TString>());
                    }
                    ShuffleRange(storedUrls);
                }
            }
        }

        if (!config["noc_sla"].IsNull()) {
            const auto& nocSlaConfig = config["noc_sla"];

            if (!nocSlaConfig["per_switch"].IsNull()) {
                const auto& perSwitchConfig = nocSlaConfig["per_switch"];
                NocSlaSwitchMapCapacity = perSwitchConfig["map_capacity"].As<size_t>(NocSlaSwitchMapCapacity);
            }

            TDuration::TryParse(nocSlaConfig["metric_window"].As<TString>(), NocSlaMetricWindow);
        }

        if (!config["reports"].IsNull()) {
            const auto& reportsConfig = config["reports"];

            if (!reportsConfig["rate_limiter"].IsNull()) {
                const auto& reportsRateLimiterConfig = reportsConfig["rate_limiter"];

                ReportsRateLimiterCapacity = reportsRateLimiterConfig["capacity"].As<size_t>(ReportsRateLimiterCapacity);
                ReportsRateLimiterRefillRatePerMs = reportsRateLimiterConfig["refill_rate_per_ms"].As<size_t>(ReportsRateLimiterRefillRatePerMs);
            }
        }

        if (!config["solomon"].IsNull()) {
            const auto& solomonConfig = config["solomon"];

            SolomonPushUrl = solomonConfig["push_url"].As<TString>();
            SolomonPushToken = solomonConfig["push_token"].As<TString>();

            if (!solomonConfig["labels"].IsNull()) {
                SolomonLabels.clear();
                for (const auto& [label, value] : solomonConfig["labels"].Get<NConfig::TDict>()) {
                    SolomonLabels.emplace_back(label, value.As<TString>());
                }
            }
        }

        InfraUrl = config["infra_url"].As<TString>(InfraUrl);
        InfraEnvironmentId = config["infra_environment_id"].As<TString>(InfraEnvironmentId);

        if (!config["rt"].IsNull()) {
            const auto& rtConfig = config["rt"];

            RtAuthToken = rtConfig["token"].As<TString>(RtAuthToken);
            RtUrl = rtConfig["url"].As<TString>(RtUrl);
            TDuration::TryParse(rtConfig["update_interval"].As<TString>(), RtUpdateInterval);
        }
    }
}
