#pragma once

#include <infra/netmon/library/settings.h>
#include <infra/netmon/topology/types.h>
#include <infra/netmon/topology/settings.h>

#include <util/datetime/base.h>
#include <util/generic/map.h>
#include <util/generic/set.h>
#include <util/generic/hash_set.h>
#include <util/folder/path.h>

class IInputStream;

namespace NNetmon {
    class THost;

    class TSettings {
        Y_DECLARE_SINGLETON_FRIEND()
    public:
        using TShard = TLibrarySettings::TShard;
        using TShardList = TLibrarySettings::TShardList;

        class TVlanInversionSelector {
        public:
            TVlanInversionSelector(const THashSet<TString>& names)
                : Names(names)
            {
            }

            bool Contains(const THost& host) const;
            bool Empty() const {
                return Names.empty();
            }

        private:
            THashSet<TString> Names;
        };

        const TShardList& GetSlicerShards() const {
            return SlicerShards;
        }

        const TSet<TString>& GetSudoers() const {
            return Sudoers;
        }

        const TSet<TString>& GetTasksUsers() const {
            return TasksUsers;
        }

        const TSet<TString>& GetWhitelistedExpressions() const {
            return WhitelistedExpressions;
        }

        bool AreClickhouseProbesReplicated() const {
            return ClickhouseProbesReplicated;
        }
        size_t GetClickhouseTablesCount() const {
            return ClickhouseTablesCount;
        }

        bool AreExpressionsFreezed() const {
            return ExpressionsFreezed;
        }

        bool AreMtnVlanProbesEnabled() const {
            return MtnVlanProbesEnabled;
        }

        const TDuration& GetProbeFetchingInterval() const {
            return ProbeFetchingInterval;
        }
        const TDuration& GetProbeFetchingForwardWindow() const {
            return ProbeFetchingForwardWindow;
        }
        const TDuration& GetProbeFetchingBackwardWindow() const {
            return ProbeFetchingBackwardWindow;
        }

        const TDuration& GetKeyFetchingInterval() const {
            return KeyFetchingInterval;
        }

        const TDuration& GetDcAggregationInterval() const {
            return DcAggregationInterval;
        }
        const TDuration& GetSwitchAggregationInterval() const {
            return SwitchAggregationInterval;
        }
        const TDuration& GetLineAggregationInterval() const {
            return QueueAggregationInterval;
        }

        const TDuration& GetDcAggregationWindow() const {
            return DcAggregationWindow;
        }
        const TDuration& GetSwitchAggregationWindow() const {
            return SwitchAggregationWindow;
        }
        const TDuration& GetLineAggregationWindow() const {
            return QueueAggregationWindow;
        }

        const TDuration& GetSeenHostsInterval() const {
            return SeenHostsInterval;
        }
        const TDuration& GetTerminatedHostsInterval() const {
            return TerminatedHostsInterval;
        }
        const TDuration& GetIndexDumpInterval() const {
            return IndexDumpInterval;
        }
        const TDuration& GetMonitorInterval() const {
            return MonitorInterval;
        }
        const TDuration& GetSchemaInterval() const {
            return SchemaInterval;
        }

        void SetProbeScheduleDcs(const TVector<TString>& dcs) {
            ProbeScheduleRawDcs = dcs;

            THashSet<TString> allMappedDcs;
            for (const auto& dc : dcs) {
                const auto& currentMappedDcs = TTopologySettings::Get()->GetMappedDcs(dc);
                allMappedDcs.insert(begin(currentMappedDcs), end(currentMappedDcs));
            }
            ProbeScheduleDcs.assign(begin(allMappedDcs), end(allMappedDcs));
        }

        const TVector<TString>& GetProbeScheduleDcs() const {
            return ProbeScheduleDcs;
        }
        const TVector<TString>& GetProbeScheduleRawDcs() const {
            return ProbeScheduleRawDcs;
        }

        const TDuration& GetProbeRescheduleInterval() const {
            return ProbeRescheduleInterval;
        }
        void SetProbeRescheduleInterval(const TDuration& value) {
            ProbeRescheduleInterval = value;
        }

        const TDuration& GetLinkPollerHostMuteDuration() const {
            return LinkPollerHostMuteDuration;
        }
        void SetLinkPollerHostMuteDuration(const TDuration& value) {
            LinkPollerHostMuteDuration = value;
        }

        const TDuration& GetProbeScheduleTtl() const {
            return ProbeScheduleTtl;
        }
        void SetProbeScheduleTtl(const TDuration& value) {
            ProbeScheduleTtl = value;
        }
        const THashSet<TString>& GetProbeScheduleExcludedSwitches() const {
            return ProbeScheduleExcludedSwitches;
        }
        const THashSet<EProtocolType>& GetProbeScheduleProtocols() const {
            return ProbeScheduleProtocols;
        }
        const THashSet<ETrafficClassType>& GetProbeScheduleTrafficClasses() const {
            return ProbeScheduleTrafficClasses;
        }
        const THashSet<ETrafficClassType>& GetProbeScheduleCrossDcTrafficClasses() const {
            return ProbeScheduleCrossDcTrafficClasses;
        }
        const TVlanInversionSelector& GetVlanInversionSelector() const {
            return VlanInversionSelector;
        }
        THashMap<TString, size_t> GetScheduledProbesBetweenTwoSwitches() const {
            return ScheduledProbesBetweenTwoSwitches;
        }

        size_t GetScheduledCrossDcProbesBetweenTwoPods() const {
            return ScheduledCrossDcProbesBetweenTwoPods;
        }
        void SetScheduledCrossDcProbesBetweenTwoPods(size_t value) {
            ScheduledCrossDcProbesBetweenTwoPods = value;
        }

        bool IsCrossDcProbeScheduleFullMesh() const {
            return CrossDcProbeScheduleFullMesh;
        }
        void SetCrossDcProbeScheduleFullMesh(bool value) {
            CrossDcProbeScheduleFullMesh = value;
        }

        bool IsIpv4IntraDcScheduleEnabled() const {
            return Ipv4IntraDcScheduleEnabled;
        }
        void SetIpv4IntraDcScheduleEnabled(bool value) {
            Ipv4IntraDcScheduleEnabled = value;
        }

        size_t GetScheduledProbePacketCount() const {
            return ScheduledProbePacketCount;
        }

        double GetLinkPollerFailsThresholdToHostMute() const {
            return LinkPollerFailsThresholdToHostMute;
        }

        double GetInterestedHostsChangeThreshold() const {
            return InterestedHostsChangeThreshold;
        }
        // returns map of dc name => vector of replica urls
        const TMap<TString, TVector<TString>>& GetNetmonUrls() const {
            return NetmonUrls;
        }

        const TDuration& GetNocSlaMetricWindow() const {
            return NocSlaMetricWindow;
        }
        void SetNocSlaMetricWindow(const TDuration& duration) {
            NocSlaMetricWindow = duration;
        }
        size_t GetNocSlaSwitchMapCapacity() const {
            return NocSlaSwitchMapCapacity;
        }
        void SetNocSlaSwitchMapCapacity(size_t capacity) {
            NocSlaSwitchMapCapacity = capacity;
        }

        const TFsPath& GetHistorySeriesPath() const {
            return HistorySeriesPath;
        }
        void SetHistorySeriesPath(const TFsPath& path) {
            HistorySeriesPath = path;
        }

        const TFsPath& GetHistoryStatePath() const {
            return HistoryStatePath;
        }
        void SetHistoryStatePath(const TFsPath& path) {
            HistoryStatePath = path;
        }

        const TDuration GetDcHistoryTimeToLive() const {
            return DcHistoryTimeToLive;
        }
        const TDuration GetQueueHistoryTimeToLive() const {
            return QueueHistoryTimeToLive;
        }
        const TDuration GetSwitchHistoryTimeToLive() const {
            return SwitchHistoryTimeToLive;
        }

        size_t GetReportsRateLimiterCapacity() const {
            return ReportsRateLimiterCapacity;
        }
        size_t GetReportsRateLimiterRefillRatePerMs() const {
            return ReportsRateLimiterRefillRatePerMs;
        }

        const TString& GetSolomonPushUrl() const {
            return SolomonPushUrl;
        }
        const TString& GetSolomonPushToken() const {
            return SolomonPushToken;
        }
        const TVector<std::pair<TString, TString>>& GetSolomonLabels() const {
            return SolomonLabels;
        }

        const TString& GetInfraUrl() const {
            return InfraUrl;
        }
        void SetInfraUrl(const TString& url) {
            InfraUrl = url;
        }

        const TString& GetInfraEnvironmentId() const {
            return InfraEnvironmentId;
        }
        void SetInfraEnvironmentId(const TString& environmentId) {
            InfraEnvironmentId = environmentId;
        }

        const TString& GetRtAuthToken() const {
            return RtAuthToken;
        }
        const TString& GetRtUrl() const {
            return RtUrl;
        }
        const TDuration& GetRtUpdateInterval() const {
            return RtUpdateInterval;
        }

        void Load(const NConfig::TConfig& config);

        static TSettings* Get() {
            return SingletonWithPriority<TSettings, 100000>();
        }

    private:
        TSettings();

        TShardList SlicerShards;

        TSet<TString> Sudoers;
        TSet<TString> TasksUsers;
        TSet<TString> WhitelistedExpressions;

        bool ClickhouseProbesReplicated;
        size_t ClickhouseTablesCount;

        bool ExpressionsFreezed;

        bool MtnVlanProbesEnabled;

        TDuration ProbeFetchingInterval;
        TDuration ProbeFetchingForwardWindow;
        TDuration ProbeFetchingBackwardWindow;

        TDuration KeyFetchingInterval;

        TDuration DcAggregationInterval;
        TDuration QueueAggregationInterval;
        TDuration SwitchAggregationInterval;

        TDuration DcAggregationWindow;
        TDuration QueueAggregationWindow;
        TDuration SwitchAggregationWindow;

        TDuration SeenHostsInterval;
        TDuration TerminatedHostsInterval;
        TDuration IndexDumpInterval;
        TDuration MonitorInterval;
        TDuration SchemaInterval;

        TVector<TString> ProbeScheduleRawDcs;
        TVector<TString> ProbeScheduleDcs;
        TDuration ProbeRescheduleInterval;
        TDuration ProbeScheduleTtl;
        TDuration LinkPollerHostMuteDuration;
        THashSet<TString> ProbeScheduleExcludedSwitches;
        THashSet<EProtocolType> ProbeScheduleProtocols;
        THashSet<ETrafficClassType> ProbeScheduleTrafficClasses;
        THashSet<ETrafficClassType> ProbeScheduleCrossDcTrafficClasses;
        TVlanInversionSelector VlanInversionSelector;
        THashMap<TString, size_t> ScheduledProbesBetweenTwoSwitches;
        size_t ScheduledCrossDcProbesBetweenTwoPods;
        bool CrossDcProbeScheduleFullMesh;
        bool Ipv4IntraDcScheduleEnabled;
        size_t ScheduledProbePacketCount;
        double LinkPollerFailsThresholdToHostMute;
        double InterestedHostsChangeThreshold;
        TMap<TString, TVector<TString>> NetmonUrls;

        TDuration NocSlaMetricWindow;
        size_t NocSlaSwitchMapCapacity;

        TFsPath HistorySeriesPath;
        TFsPath HistoryStatePath;

        TDuration DcHistoryTimeToLive;
        TDuration QueueHistoryTimeToLive;
        TDuration SwitchHistoryTimeToLive;

        size_t ReportsRateLimiterCapacity;
        size_t ReportsRateLimiterRefillRatePerMs;

        TString SolomonPushUrl;
        TString SolomonPushToken;
        TVector<std::pair<TString, TString>> SolomonLabels;

        TString InfraUrl;
        TString InfraEnvironmentId;

        TString RtAuthToken;
        TString RtUrl;
        TDuration RtUpdateInterval;
    };
}
