#include <infra/netmon/share_dead_hosts.h>
#include <infra/netmon/library/requester.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/topology/settings.h>

#include <library/cpp/json/writer/json.h>

#include <util/generic/algorithm.h>
#include <util/generic/vector.h>
#include <util/random/random.h>
#include <util/stream/zlib.h>
#include <util/system/hostname.h>

namespace {
    TMaybe<size_t> GetCurrentReplicaIndex(const TVector<TString>& urls) {
        auto sortedUrls = urls;
        Sort(sortedUrls);
        auto it = FindIf(sortedUrls, [](const TString& url) {
            return url.Contains(HostName());
        });
        if (it == end(sortedUrls)) {
            return Nothing();
        } else {
            return it - begin(sortedUrls);
        }
    }
}

namespace NNetmon {
    TDuration CalculateWalleUpdaterTimeOffset() {
        TMaybe<size_t> replicaIndex, replicaCount;
        const auto& netmonUrls = TSettings::Get()->GetNetmonUrls();
        if (!TSettings::Get()->GetProbeScheduleRawDcs().empty()) {
            const auto& dc = TSettings::Get()->GetProbeScheduleRawDcs().front();
            if (netmonUrls.contains(dc)) {
                const auto& replicaUrls = netmonUrls.at(dc);
                replicaCount = replicaUrls.size();
                replicaIndex = GetCurrentReplicaIndex(replicaUrls);
            } else if (!netmonUrls.empty()) {
                ERROR_LOG << "Dc " << dc << " not found in netmon urls" << Endl;
            }
        }

        auto interval = TTopologySettings::Get()->GetWalleInterval();
        if (replicaIndex && replicaCount) {
            auto offset = *replicaIndex * (interval / *replicaCount);
            INFO_LOG << "Walle updater offset is " << offset << Endl;
            return offset;
        } else {
            return TDuration::MilliSeconds(RandomNumber<ui64>(interval.MilliSeconds()));
        }
    }

    NThreading::TFuture<void> ShareDeadHosts(const TWalleUpdater& walleUpdater) {
        const auto& netmonUrls = TSettings::Get()->GetNetmonUrls();
        const auto& dc = TSettings::Get()->GetProbeScheduleRawDcs().front();
        if (!netmonUrls.contains(dc)) {
            return NThreading::MakeFuture();
        }

        NJsonWriter::TBuf jsonHosts;
        jsonHosts
            .BeginObject()
            .WriteKey("hosts")
            .BeginList();
        for (const auto& host : *walleUpdater.GetDeadHosts()) {
            jsonHosts.WriteString(host->GetName());
        }
        jsonHosts
            .EndList()
            .EndObject();

        TStringStream compressed;
        TZLibCompress compressor(&compressed, ZLib::ZLib);
        compressor << jsonHosts.Str();
        compressor.Flush();

        TVector<THttpRequester::TFuture> futures;
        for (const auto& baseUrl : netmonUrls.at(dc)) {
            if (baseUrl.Contains(HostName())) {
                continue;
            }
            TString endpoint = baseUrl + "/api/v1/replace_dead_hosts";
            try {
                futures.emplace_back(THttpRequester::Get()->MakeRequest(
                    endpoint,
                    {},
                    NHttp::TFetchOptions()
                        .SetContentType("application/zlib")
                        .SetPostData(compressed.Str())
                ).Subscribe([endpoint](const THttpRequester::TFuture& future) {
                    try {
                        future.GetValue();
                    } catch (...) {
                        ERROR_LOG << "Request to " << endpoint << " failed: "
                                  << CurrentExceptionMessage() << Endl;
                    }
                }));
            } catch (...) {
                ERROR_LOG << "Can't make request to " << endpoint << ": "
                          << CurrentExceptionMessage() << Endl;
            }
        }
        return WaitAll(futures);
    }
}
