#include <infra/netmon/slice_collector.h>
#include <infra/netmon/response_gatherer.h>
#include <infra/netmon/state_picker.h>
#include <infra/netmon/library/scheduler.h>
#include <infra/netmon/metrics.h>

namespace NNetmon {
    namespace {
        template <class T, class TIndexType>
        class TBaseCollector: public TScheduledTask, public TAtomicRefCount<TBaseCollector<T, TIndexType>> {
        public:
            using TRef = TIntrusivePtr<T>;

            using TIndex = TIndexType;
            using TIndexMap = THashMap<TProbeAggregatorKey, const typename TIndex::TRef>;
            using TIndexMapBox = TAtomicLockedBox<TIndexMap>;

            TBaseCollector()
                : TScheduledTask(TSettings::Get()->GetKeyFetchingInterval())
            {
            }

            virtual ~TBaseCollector() = default;

            virtual TInstant GetDeadline() const = 0;

            void FillKeys(TProbeAggregatorKeySet& keys) const {
                const auto stateMap(IndexMapBox.Get());
                for (const auto& pair : *stateMap) {
                    keys.insert(pair.first);
                }
            }

            typename TIndex::TRef GetIndex(const TProbeAggregatorKey& key) const {
                const auto indexMap(IndexMapBox.Get());
                auto it = indexMap->find(key);
                if (!it.IsEnd()) {
                    return it->second;
                } else {
                    return {};
                }
            }

            bool IsReady() const {
                auto indexMap(IndexMapBox.Get());
                if (!indexMap->empty()) {
                    TInstant deadline(GetDeadline());
                    for (const auto& pair : *indexMap) {
                        if (pair.second->GetGenerated() >= deadline) {
                            return true;
                        }
                    }
                    return false;
                }
                return true;
            }

            void Merge(const TIndexMap& incomingIndexMap) {
                IndexMapBox.Apply([&incomingIndexMap, this] (typename TIndexMapBox::TConstValueRef& prevIndexMap) {
                    typename TIndexMapBox::TValueRef indexMap(MakeAtomicShared<TIndexMap>());
                    indexMap->reserve(prevIndexMap->size());

                    TInstant deadline(GetDeadline());
                    for (const auto& pair : *prevIndexMap) {
                        if (pair.second->GetGenerated() >= deadline) {
                            indexMap->emplace(pair.first, pair.second);
                        }
                    }

                    for (const auto& pair : incomingIndexMap) {
                        auto it = indexMap->find(pair.first);
                        if (!it.IsEnd()) {
                            indexMap->erase(it);
                        }
                        indexMap->emplace(pair.first, pair.second);
                    }

                    return indexMap;
                });
            }

        private:
            TIndexMapBox IndexMapBox;
        };

        template <class TCollector, class TResponse, class TExecutor>
        class TSliceRequester: public IStateRequester {
        public:
            using TIndex = typename TCollector::TIndex;
            using TInstantiatedGatherer = TGatherer<TResponse, TExecutor>;

            TSliceRequester(const TTopologyStorage& topologyStorage, TCollector& collector,
                            NApi::ELevel level, const TDuration& interval, std::size_t batchSize)
                : TopologyStorage(topologyStorage)
                , Collector(collector)
                , Level(level)
                , Interval(interval)
                , BatchSize(batchSize)
            {
            }

            inline NApi::ELevel GetLevel() const override {
                return Level;
            }

            inline const TDuration& GetInterval() const override {
                return Interval;
            }

            inline std::size_t GetRequestLimit() const override {
                return TExecutor::Get()->Size();
            }

            inline std::size_t GetBatchSize() const override {
                return BatchSize;
            }

            TThreadPool::TFuture Collect(const TString& path, const flatbuffers::FlatBufferBuilder& builder) override {
                return TInstantiatedGatherer::Collect(path, builder).Apply([this] (const typename TInstantiatedGatherer::TFuture& future_) {
                    typename TCollector::TIndexMap indexMap;
                    for (const auto& response : future_.GetValue()) {
                        for (const auto& index : *response->Indexes()) {
                            auto partialIndex(TIndex::Make(TopologyStorage, *index));
                            auto it = indexMap.find(partialIndex->GetKey());
                            if (it.IsEnd()) {
                                indexMap.emplace(partialIndex->GetKey(), partialIndex);
                            } else {
                                it->second->MergeFrom(*partialIndex);
                            }
                        }
                    }
                    Collector.Merge(indexMap);
                });
            }

        private:
            const TTopologyStorage& TopologyStorage;
            TCollector& Collector;
            const NApi::ELevel Level;
            const TDuration Interval;
            const std::size_t BatchSize;
        };

        class TSwitchCollector: public TBaseCollector<TSwitchCollector, TSwitchPairIndex> {
        public:
            TSwitchCollector(const TTopologyStorage& topologyStorage)
                : TBaseCollector()
                , Requester(
                    topologyStorage, *this, NApi::ELevel_LEVEL_SWITCH,
                    TSettings::Get()->GetSwitchAggregationInterval(), 16
                )
                , Picker(Requester)
            {
            }

            TThreadPool::TFuture Run() override {
                return Picker.Run();
            }

        protected:
            TInstant GetDeadline() const override {
                return Picker.GetDeadline();
            }

        private:
            class TExecutor: public TCustomThreadExecutor<TExecutor> {
            public:
                TExecutor()
                    : TCustomThreadExecutor("SwitchCollector")
                {
                }
            };

            TSliceRequester<TSwitchCollector, NApi::TSwitchStateResponse, TExecutor> Requester;
            TStatePicker Picker;
        };

        class TLineCollector: public TBaseCollector<TLineCollector, TLinePairIndex> {
        public:
            TLineCollector(const TTopologyStorage& topologyStorage)
                : TBaseCollector()
                , Requester(
                    topologyStorage, *this, NApi::ELevel_LEVEL_QUEUE,
                    TSettings::Get()->GetLineAggregationInterval(), 128
                )
                , Picker(Requester)
            {
            }

            TThreadPool::TFuture Run() override {
                return Picker.Run();
            }

        protected:
            TInstant GetDeadline() const override {
                return Picker.GetDeadline();
            }

        private:
            class TExecutor: public TCustomThreadExecutor<TExecutor> {
            public:
                TExecutor()
                    : TCustomThreadExecutor("LineCollector")
                {
                }
            };

            TSliceRequester<TLineCollector, NApi::TLineStateResponse, TExecutor> Requester;
            TStatePicker Picker;
        };

        class TDatacenterCollector: public TBaseCollector<TDatacenterCollector, TDatacenterPairIndex> {
        public:
            TDatacenterCollector(const TTopologyStorage& topologyStorage)
                : TBaseCollector()
                , Requester(
                    topologyStorage, *this, NApi::ELevel_LEVEL_DATACENTER,
                    TSettings::Get()->GetDcAggregationInterval(), 256
                )
                , Picker(Requester)
            {
            }

            TThreadPool::TFuture Run() override {
                return Picker.Run();
            }

        protected:
            TInstant GetDeadline() const override {
                return Picker.GetDeadline();
            }

        private:
            class TExecutor: public TCustomThreadExecutor<TExecutor> {
            public:
                TExecutor()
                    : TCustomThreadExecutor("DcCollector")
                {
                }
            };

            TSliceRequester<TDatacenterCollector, NApi::TDatacenterStateResponse, TExecutor> Requester;
            TStatePicker Picker;
        };
    }

    class TSliceCollector::TImpl {
    public:
        TImpl(const TTopologyStorage& topologyStorage)
            : TopologyStorage(topologyStorage)
            , SwitchCollector(MakeIntrusive<TSwitchCollector>(TopologyStorage))
            , QueueCollector(MakeIntrusive<TLineCollector>(TopologyStorage))
            , DatacenterCollector(MakeIntrusive<TDatacenterCollector>(TopologyStorage))
            , SwitchSchedulerGuard(SwitchCollector->Schedule())
            , QueueSchedulerGuard(QueueCollector->Schedule())
            , DatacenterSchedulerGuard(DatacenterCollector->Schedule())
        {
        }

        inline void FillKeys(TProbeAggregatorKeySet& keys) const {
            DatacenterCollector->FillKeys(keys);
        }

        inline TSwitchPairIndex::TRef GetSwitchIndex(const TProbeAggregatorKey& key) const {
            return SwitchCollector->GetIndex(key);
        }

        inline TLinePairIndex::TRef GetLineIndex(const TProbeAggregatorKey& key) const {
            return QueueCollector->GetIndex(key);
        }

        inline TDatacenterPairIndex::TRef GetDatacenterIndex(const TProbeAggregatorKey& key) const {
            return DatacenterCollector->GetIndex(key);
        }

        inline bool IsReady() const {
            return (
                DatacenterCollector->IsReady()
                && QueueCollector->IsReady()
                && SwitchCollector->IsReady()
            );
        }

    private:
        const TTopologyStorage& TopologyStorage;

        TSwitchCollector::TRef SwitchCollector;
        TLineCollector::TRef QueueCollector;
        TDatacenterCollector::TRef DatacenterCollector;

        TScheduledTask::TTaskGuard SwitchSchedulerGuard;
        TScheduledTask::TTaskGuard QueueSchedulerGuard;
        TScheduledTask::TTaskGuard DatacenterSchedulerGuard;
    };

    TSliceCollector::TSliceCollector(const TTopologyStorage& topologyStorage)
        : Impl(MakeHolder<TImpl>(topologyStorage))
    {
    }

    TSliceCollector::~TSliceCollector() {
    }

    void TSliceCollector::FillKeys(TProbeAggregatorKeySet& keys) const {
        return Impl->FillKeys(keys);
    }

    TSwitchPairIndex::TRef TSliceCollector::GetSwitchIndex(const TProbeAggregatorKey& key) const {
        return Impl->GetSwitchIndex(key);
    }

    TLinePairIndex::TRef TSliceCollector::GetLineIndex(const TProbeAggregatorKey& key) const {
        return Impl->GetLineIndex(key);
    }

    TDatacenterPairIndex::TRef TSliceCollector::GetDatacenterIndex(const TProbeAggregatorKey& key) const {
        return Impl->GetDatacenterIndex(key);
    }

    NThreading::TFuture<TSliceCollector::TProbesResponseList> TSliceCollector::CollectProbes(
            const flatbuffers::FlatBufferBuilder& builder) const {
        return TGatherer<NApi::TProbesResponse>::Collect("/slicer/v1/probes", builder);
    }

    bool TSliceCollector::IsReady() const {
        return Impl->IsReady();
    }
}
