#pragma once

#include <util/stream/str.h>

#include <infra/netmon/history_chunk.h>
#include <infra/netmon/probe.h>

namespace NNetmon {
    namespace {
        inline TProbeAggregatorKey ParseAggregatorKey(const THistoryTuple& raw) noexcept {
            return TProbeAggregatorKey(
                static_cast<TExpressionId>(std::get<0>(raw)),
                static_cast<ENetworkType>(std::get<1>(raw)),
                static_cast<EProtocolType>(std::get<2>(raw))
            );
        }

        template <class TPairKey>
        inline TPairKey ParseRawTuple(const TTopologyStorage& topologyStorage, const THistoryTuple& raw) noexcept;

        template <>
        inline TSwitchPairKey ParseRawTuple<TSwitchPairKey>(const TTopologyStorage& topologyStorage, const THistoryTuple& raw) noexcept {
            const NCommon::TSwitch target(std::get<3>(raw));
            const NCommon::TSwitch source(std::get<4>(raw));
            const NCommon::TSwitchPairKey key(source, target);
            return {topologyStorage, key};
        }

        template <>
        inline TLinePairKey ParseRawTuple<TLinePairKey>(const TTopologyStorage& topologyStorage, const THistoryTuple& raw) noexcept {
            const NCommon::TLine target(std::get<3>(raw));
            const NCommon::TLine source(std::get<4>(raw));
            const NCommon::TLinePairKey key(source, target);
            return {topologyStorage, key};
        }

        template <>
        inline TDatacenterPairKey ParseRawTuple<TDatacenterPairKey>(const TTopologyStorage& topologyStorage, const THistoryTuple& raw) noexcept {
            const NCommon::TDatacenter target(std::get<3>(raw));
            const NCommon::TDatacenter source(std::get<4>(raw));
            const NCommon::TDatacenterPairKey key(source, target);
            return {topologyStorage, key};
        }

        template <class T, class TPairKey>
        class THistoryBaseKey {
        public:
            explicit inline THistoryBaseKey(const TProbeAggregatorKey& aggregatorKey, const TPairKey& pairKey)
                : AggregatorKey(aggregatorKey)
                , PairKey(pairKey)
            {
            }

            explicit inline THistoryBaseKey(const TTopologyStorage& topologyStorage, const THistoryTuple& raw)
                : AggregatorKey(ParseAggregatorKey(raw))
                , PairKey(ParseRawTuple<TPairKey>(topologyStorage, raw))
            {
            }

            inline const TProbeAggregatorKey& GetAggregatorKey() const noexcept {
                return AggregatorKey;
            }
            inline const TPairKey& GetPairKey() const noexcept {
                return PairKey;
            }

            inline THistoryTuple AsTuple() const {
                return {
                    AggregatorKey.GetExpressionId(),
                    AggregatorKey.GetNetwork(),
                    AggregatorKey.GetProtocol(),
                    PairKey.GetTarget()->GetReducedId(),
                    PairKey.GetSource()->GetReducedId()
                };
            }

            inline bool operator<(const T& rhs) const noexcept {
                return std::tie(AggregatorKey, PairKey) < std::tie(rhs.AggregatorKey, rhs.PairKey);
            }
            inline bool operator==(const T& rhs) const noexcept {
                return std::tie(AggregatorKey, PairKey) == std::tie(rhs.AggregatorKey, rhs.PairKey);
            }

        private:
            const TProbeAggregatorKey AggregatorKey;
            const TPairKey PairKey;
        };
    }

    class THistorySwitchPairKey : public THistoryBaseKey<THistorySwitchPairKey, TSwitchPairKey> {
    public:
        using THistoryBaseKey::THistoryBaseKey;
    };

    class THistoryLinePairKey : public THistoryBaseKey<THistoryLinePairKey, TLinePairKey> {
    public:
        using THistoryBaseKey::THistoryBaseKey;
    };

    class THistoryDatacenterPairKey : public THistoryBaseKey<THistoryDatacenterPairKey, TDatacenterPairKey> {
    public:
        using THistoryBaseKey::THistoryBaseKey;
    };

    class THistoryRootKey {
    public:
        explicit inline THistoryRootKey(const TProbeAggregatorKey& aggregatorKey)
            : AggregatorKey(aggregatorKey)
        {
        }

        explicit inline THistoryRootKey(const TTopologyStorage& topologyStorage, const THistoryTuple& raw)
            : AggregatorKey(ParseAggregatorKey(raw))
        {
            Y_UNUSED(topologyStorage);
        }

        inline const TProbeAggregatorKey& GetAggregatorKey() const noexcept {
            return AggregatorKey;
        }

        inline THistoryTuple AsTuple() const {
            return {
                AggregatorKey.GetExpressionId(),
                AggregatorKey.GetNetwork(),
                AggregatorKey.GetProtocol(),
                0,
                0
            };
        }

        inline bool operator<(const THistoryRootKey& rhs) const noexcept {
            return AggregatorKey < rhs.AggregatorKey;
        }
        inline bool operator==(const THistoryRootKey& rhs) const noexcept {
            return AggregatorKey == rhs.AggregatorKey;
        }

    private:
        const TProbeAggregatorKey AggregatorKey;
    };

    inline THistoryRootKey ToHistoryKey(const TProbeAggregatorKey& aggregatorKey) noexcept {
        return THistoryRootKey(aggregatorKey);
    }

    inline THistoryDatacenterPairKey ToHistoryKey(const TProbeAggregatorKey& aggregatorKey, const TDatacenterPairKey& pairKey) noexcept {
        return THistoryDatacenterPairKey(aggregatorKey, pairKey);
    }

    inline THistoryLinePairKey ToHistoryKey(const TProbeAggregatorKey& aggregatorKey, const TLinePairKey& pairKey) noexcept {
        return THistoryLinePairKey(aggregatorKey, pairKey);
    }

    inline THistorySwitchPairKey ToHistoryKey(const TProbeAggregatorKey& aggregatorKey, const TSwitchPairKey& pairKey) noexcept {
        return THistorySwitchPairKey(aggregatorKey, pairKey);
    }
}
