#include <infra/netmon/state_index.h>

#include <util/generic/xrange.h>

namespace NNetmon {
    TAtomic MemPoolStatsPairStateCount;

    namespace {
        template <class TIndex>
        inline void MergeIndexes(TIndex& left, const TIndex& right) {
            for (auto& pair : TMergeIterator<TIndex, TIndex>(left, right)) {
                auto* first(pair.first);
                if (first == nullptr) {
                    first = &left.CreateState(pair.second->GetKey());
                }
                if (pair.second != nullptr) {
                    first->MergeFrom(*pair.second);
                }
            }
        }
    }

    void TSwitchPairState::AppendDeadProbe() {
        DeadProbeCount++;
    }

    void TSwitchPairState::AppendAliveProbe(const TProbe& probe, const TInstant& now) {
        ProbeCount++;
        // if host can send probe - then source switch is alive
        SourceAlive = true;
        if (probe.GetScore() >= 0.0) {
            auto age(now - probe.GetGenerated());
            ConnectivityHistogram.Append(probe.GetScore(), GetAgeWeight(age.Seconds()));
            TargetAlive = true;
        }
        if (probe.GetRoundTripTime() > 0.0) {
            RttHistogram.Append(probe.GetRoundTripTime());
        }
    }

    void TSwitchPairState::MergeFrom(const TSwitchPairState& state) {
        ProbeCount += state.ProbeCount;
        DeadProbeCount += state.DeadProbeCount;
        ConnectivityHistogram.Merge(state.ConnectivityHistogram);
        RttHistogram.Merge(state.RttHistogram);
        SourceAlive = SourceAlive | state.SourceAlive;
        TargetAlive = TargetAlive | state.TargetAlive;
    }

    void TSwitchPairState::MergeInto(TSwitchPairIndex& pairIndex) const {
        pairIndex.GetState(GetKey()).MergeFrom(*this);
    }

    ui32 TSwitchPairState::GetWeight(const TTopologySelector& selector) const {
        return selector.GetWeight(GetSourceRef(), GetExpressionId());
    }

    flatbuffers::Offset<NAggregation::TSwitchPairState> TSwitchPairState::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        return NAggregation::CreateTSwitchPairState(
            builder,
            &Key.ToProto(),
            ProbeCount,
            DeadProbeCount,
            ConnectivityHistogram.ToProto(builder),
            RttHistogram.ToProto(builder),
            SourceAlive,
            TargetAlive
        );
    }

    void TSwitchPairState::FromProto(const NAggregation::TSwitchPairState& state)
    {
        // key should be already set
        TRealtimePairState::FromProto(state);
        ConnectivityHistogram.FromProto(*state.ConnectivityHistogram());
        RttHistogram.FromProto(*state.RttHistogram());
        SourceAlive = state.SourceAlive();
        TargetAlive = state.TargetAlive();
    }

    TSwitchPairIndex::TSwitchPairIndex(const TProbeAggregatorKey& key, const TInstant& generated)
        : TRealtimePairIndex(key, generated)
    {
    }

    TSwitchPairIndex::TSwitchPairIndex(
            const TTopologyStorage& topologyStorage,
            const NAggregation::TSwitchPairIndex& pairIndex)
        : TSwitchPairIndex(
            TProbeAggregatorKey(*pairIndex.Key()),
            TInstant::MicroSeconds(pairIndex.Generated())
        )
    {
        for (const auto& state : *pairIndex.Pairs()) {
            TSwitchPairKey key(topologyStorage, *state->PairKey());
            if (key.IsValid()) {
                auto& restoredState(GetState(key));
                restoredState.FromProto(*state);
            }
        }

        if (pairIndex.SwitchAvailability()) {
            NetworkAvailability.FromProto(topologyStorage, *pairIndex.SwitchAvailability());
        }
    }

    TSwitchPairState& TSwitchPairIndex::CreateState(const TSwitchPairKey& key) {
        TSwitchPairState::TRef newState(TSwitchPairState::Make(
            key.GetTarget(), key.GetSource(), this->Key.GetExpressionId()));
        Tree.Insert(newState.Get());
        PairCount++;
        return *newState.Release();
    }

    TSwitchPairState& TSwitchPairIndex::GetState(const TSwitchPairKey& key) {
        auto* state = Tree.Find(key);
        if (state == nullptr) {
            return CreateState(key);
        } else {
            return *state;
        }
    }

    void TSwitchPairIndex::UpdateAvailability(const TTopologySelector& selector) {
        for (auto it(GetTree().Begin()); it != GetTree().End(); ++it) {
            NetworkAvailability.AddState(*it, selector);
        }
    }

    void TSwitchPairIndex::AppendDeadProbe(const TProbe& probe) {
        auto& state(GetState(probe.GetSwitchKey()));
        state.AppendDeadProbe();
    }

    void TSwitchPairIndex::AppendAliveProbe(const TProbe& probe, const TInstant& now) {
        auto& state(GetState(probe.GetSwitchKey()));
        state.AppendAliveProbe(probe, now);
    }

    void TSwitchPairIndex::MergeFrom(const TSwitchPairIndex& pairIndex) {
        MergeIndexes(*this, pairIndex);
        NetworkAvailability.MergeFrom(pairIndex.NetworkAvailability);
        Generated = Max(Generated, pairIndex.GetGenerated());
    }

    TSwitchPairIndex::TRef TSwitchPairIndex::MakeSubset(const TLinePairKey& pairKey) const {
        auto result(TSwitchPairIndex::Make(GetKey(), GetGenerated()));
        for (TTree::TConstIterator it(GetTree().LowerBound(pairKey)); it != GetTree().End(); ++it) {
            if (pairKey.Contains(it->GetKey())) {
                it->MergeInto(*result);
            } else {
                break;
            }
        }
        return result;
    }

    flatbuffers::Offset<NAggregation::TSwitchPairIndex> TSwitchPairIndex::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        std::vector<flatbuffers::Offset<NAggregation::TSwitchPairState>> pairs;
        pairs.reserve(PairCount);
        for (auto it(GetTree().Begin()); it != GetTree().End(); ++it) {
            pairs.push_back(it->ToProto(builder));
        }
        Y_VERIFY(pairs.size() == PairCount);

        return NAggregation::CreateTSwitchPairIndex(
            builder,
            &Key.ToProto(),
            PairCount,
            builder.CreateVector(pairs),
            NetworkAvailability.ToProto(builder),
            GetGenerated().MicroSeconds()
        );
    }

    void TLinePairState::AppendSwitchState(const TSwitchPairState& state, const TTopologySelector& selector) {
        ProbeCount += state.GetProbeCount();
        DeadProbeCount += state.GetDeadProbeCount();
        auto weight = state.GetWeight(selector);
        RttHistogram.Merge(state.GetRttHistogram(), weight);
        ConnectivityHistogram.Merge(state.GetConnectivityHistogram(), weight);
    }

    void TLinePairState::MergeFrom(const TLinePairState& state) {
        ProbeCount += state.ProbeCount;
        DeadProbeCount += state.DeadProbeCount;
        RttHistogram.Merge(state.RttHistogram);
        ConnectivityHistogram.Merge(state.ConnectivityHistogram);
    }

    void TLinePairState::MergeInto(TLinePairIndex& pairIndex) const {
        pairIndex.GetState(GetKey()).MergeFrom(*this);
    }

    flatbuffers::Offset<NAggregation::TLinePairState> TLinePairState::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        return NAggregation::CreateTLinePairState(
            builder,
            &Key.ToProto(),
            ProbeCount,
            DeadProbeCount,
            ConnectivityHistogram.ToProto(builder),
            RttHistogram.ToProto(builder)
        );
    }

    void TLinePairState::FromProto(const NAggregation::TLinePairState& state)
    {
        // key should be already set
        TRealtimePairState::FromProto(state);
        ConnectivityHistogram.FromProto(*state.ConnectivityHistogram());
        RttHistogram.FromProto(*state.RttHistogram());
    }

    TLinePairIndex::TLinePairIndex(const TProbeAggregatorKey& key, const TInstant& generated)
        : TRealtimePairIndex(key, generated)
    {
    }

    TLinePairIndex::TLinePairIndex(const TTopologySelector& selector,
                                     const TSwitchPairIndex& switchPairIndex,
                                     const TProbeAggregatorKey& key)
        : TRealtimePairIndex(key, switchPairIndex.GetGenerated())
    {
        TLinePairState* queueState = nullptr;
        const auto& tree(switchPairIndex.GetTree());
        for (auto switchStateIt(tree.Begin()); switchStateIt != tree.End(); ++switchStateIt) {
            if (queueState == nullptr || !queueState->GetKey().Contains(switchStateIt->GetKey())) {
                const TLinePairKey queueKey(
                        switchStateIt->GetTargetLine(), switchStateIt->GetSourceLine());
                queueState = &GetState(queueKey);
            }
            queueState->AppendSwitchState(*switchStateIt, selector);
            NetworkAvailability.AddState(*switchStateIt, selector);
        }
    }

    TLinePairIndex::TLinePairIndex(
            const TTopologyStorage& topologyStorage,
            const NAggregation::TLinePairIndex& pairIndex)
        : TLinePairIndex(
            TProbeAggregatorKey(*pairIndex.Key()),
            TInstant::MicroSeconds(pairIndex.Generated())
        )
    {
        for (const auto& state : *pairIndex.Pairs()) {
            TLinePairKey key(topologyStorage, *state->PairKey());
            if (key.IsValid()) {
                GetState(key).FromProto(*state);
            }
        }

        if (pairIndex.SwitchAvailability()) {
            NetworkAvailability.FromProto(topologyStorage, *pairIndex.SwitchAvailability());
        }
    }

    TLinePairState& TLinePairIndex::CreateState(const TLinePairKey& key) {
        TLinePairState::TRef newState(TLinePairState::Make(
            key.GetTarget(), key.GetSource(), this->Key.GetExpressionId()
        ));
        Tree.Insert(newState.Get());
        PairCount++;
        return *newState.Release();
    }

    TLinePairState& TLinePairIndex::GetState(const TLinePairKey& key) {
        auto* state = Tree.Find(key);
        if (state == nullptr) {
            return CreateState(key);
        } else {
            return *state;
        }
    }

    void TLinePairIndex::MergeFrom(const TLinePairIndex& pairIndex) {
        MergeIndexes(*this, pairIndex);
        NetworkAvailability.MergeFrom(pairIndex.GetAvailability());
        Generated = Max(Generated, pairIndex.GetGenerated());
    }

    TLinePairIndex::TRef TLinePairIndex::MakeSubset(const TDatacenterPairKey& pairKey) const {
        auto result(TLinePairIndex::Make(GetKey(), GetGenerated()));
        for (TTree::TConstIterator it(GetTree().LowerBound(pairKey)); it != GetTree().End(); ++it) {
            if (pairKey.Contains(it->GetKey())) {
                it->MergeInto(*result);
            } else {
                break;
            }
        }
        return result;
    }

    flatbuffers::Offset<NAggregation::TLinePairIndex> TLinePairIndex::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        std::vector<flatbuffers::Offset<NAggregation::TLinePairState>> pairs;
        pairs.reserve(PairCount);
        for (auto it(GetTree().Begin()); it != GetTree().End(); ++it) {
            pairs.push_back(it->ToProto(builder));
        }
        Y_VERIFY(pairs.size() == PairCount);

        return NAggregation::CreateTLinePairIndex(
            builder,
            &Key.ToProto(),
            PairCount,
            builder.CreateVector(pairs),
            NetworkAvailability.ToProto(builder),
            GetGenerated().MicroSeconds()
        );
    }

    TLineIndexMap::TLineIndexMap(const TTopologySelector& selector, const TSwitchIndexMap& switchMap)
        : TRealtimePairIndexMap(switchMap.GetKey())
    {
        for (const auto& pair : switchMap) {
            const auto key(TProbeAggregatorKey::FromSectionKey(switchMap.GetKey(), pair.first));
            Insert(TLinePairIndex::Make(selector, *pair.second, key));
        }
    }

    void TDatacenterPairState::AppendSwitchState(const TSwitchPairState& state, const TTopologySelector& selector) {
        ProbeCount += state.GetProbeCount();
        DeadProbeCount += state.GetDeadProbeCount();
        auto weight = state.GetWeight(selector);
        RttHistogram.Merge(state.GetRttHistogram(), weight);
        ConnectivityHistogram.Merge(state.GetConnectivityHistogram(), weight);
    }

    void TDatacenterPairState::MergeFrom(const TDatacenterPairState& state) {
        ProbeCount += state.ProbeCount;
        DeadProbeCount += state.DeadProbeCount;
        RttHistogram.Merge(state.RttHistogram);
        ConnectivityHistogram.Merge(state.ConnectivityHistogram);
    }

    void TDatacenterPairState::MergeInto(TDatacenterPairIndex& pairIndex) const {
        pairIndex.GetState(GetKey()).MergeFrom(*this);
    }

    flatbuffers::Offset<NAggregation::TDatacenterPairState> TDatacenterPairState::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        return NAggregation::CreateTDatacenterPairState(
            builder,
            &Key.ToProto(),
            ProbeCount,
            DeadProbeCount,
            ConnectivityHistogram.ToProto(builder),
            RttHistogram.ToProto(builder)
        );
    }

    void TDatacenterPairState::FromProto(const NAggregation::TDatacenterPairState& state)
    {
        // key should be already set
        TRealtimePairState::FromProto(state);
        ConnectivityHistogram.FromProto(*state.ConnectivityHistogram());
        RttHistogram.FromProto(*state.RttHistogram());
    }

    TDatacenterPairIndex::TDatacenterPairIndex(const TProbeAggregatorKey& key,
                                               const TInstant& generated)
        : TRealtimePairIndex(key, generated)
    {
    }

    TDatacenterPairIndex::TDatacenterPairIndex(const TTopologySelector& selector,
                                               const TSwitchPairIndex& switchPairIndex,
                                               const TProbeAggregatorKey& key)
        : TRealtimePairIndex(key, switchPairIndex.GetGenerated())
    {
        TDatacenterPairState* datacenterState = nullptr;
        const auto& tree(switchPairIndex.GetTree());
        for (auto switchStateIt(tree.Begin()); switchStateIt != tree.End(); ++switchStateIt) {
            if (datacenterState == nullptr || !datacenterState->GetKey().Contains(switchStateIt->GetKey())) {
                const TDatacenterPairKey datacenterKey(
                    switchStateIt->GetTargetDatacenter(), switchStateIt->GetSourceDatacenter());
                datacenterState = &GetState(datacenterKey);
            }
            datacenterState->AppendSwitchState(*switchStateIt, selector);
            NetworkAvailability.AddState(*switchStateIt, selector);
        }
    }

    TDatacenterPairIndex::TDatacenterPairIndex(
            const TTopologyStorage& topologyStorage,
            const NAggregation::TDatacenterPairIndex& pairIndex)
        : TDatacenterPairIndex(
            TProbeAggregatorKey(*pairIndex.Key()),
            TInstant::MicroSeconds(pairIndex.Generated())
        )
    {
        for (const auto& state : *pairIndex.Pairs()) {
            TDatacenterPairKey key(topologyStorage, *state->PairKey());
            if (key.IsValid()) {
                GetState(key).FromProto(*state);
            }
        }

        if (pairIndex.SwitchAvailability()) {
            NetworkAvailability.FromProto(topologyStorage, *pairIndex.SwitchAvailability());
        }
    }

    TDatacenterPairState& TDatacenterPairIndex::CreateState(const TDatacenterPairKey& key) {
        TDatacenterPairState::TRef newState(TDatacenterPairState::Make(
            key.GetTarget(), key.GetSource(), this->Key.GetExpressionId()
        ));
        Tree.Insert(newState.Get());
        PairCount++;
        return *newState.Release();
    }

    TDatacenterPairState& TDatacenterPairIndex::GetState(const TDatacenterPairKey& key) {
        auto* state = Tree.Find(key);
        if (state == nullptr) {
            return CreateState(key);
        } else {
            return *state;
        }
    }

    void TDatacenterPairIndex::MergeFrom(const TDatacenterPairIndex& pairIndex) {
        MergeIndexes(*this, pairIndex);
        NetworkAvailability.MergeFrom(pairIndex.GetAvailability());
        Generated = Max(Generated, pairIndex.GetGenerated());
    }

    flatbuffers::Offset<NAggregation::TDatacenterPairIndex> TDatacenterPairIndex::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        std::vector<flatbuffers::Offset<NAggregation::TDatacenterPairState>> pairs;
        pairs.reserve(PairCount);
        for (auto it(GetTree().Begin()); it != GetTree().End(); ++it) {
            pairs.push_back(it->ToProto(builder));
        }
        Y_VERIFY(pairs.size() == PairCount);

        return NAggregation::CreateTDatacenterPairIndex(
            builder,
            &Key.ToProto(),
            PairCount,
            builder.CreateVector(pairs),
            NetworkAvailability.ToProto(builder),
            GetGenerated().MicroSeconds()
        );
    }

    TDatacenterIndexMap::TDatacenterIndexMap(const TTopologySelector& selector, const TSwitchIndexMap& switchMap)
        : TRealtimePairIndexMap(switchMap.GetKey())
    {
        for (const auto& pair : switchMap) {
            const auto key(TProbeAggregatorKey::FromSectionKey(switchMap.GetKey(), pair.first));
            Insert(TDatacenterPairIndex::Make(selector, *pair.second, key));
        }
    }
}
