#pragma once

#include <infra/netmon/state_index_base.h>
#include <infra/netmon/network_availability.h>
#include <infra/netmon/topology/topology_selector.h>
#include <infra/netmon/statistics/histograms.h>
#include <infra/netmon/idl/aggregation.fbs.h>

namespace NNetmon {
    namespace {
        template <class T, class TKeyType>
        class TRealtimePairState: public TObjectPairState<T, TKeyType> {
        public:
            inline ui64 GetProbeCount() const {
                return ProbeCount;
            }
            inline ui64 GetDeadProbeCount() const {
                return DeadProbeCount;
            }
            inline TExpressionId GetExpressionId() const noexcept {
                return ExpressionId;
            }

        protected:
            using TParent = TObjectPairState<T, TKeyType>;

            TRealtimePairState(const typename TParent::TKey::TTypeRef& target,
                               const typename TParent::TKey::TTypeRef& source,
                               TExpressionId expressionId)
                : TParent::TObjectPairState(target, source)
                , ExpressionId(expressionId)
            {
            }

            template <class TState>
            void FromProto(const TState& state) noexcept {
                ProbeCount = state.ProbeCount();
                DeadProbeCount = state.DeadProbeCount();
            }

            const TExpressionId ExpressionId;
            ui64 ProbeCount = 0;
            ui64 DeadProbeCount = 0;
        };

        template <class T, class TState>
        class TRealtimePairIndex: public TObjectPairIndex<T, TState> {
        public:
            inline const TInstant& GetGenerated() const {
                return Generated;
            }

            inline ui64 GetPairCount() const {
                return PairCount;
            }

            // total, dead
            inline std::tuple<ui64, ui64> GetProbeCount() const {
                ui64 probeCount = 0, deadProbeCount = 0;
                for (auto it(this->Tree.Begin()); it != this->Tree.End(); ++it) {
                    probeCount += it->GetProbeCount();
                    deadProbeCount += it->GetDeadProbeCount();
                }
                return {probeCount, deadProbeCount};
            }

        protected:
            TRealtimePairIndex(const TProbeAggregatorKey& key, const TInstant& generated)
                : TObjectPairIndex<T, TState>(key)
                , Generated(generated)
            {
            }

            TInstant Generated;
            ui64 PairCount = 0;
        };

        template <class T, class TIndexType>
        class TRealtimePairIndexMap: public THashMap<TExpressionId, const typename TIndexType::TRef> {
        public:
            using TRef = TAtomicSharedPtr<T>;
            using TIndex = TIndexType;

            TRealtimePairIndexMap(const TProbeSliceKey& sliceKey)
                : SectionKey(sliceKey.GetSectionKey())
            {
            }
            TRealtimePairIndexMap(const TProbeSectionKey& sectionKey)
                : SectionKey(sectionKey)
            {
            }

            void Insert(const typename TIndex::TRef index) noexcept {
                this->emplace(index->GetKey().GetExpressionId(), index);
            }

            inline const TProbeSectionKey& GetKey() const noexcept {
                return SectionKey;
            }

        private:
            const TProbeSectionKey SectionKey;
        };
    }

    class TSwitchPairIndex;

    class TSwitchPairState: public TRealtimePairState<TSwitchPairState, TSwitchPairKey> {
    public:
        friend TObjectPairState<TSwitchPairState, TSwitchPairKey>;

        using TFlat = NAggregation::TSwitchPairState;

        inline const TTopology::TLineRef GetSourceLine() const {
            return Key.GetSource().GetLine();
        }
        inline const TTopology::TLineRef GetTargetLine() const {
            return Key.GetTarget().GetLine();
        }

        inline const TTopology::TDatacenterRef GetSourceDatacenter() const {
            return Key.GetSource().GetDatacenter();
        }
        inline const TTopology::TDatacenterRef GetTargetDatacenter() const {
            return Key.GetTarget().GetDatacenter();
        }

        inline bool IsSourceAlive() const {
            return SourceAlive;
        }
        inline bool IsTargetAlive() const {
            return TargetAlive;
        }

        void AppendDeadProbe();
        void AppendAliveProbe(const TProbe& probe, const TInstant& now);

        inline const TConnectivityHistogram& GetConnectivityHistogram() const {
            return ConnectivityHistogram;
        }
        inline const TSampleHistogram& GetRttHistogram() const {
            return RttHistogram;
        }

        ui32 GetWeight(const TTopologySelector& selector) const;

        void CountHosts(const TTopologySelector& selector);

        void MergeFrom(const TSwitchPairState& state);
        void MergeInto(TSwitchPairIndex& pairIndex) const;

        flatbuffers::Offset<TFlat> ToProto(flatbuffers::FlatBufferBuilder& builder) const;
        void FromProto(const TFlat& state);

    private:
        TSwitchPairState(const TTopology::TSwitchRef& target,
                         const TTopology::TSwitchRef& source,
                         TExpressionId expressionId)
            : TRealtimePairState(target, source, expressionId)
            , SourceAlive(false)
            , TargetAlive(false)
        {
        }

        TConnectivityHistogram ConnectivityHistogram;
        TSampleHistogram RttHistogram;

        bool SourceAlive;
        bool TargetAlive;
    };

    class TSwitchPairIndex: public TRealtimePairIndex<TSwitchPairIndex, TSwitchPairState> {
    public:
        using TFlat = NAggregation::TSwitchPairIndex;

        friend TObjectPairIndex<TSwitchPairIndex, TSwitchPairState>;
        friend TSwitchPairState;

        inline TNetworkAvailability& GetAvailability() {
            return NetworkAvailability;
        }
        inline const TNetworkAvailability& GetAvailability() const {
            return NetworkAvailability;
        }

        void AppendDeadProbe(const TProbe& probe);
        void AppendAliveProbe(const TProbe& probe, const TInstant& now);

        void MergeFrom(const TSwitchPairIndex& pairIndex);
        void UpdateAvailability(const TTopologySelector& selector);

        TSwitchPairState& CreateState(const TSwitchPairKey& key);
        TSwitchPairState& GetState(const TSwitchPairKey& key);

        TSwitchPairIndex::TRef MakeSubset(const TLinePairKey& pairKey) const;

        flatbuffers::Offset<TFlat> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TSwitchPairIndex(const TProbeAggregatorKey& key, const TInstant& generated);
        TSwitchPairIndex(const TTopologyStorage& topologyStorage, const TFlat& pairIndex);

        TNetworkAvailability NetworkAvailability;
    };

    class TSwitchIndexMap: public TRealtimePairIndexMap<TSwitchIndexMap, TSwitchPairIndex> {
    public:
        using TRealtimePairIndexMap<TSwitchIndexMap, TSwitchPairIndex>::TRealtimePairIndexMap;
    };

    class TLinePairIndex;

    class TLinePairState: public TRealtimePairState<TLinePairState, TLinePairKey> {
    public:
        friend TObjectPairState<TLinePairState, TLinePairKey>;

        using TFlat = NAggregation::TLinePairState;

        void AppendSwitchState(const TSwitchPairState& state, const TTopologySelector& selector);

        void CountHosts(const TTopologySelector& selector);

        void MergeFrom(const TLinePairState& state);
        void MergeInto(TLinePairIndex& pairIndex) const;

#ifdef NOC_SLA_BUILD
        using TConnectivityHistogramType = TConnectivityHistogram;
#else
        using TConnectivityHistogramType = TAverageHistogram;
#endif

        inline const TConnectivityHistogramType& GetConnectivityHistogram() const {
            return ConnectivityHistogram;
        }
        inline const TSampleHistogram& GetRttHistogram() const {
            return RttHistogram;
        }

        flatbuffers::Offset<TFlat> ToProto(flatbuffers::FlatBufferBuilder& builder) const;
        void FromProto(const TFlat& state);

    private:
        TLinePairState(const TTopology::TLineRef& target,
                       const TTopology::TLineRef& source,
                       TExpressionId expressionId)
            : TRealtimePairState(target, source, expressionId)
        {
        }

        TConnectivityHistogramType ConnectivityHistogram;
        TSampleHistogram RttHistogram;
    };

    class TLinePairIndex: public TRealtimePairIndex<TLinePairIndex, TLinePairState> {
    public:
        using TFlat = NAggregation::TLinePairIndex;

        friend TObjectPairIndex<TLinePairIndex, TLinePairState>;
        friend TLinePairState;

        inline const TNetworkAvailability& GetAvailability() const {
            return NetworkAvailability;
        }

        void MergeFrom(const TLinePairIndex& pairIndex);

        TLinePairState& CreateState(const TLinePairKey& key);
        TLinePairState& GetState(const TLinePairKey& key);

        TLinePairIndex::TRef MakeSubset(const TDatacenterPairKey& pairKey) const;

        flatbuffers::Offset<TFlat> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TLinePairIndex(const TProbeAggregatorKey& key,
                       const TInstant& generated);
        TLinePairIndex(const TTopologySelector& selector,
                       const TSwitchPairIndex& switchPairIndex,
                       const TProbeAggregatorKey& key);
        TLinePairIndex(const TTopologyStorage& topologyStorage,
                       const TFlat& pairIndex);

        TNetworkAvailability NetworkAvailability;
    };

    class TLineIndexMap: public TRealtimePairIndexMap<TLineIndexMap, TLinePairIndex> {
    public:
        using TRealtimePairIndexMap<TLineIndexMap, TLinePairIndex>::TRealtimePairIndexMap;

        TLineIndexMap(const TTopologySelector& selector,
                      const TSwitchIndexMap& switchMap);
    };

    class TDatacenterPairIndex;

    class TDatacenterPairState: public TRealtimePairState<TDatacenterPairState, TDatacenterPairKey> {
    public:
        friend TObjectPairState<TDatacenterPairState, TDatacenterPairKey>;

        using TFlat = NAggregation::TDatacenterPairState;

        void AppendSwitchState(const TSwitchPairState& state, const TTopologySelector& selector);

        void CountHosts(const TTopologySelector& selector);

        void MergeFrom(const TDatacenterPairState& state);
        void MergeInto(TDatacenterPairIndex& pairIndex) const;

#ifdef NOC_SLA_BUILD
        using TConnectivityHistogramType = TConnectivityHistogram;
#else
        using TConnectivityHistogramType = TAverageHistogram;
#endif

        inline const TConnectivityHistogramType& GetConnectivityHistogram() const {
            return ConnectivityHistogram;
        }
        inline const TSampleHistogram& GetRttHistogram() const {
            return RttHistogram;
        }

        flatbuffers::Offset<TFlat> ToProto(flatbuffers::FlatBufferBuilder& builder) const;
        void FromProto(const TFlat& key);

    private:
        TDatacenterPairState(const TTopology::TDatacenterRef& target,
                             const TTopology::TDatacenterRef& source,
                             TExpressionId expressionId)
            : TRealtimePairState(target, source, expressionId)
        {
        }

        TConnectivityHistogramType ConnectivityHistogram;
        TSampleHistogram RttHistogram;
    };

    class TDatacenterPairIndex: public TRealtimePairIndex<TDatacenterPairIndex, TDatacenterPairState> {
    public:
        using TFlat = NAggregation::TDatacenterPairIndex;

        friend TObjectPairIndex<TDatacenterPairIndex, TDatacenterPairState>;
        friend TDatacenterPairState;

        inline const TNetworkAvailability& GetAvailability() const {
            return NetworkAvailability;
        }

        void MergeFrom(const TDatacenterPairIndex& pairIndex);

        TDatacenterPairState& CreateState(const TDatacenterPairKey& key);
        TDatacenterPairState& GetState(const TDatacenterPairKey& key);

        flatbuffers::Offset<TFlat> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TDatacenterPairIndex(const TProbeAggregatorKey& key,
                             const TInstant& generated);
        TDatacenterPairIndex(const TTopologySelector& selector,
                             const TSwitchPairIndex& switchPairIndex,
                             const TProbeAggregatorKey& key);
        TDatacenterPairIndex(const TTopologyStorage& topologyStorage,
                             const TFlat& pairIndex);

        TNetworkAvailability NetworkAvailability;
    };

    class TDatacenterIndexMap: public TRealtimePairIndexMap<TDatacenterIndexMap, TDatacenterPairIndex> {
    public:
        using TRealtimePairIndexMap<TDatacenterIndexMap, TDatacenterPairIndex>::TRealtimePairIndexMap;

        TDatacenterIndexMap(const TTopologySelector& selector,
                            const TSwitchIndexMap& switchMap);
    };
}
