#pragma once

#include <infra/netmon/state_index.h>
#include <infra/netmon/statistics/series.h>
#include <infra/netmon/idl/history.fbs.h>

namespace NNetmon {
    namespace {
        template <class T, class TKeyType, class TRealtimeIndexType, class TConnectivitySeriesType, class TFlatType>
        class THistoryPairState: public TObjectPairState<T, TKeyType> {
        public:
            using TConnectivitySeries = TConnectivitySeriesType;
            using TRealtimeIndex = TRealtimeIndexType;
            using TRealtimeState = typename TRealtimeIndex::TState;
            using TFlat = TFlatType;

            inline TDuration GetInterval() const {
                return TimestampSeries.GetInterval();
            }

            inline TConnectivitySeriesType GetConnectivitySeries() const {
                TGuard<TAdaptiveLock> guard(UpdateLock);
                return ConnectivitySeries;
            }
            inline TSampleHistogramSeries GetRttSeries() const {
                TGuard<TAdaptiveLock> guard(UpdateLock);
                return RttSeries;
            }
            inline TTimestampSeries GetTimestampSeries() const {
                TGuard<TAdaptiveLock> guard(UpdateLock);
                return TimestampSeries;
            }

            // not thread safe methods
            inline TConnectivitySeriesType& MutableConnectivitySeries() {
                return ConnectivitySeries;
            }
            inline TSampleHistogramSeries& MutableRttSeries() {
                return RttSeries;
            }
            inline TTimestampSeries& MutableTimestampSeries() {
                return TimestampSeries;
            }

            void Append(const TRealtimeState& other, const TInstant& generated) {
                TGuard<TAdaptiveLock> guard(UpdateLock);
                ConnectivitySeries.Append(other.GetConnectivityHistogram());
                RttSeries.Append(other.GetRttHistogram());
                TimestampSeries.Append(RoundInstant(generated, TimestampSeries.GetInterval()));
            }

        protected:
            using TParent = TObjectPairState<T, TKeyType>;

            THistoryPairState(const typename TParent::TKey::TTypeRef& target,
                              const typename TParent::TKey::TTypeRef& source,
                              const TDuration& interval)
                : TParent(target, source)
                , TimestampSeries(interval)
            {
            }

            THistoryPairState(const typename TParent::TKey::TTypeRef& target,
                              const typename TParent::TKey::TTypeRef& source,
                              const TFlatType& state)
                : TParent(target, source)
                , ConnectivitySeries(*state.ConnectivitySeries())
                , RttSeries(*state.RttSeries())
                , TimestampSeries(*state.TimestampSeries())
            {
            }

            THistoryPairState(const T& other)
                : TParent(other.GetTargetRef(), other.GetSourceRef())
                , ConnectivitySeries(other.GetConnectivitySeries())
                , RttSeries(other.GetRttSeries())
                , TimestampSeries(other.GetTimestampSeries())
            {
            }

            mutable TAdaptiveLock UpdateLock;
            mutable TConnectivitySeriesType ConnectivitySeries;
            mutable TSampleHistogramSeries RttSeries;
            mutable TTimestampSeries TimestampSeries;
        };

        template <class T, class TState>
        class THistoryPairIndex: public TObjectPairIndex<T, TState> {
        public:
            using TRealtimeIndex = typename TState::TRealtimeIndex;

            void Append(const TRealtimeIndex& other, bool checkTime=true) {
                TGuard<TAdaptiveLock> guard(UpdateLock);
                if (checkTime && other.GetGenerated() <= LastTimestamp) {
                    return;
                }
                for (auto& pair : TMergeIterator<THistoryPairIndex, TRealtimeIndex>(*this, other)) {
                    auto* left = pair.first;
                    if (left == nullptr) {
                        typename TState::TRef state(TState::Make(
                            pair.second->GetTargetRef(), pair.second->GetSourceRef()
                        ));
                        this->Tree.Insert(state.Get());
                        left = state.Release();
                    }
                    if (pair.second != nullptr) {
                        auto unguard(Unguard(guard));
                        left->Append(*pair.second, other.GetGenerated());
                    }
                }
                LastTimestamp = other.GetGenerated();
            }

            typename TState::TRef Find(const typename TState::TKey& key) const {
                TGuard<TAdaptiveLock> guard(UpdateLock);
                const auto* state(this->Tree.Find(key));
                guard.Release();
                if (state) {
                    return TState::Make(*state);
                } else {
                    return nullptr;
                }
            }

        protected:
            THistoryPairIndex(const TProbeAggregatorKey& key)
                : TObjectPairIndex<T, TState>(key)
            {
            }

            mutable TAdaptiveLock UpdateLock;
            TInstant LastTimestamp;
        };

        template <class T, class TIndexType>
        class TIndexHistoryMap: public THashMap<TProbeAggregatorKey, const typename TIndexType::TRef> {
        public:
            using TRef = TAtomicSharedPtr<T>;
            using TIndex = TIndexType;
            using TRealtimeIndex = typename TIndex::TRealtimeIndex;
            using TState = typename TIndex::TState;

            TIndexHistoryMap(const TInstant& generated)
                : Generated(generated)
            {
            }

            inline TInstant GetGenerated() const {
                return Generated;
            }

            void Append(const TRealtimeIndex& other, bool checkTime=true) {
                TGuard<TAdaptiveLock> guard(UpdateLock);
                typename TIndex::TRef index;
                auto it(this->find(other.GetKey()));
                if (it != this->end()) {
                    index.Reset(it->second);
                } else {
                    index = TIndex::Make(other.GetKey());
                    this->emplace(index->GetKey(), index);
                }
                guard.Release();
                index->Append(other, checkTime);
            }

            typename TIndex::TRef Find(const TProbeAggregatorKey& key) const {
                TGuard<TAdaptiveLock> guard(UpdateLock);
                const auto it(this->find(key));
                if (it != this->end()) {
                    return it->second;
                } else {
                    return nullptr;
                }
            }

            typename TState::TRef FindState(const TProbeAggregatorKey& key, const typename TState::TKey& pairKey) const {
                const auto index(Find(key));
                if (index) {
                    return index->Find(pairKey);
                } else {
                    return nullptr;
                }
            }

        private:
            const TInstant Generated;
            mutable TAdaptiveLock UpdateLock;
        };
    }

    class TSwitchPairStateHistory: public THistoryPairState<TSwitchPairStateHistory, TSwitchPairKey, TSwitchPairIndex,
                                                            TConnectivityHistogramSeries, NHistory::TSwitchPairStateHistory> {
    public:
        friend TObjectPairState<TSwitchPairStateHistory, TSwitchPairKey>;

        flatbuffers::Offset<NHistory::TSwitchPairStateHistory> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TSwitchPairStateHistory(const TTopology::TSwitchRef& target,
                                const TTopology::TSwitchRef& source);
        TSwitchPairStateHistory(const TTopology::TSwitchRef& target,
                                const TTopology::TSwitchRef& source,
                                const NHistory::TSwitchPairStateHistory& state);
        TSwitchPairStateHistory(const TSwitchPairStateHistory& other);
    };

    class TSwitchPairIndexHistory: public THistoryPairIndex<TSwitchPairIndexHistory, TSwitchPairStateHistory> {
    public:
        friend TObjectPairIndex<TSwitchPairIndexHistory, TSwitchPairStateHistory>;

    private:
        TSwitchPairIndexHistory(const TProbeAggregatorKey& key);
    };

    class TSwitchIndexHistoryMap: public TIndexHistoryMap<TSwitchIndexHistoryMap, TSwitchPairIndexHistory> {
    public:
        using TIndexHistoryMap<TSwitchIndexHistoryMap, TSwitchPairIndexHistory>::TIndexHistoryMap;
    };

    class TLinePairStateHistory: public THistoryPairState<TLinePairStateHistory, TLinePairKey, TLinePairIndex,
                                                          TLinePairState::TConnectivityHistogramType::TSeries,
                                                          NHistory::TLinePairStateHistory> {
    public:
        friend TObjectPairState<TLinePairStateHistory, TLinePairKey>;

        flatbuffers::Offset<NHistory::TLinePairStateHistory> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TLinePairStateHistory(const TTopology::TLineRef& target,
                              const TTopology::TLineRef& source);
        TLinePairStateHistory(const TTopology::TLineRef& target,
                              const TTopology::TLineRef& source,
                              const NHistory::TLinePairStateHistory& state);
        TLinePairStateHistory(const TLinePairStateHistory& other);
    };

    class TLinePairIndexHistory: public THistoryPairIndex<TLinePairIndexHistory, TLinePairStateHistory> {
    public:
        friend TObjectPairIndex<TLinePairIndexHistory, TLinePairStateHistory>;

    private:
        TLinePairIndexHistory(const TProbeAggregatorKey& key);
    };

    class TLineIndexHistoryMap: public TIndexHistoryMap<TLineIndexHistoryMap, TLinePairIndexHistory> {
    public:
        using TIndexHistoryMap<TLineIndexHistoryMap, TLinePairIndexHistory>::TIndexHistoryMap;
    };

    class TDatacenterPairStateHistory: public THistoryPairState<TDatacenterPairStateHistory, TDatacenterPairKey, TDatacenterPairIndex,
                                                                TDatacenterPairState::TConnectivityHistogramType::TSeries,
                                                                NHistory::TDatacenterPairStateHistory> {
    public:
        friend TObjectPairState<TDatacenterPairStateHistory, TDatacenterPairKey>;

        flatbuffers::Offset<NHistory::TDatacenterPairStateHistory> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TDatacenterPairStateHistory(const TTopology::TDatacenterRef& target,
                                    const TTopology::TDatacenterRef& source);
        TDatacenterPairStateHistory(const TTopology::TDatacenterRef& target,
                                    const TTopology::TDatacenterRef& source,
                                    const NHistory::TDatacenterPairStateHistory& state);
        TDatacenterPairStateHistory(const TDatacenterPairStateHistory& other);
    };

    class TDatacenterPairIndexHistory: public THistoryPairIndex<TDatacenterPairIndexHistory, TDatacenterPairStateHistory> {
    public:
        friend TObjectPairIndex<TDatacenterPairIndexHistory, TDatacenterPairStateHistory>;

    private:
        TDatacenterPairIndexHistory(const TProbeAggregatorKey& key);
    };

    class TDatacenterIndexHistoryMap: public TIndexHistoryMap<TDatacenterIndexHistoryMap, TDatacenterPairIndexHistory> {
    public:
        using TIndexHistoryMap<TDatacenterIndexHistoryMap, TDatacenterPairIndexHistory>::TIndexHistoryMap;
    };

    inline TSwitchPairStateHistory::TRef CreateEmptyStateHistory(const TTopology::TSwitchRef& target, const TTopology::TSwitchRef& source) {
        return TSwitchPairStateHistory::Make(target, source);
    }

    inline TLinePairStateHistory::TRef CreateEmptyStateHistory(const TTopology::TLineRef& target, const TTopology::TLineRef& source) {
        return TLinePairStateHistory::Make(target, source);
    }

    inline TDatacenterPairStateHistory::TRef CreateEmptyStateHistory(const TTopology::TDatacenterRef& target, const TTopology::TDatacenterRef& source) {
        return TDatacenterPairStateHistory::Make(target, source);
    }
}
