#include <infra/netmon/state_index_history.h>
#include <infra/netmon/common_ut.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NNetmon;

class TStateIndexHistoryTest: public TTestBase {
    UNIT_TEST_SUITE(TStateIndexHistoryTest);
    UNIT_TEST(TestAppendAndFind)
    UNIT_TEST(TestSerialization)
    UNIT_TEST_SUITE_END();

public:
    TStateIndexHistoryTest()
        : Key(
            GetTopologyStorage().DefaultExpressionId(),
            BACKBONE6,
            UDP
        )
        , Timestamp(TInstant::Now())
    {
    }

private:
    inline TSwitchPairIndex::TRef CreateSwitchIndex(const TProbe& probe) {
        const auto index(TSwitchPairIndex::Make(Key, Timestamp));
        index->AppendDeadProbe(probe);
        index->AppendAliveProbe(probe, Timestamp);
        return index;
    }

    inline TLinePairIndex::TRef CreateLineIndex(const TSwitchPairIndex& index) {
        return TLinePairIndex::Make(*GetTopologyStorage().GetTopologySelector(), index, Key);
    }

    inline TDatacenterPairIndex::TRef CreateDatacenterIndex(const TSwitchPairIndex& index) {
        return TDatacenterPairIndex::Make(*GetTopologyStorage().GetTopologySelector(), index, Key);
    }

    template <class TMap, class TKey>
    inline auto FindState(const TMap& indexMap, const TKey& networkKey) {
        const auto indexIt(indexMap.find(Key));
        UNIT_ASSERT(!indexIt.IsEnd());
        auto foundState(indexIt->second->Find(networkKey));
        UNIT_ASSERT(foundState);
        return foundState;
    }

    inline TLinePairKey CreateLineKey(const TSwitchPairKey& switchKey) {
        return {switchKey.GetTarget().GetLine(), switchKey.GetSource().GetLine()};
    }

    inline TDatacenterPairKey CreateDatacenterKey(const TSwitchPairKey& switchKey) {
        return {switchKey.GetTarget().GetDatacenter(), switchKey.GetSource().GetDatacenter()};
    }

    inline void AssertHasState(const TSwitchIndexHistoryMap& switchMap, const TSwitchPairKey& switchKey) {
        UNIT_ASSERT_EQUAL(FindState(switchMap, switchKey)->GetKey(), switchKey);
    }

    inline void AssertHasState(const TLineIndexHistoryMap& lineMap, const TLinePairKey& lineKey) {
        UNIT_ASSERT_EQUAL(FindState(lineMap, lineKey)->GetKey(), lineKey);
    }

    inline void AssertHasState(const TDatacenterIndexHistoryMap& datacenterMap, const TDatacenterPairKey& datacenterKey) {
        UNIT_ASSERT_EQUAL(FindState(datacenterMap, datacenterKey)->GetKey(), datacenterKey);
    }

    inline void TestAppendAndFind() {
        const auto probe(CreateProbe("jmon-test.search.yandex.net", "man1-0015.search.yandex.net"));
        const auto switchIndex(CreateSwitchIndex(*probe));
        const auto lineIndex(CreateLineIndex(*switchIndex));
        const auto datacenterIndex(CreateDatacenterIndex(*switchIndex));

        TSwitchIndexHistoryMap switchMap(TInstant::Now());
        switchMap.Append(*switchIndex);
        AssertHasState(switchMap, probe->GetSwitchKey());

        TLineIndexHistoryMap lineMap(TInstant::Now());
        lineMap.Append(*lineIndex);
        AssertHasState(lineMap, CreateLineKey(probe->GetSwitchKey()));

        TDatacenterIndexHistoryMap datacenterMap(TInstant::Now());
        datacenterMap.Append(*datacenterIndex);
        AssertHasState(datacenterMap, CreateDatacenterKey(probe->GetSwitchKey()));
    }

    template <class TState>
    inline void AssertSeriesNotEmpty(const TState& state) {
        const auto connectivity(state.GetConnectivitySeries());
        UNIT_ASSERT(connectivity.begin() != connectivity.end());

        const auto rtt(state.GetRttSeries());
        UNIT_ASSERT(rtt.begin() != rtt.end());

        const auto timestamps(state.GetTimestampSeries());
        UNIT_ASSERT(timestamps.begin() != timestamps.end());
    }


    template <class TState>
    inline void AssertThatStateIsSerializable(TState& state) {
        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(state.ToProto(builder));
        const auto restoredState(TState::Make(
            state.GetTargetRef(),
            state.GetSourceRef(),
            *flatbuffers::GetRoot<typename TState::TFlat>(builder.GetBufferPointer())
        ));
        AssertSeriesNotEmpty(*restoredState);
    }

    inline void TestSerialization() {
        const auto probe(CreateProbe("jmon-test.search.yandex.net", "man1-0015.search.yandex.net"));

        const auto sourceSwitch(probe->GetSourceIface().GetSwitch());
        const auto targetSwitch(probe->GetTargetIface().GetSwitch());

        const auto switchIndex(CreateSwitchIndex(*probe));
        auto switchState(TSwitchPairStateHistory::Make(targetSwitch, sourceSwitch));
        switchState->Append(switchIndex->GetState(probe->GetSwitchKey()), Timestamp);
        AssertThatStateIsSerializable(*switchState);

        const auto lineIndex(CreateLineIndex(*switchIndex));
        auto lineState(TLinePairStateHistory::Make(targetSwitch.GetLine(), sourceSwitch.GetLine()));
        lineState->Append(lineIndex->GetState(CreateLineKey(probe->GetSwitchKey())), Timestamp);
        AssertThatStateIsSerializable(*lineState);

        const auto datacenterIndex(CreateDatacenterIndex(*switchIndex));
        auto datacenterState(TDatacenterPairStateHistory::Make(targetSwitch.GetDatacenter(), sourceSwitch.GetDatacenter()));
        datacenterState->Append(datacenterIndex->GetState(CreateDatacenterKey(probe->GetSwitchKey())), Timestamp);
        AssertThatStateIsSerializable(*datacenterState);
    }

    const TProbeAggregatorKey Key;
    const TInstant Timestamp;
};

UNIT_TEST_SUITE_REGISTRATION(TStateIndexHistoryTest);
