#include <infra/netmon/state_index.h>
#include <infra/netmon/common_ut.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NNetmon;

class TStateIndexTest: public TTestBase {
    UNIT_TEST_SUITE(TStateIndexTest);
    UNIT_TEST(TestSwitchSerialization)
    UNIT_TEST(TestSwitchMerging)
    UNIT_TEST(TestSwitchAvailability)
    UNIT_TEST(TestQueueSerialization)
    UNIT_TEST(TestQueueMerging)
    UNIT_TEST(TestDcSerialization)
    UNIT_TEST(TestDcMerging)
    UNIT_TEST_SUITE_END();

public:
    TStateIndexTest()
        : SectionKey{NIL_NETWORK, NIL_PROTOCOL}
        , TopologyStorage(TGlobalTopology::GetTopologyStorage())
        , TopologySelector(TopologyStorage.GetTopologySelector())
        , AggregatorKey(TProbeAggregatorKey::FromSectionKey(SectionKey, TopologyStorage.DefaultExpressionId()))
        , Now(TInstant::Now())
    {
    }

private:
    TProbe::TRef CreateFirstProbe() {
        return CreateProbe("man1-0015.search.yandex.net", "sas1-1234.search.yandex.net");
    }

    TProbe::TRef CreateSecondProbe() {
        return CreateProbe("jmon-test.search.yandex.net", "man1-0015.search.yandex.net");
    }

    inline void TestSwitchSerialization() {
        const auto probe(CreateFirstProbe());

        const auto originalIndex(TSwitchPairIndex::Make(AggregatorKey, TInstant::Now()));
        originalIndex->AppendDeadProbe(*probe);
        originalIndex->AppendAliveProbe(*probe, Now);
        originalIndex->UpdateAvailability(*TopologySelector);

        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(originalIndex->ToProto(builder));
        const NAggregation::TSwitchPairIndex& serializedIndex(
            *flatbuffers::GetRoot<NAggregation::TSwitchPairIndex>(builder.GetBufferPointer()));
        const auto restoredIndex(TSwitchPairIndex::Make(TopologyStorage, serializedIndex));

        UNIT_ASSERT_EQUAL(restoredIndex->GetKey(), originalIndex->GetKey());
        UNIT_ASSERT_EQUAL(restoredIndex->GetPairCount(), originalIndex->GetPairCount());
        UNIT_ASSERT_EQUAL(restoredIndex->GetProbeCount(), originalIndex->GetProbeCount());
        UNIT_ASSERT_EQUAL(originalIndex->GetAvailability().GetCount(),
                          restoredIndex->GetAvailability().GetCount());
    }

    TSwitchPairIndex::TRef CreateSwitchIndex(TProbe::TRef probe) {
        const auto index(TSwitchPairIndex::Make(AggregatorKey, TInstant::Now()));
        index->AppendAliveProbe(*probe, Now);
        index->UpdateAvailability(*TopologySelector);
        return index;
    }

    inline void TestSwitchMerging() {
        const auto probe(CreateFirstProbe());

        const auto firstIndex(CreateSwitchIndex(probe));
        const auto secondIndex(CreateSwitchIndex(probe));

        const auto resultingIndex(TSwitchPairIndex::Make(AggregatorKey, TInstant::Now()));
        resultingIndex->MergeFrom(*firstIndex);
        resultingIndex->MergeFrom(*secondIndex);

        auto probeCount = resultingIndex->GetProbeCount();
        UNIT_ASSERT_EQUAL(std::get<0>(probeCount), 2UL);
        UNIT_ASSERT_EQUAL(std::get<1>(probeCount), 0UL);

        UNIT_ASSERT_EQUAL(resultingIndex->GetPairCount(), 1UL);
        UNIT_ASSERT_EQUAL(resultingIndex->GetAvailability().GetCount(), 0UL);
    }

    inline void TestSwitchAvailability() {
        const auto probe(CreateProbe("sas1-1234.search.yandex.net", "sas1-2345.search.yandex.net"));

        const auto index(CreateSwitchIndex(probe));

        const auto resultingIndex(TSwitchPairIndex::Make(AggregatorKey, TInstant::Now()));
        resultingIndex->MergeFrom(*index);

        UNIT_ASSERT_EQUAL(resultingIndex->GetAvailability().GetCount(), 2UL);
    }

    inline void TestQueueSerialization() {
        const auto switchIndex(CreateSwitchIndex(CreateFirstProbe()));
        const auto originalIndex(TLinePairIndex::Make(*TopologySelector, *switchIndex, AggregatorKey));

        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(originalIndex->ToProto(builder));
        const NAggregation::TLinePairIndex& serializedIndex(
            *flatbuffers::GetRoot<NAggregation::TLinePairIndex>(builder.GetBufferPointer()));
        const auto restoredIndex(TLinePairIndex::Make(TopologyStorage, serializedIndex));

        UNIT_ASSERT_EQUAL(originalIndex->GetKey(), restoredIndex->GetKey());
        UNIT_ASSERT_EQUAL(originalIndex->GetPairCount(), restoredIndex->GetPairCount());
        UNIT_ASSERT_EQUAL(originalIndex->GetProbeCount(), restoredIndex->GetProbeCount());
        UNIT_ASSERT_EQUAL(originalIndex->GetAvailability().GetCount(),
                          restoredIndex->GetAvailability().GetCount());
    }

    inline void TestQueueMerging() {
        const auto firstSwitchIndex(CreateSwitchIndex(CreateFirstProbe()));
        const auto secondSwitchIndex(CreateSwitchIndex(CreateSecondProbe()));

        const auto firstQueueIndex(TLinePairIndex::Make(
                *TopologySelector, *firstSwitchIndex, AggregatorKey));
        const auto secondQueueIndex(TLinePairIndex::Make(
                *TopologySelector, *secondSwitchIndex, AggregatorKey));

        const auto resultingIndex(TLinePairIndex::Make(AggregatorKey, TInstant::Now()));
        resultingIndex->MergeFrom(*firstQueueIndex);
        resultingIndex->MergeFrom(*secondQueueIndex);

        auto probeCount = resultingIndex->GetProbeCount();
        UNIT_ASSERT_EQUAL(std::get<0>(probeCount), 2UL);
        UNIT_ASSERT_EQUAL(std::get<1>(probeCount), 0UL);

        UNIT_ASSERT_EQUAL(resultingIndex->GetPairCount(), 2UL);
        UNIT_ASSERT_EQUAL(resultingIndex->GetAvailability().GetCount(), 0UL);
    }

    inline void TestDcSerialization() {
        const auto switchIndex(CreateSwitchIndex(CreateFirstProbe()));
        const auto originalIndex(TDatacenterPairIndex::Make(
                *TopologySelector, *switchIndex, AggregatorKey));

        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(originalIndex->ToProto(builder));
        const NAggregation::TDatacenterPairIndex& serializedIndex(
            *flatbuffers::GetRoot<NAggregation::TDatacenterPairIndex>(builder.GetBufferPointer()));
        const auto restoredIndex(TDatacenterPairIndex::Make(
                TopologyStorage, serializedIndex));

        UNIT_ASSERT_EQUAL(originalIndex->GetKey(), restoredIndex->GetKey());
        UNIT_ASSERT_EQUAL(originalIndex->GetPairCount(), restoredIndex->GetPairCount());
        UNIT_ASSERT_EQUAL(originalIndex->GetProbeCount(), restoredIndex->GetProbeCount());
        UNIT_ASSERT_EQUAL(originalIndex->GetAvailability().GetCount(),
                          restoredIndex->GetAvailability().GetCount());
    }

    inline void TestDcMerging() {
        const auto firstSwitchIndex(CreateSwitchIndex(CreateFirstProbe()));
        const auto secondSwitchIndex(CreateSwitchIndex(CreateSecondProbe()));

        const auto firstDcIndex(TDatacenterPairIndex::Make(
                *TopologySelector, *firstSwitchIndex, AggregatorKey));
        const auto secondDcIndex(TDatacenterPairIndex::Make(
                *TopologySelector, *secondSwitchIndex, AggregatorKey));

        const auto resultingIndex(TDatacenterPairIndex::Make(AggregatorKey, TInstant::Now()));
        resultingIndex->MergeFrom(*firstDcIndex);
        resultingIndex->MergeFrom(*secondDcIndex);

        auto probeCount = resultingIndex->GetProbeCount();
        UNIT_ASSERT_EQUAL(std::get<0>(probeCount), 2UL);
        UNIT_ASSERT_EQUAL(std::get<1>(probeCount), 0UL);

        UNIT_ASSERT_EQUAL(resultingIndex->GetPairCount(), 2UL);
        UNIT_ASSERT_EQUAL(resultingIndex->GetAvailability().GetCount(), 0UL);
    }

    TProbeSectionKey SectionKey;

    TTopologyStorage& TopologyStorage;
    TTopologySelector::TBox::TConstValueRef TopologySelector;

    TProbeAggregatorKey AggregatorKey;

    const TInstant Now;
    const TIpAddress EmptyAddress;
};

UNIT_TEST_SUITE_REGISTRATION(TStateIndexTest);
