#include <infra/netmon/state_keys.h>
#include <infra/netmon/probe.h>

namespace NNetmon {
    TSwitchPairKey::TSwitchPairKey(const TProbe& probe)
        : TObjectPairKey(probe.GetTargetIface().GetSwitch(), probe.GetSourceIface().GetSwitch())
    {
    }

    TSwitchPairKey::TSwitchPairKey(const TTopologyStorage& topologyStorage,
                                   const NCommon::TSwitchPairKey& key)
        : TObjectPairKey(
            topologyStorage.FindSwitch(key.Target()),
            topologyStorage.FindSwitch(key.Source()))
    {
    }

    bool TSwitchPairKey::operator<(const TLinePairKey& rhs) const {
        return IsLess(
            GetTarget()->GetLine().GetComparator(),
            rhs.GetTarget()->GetComparator(),
            GetSource()->GetLine().GetComparator(),
            rhs.GetSource()->GetComparator()
        );
    }

    bool TSwitchPairKey::operator<(const TDatacenterPairKey& rhs) const {
        return IsLess(
            GetTarget()->GetDatacenter(),
            *rhs.GetTarget(),
            GetSource()->GetDatacenter(),
            *rhs.GetSource()
        );
    }

    bool TSwitchPairKey::Contains(const THost& targetHost, const THost& sourceHost) const {
        return GetTarget() == targetHost.GetSwitch() && GetSource() == sourceHost.GetSwitch();
    }

    bool TSwitchPairKey::Contains(const THostInterface& targetIface, const THostInterface& sourceIface) const {
        return GetTarget() == targetIface.GetSwitch() && GetSource() == sourceIface.GetSwitch();
    }

    bool TSwitchPairKey::Contains(const TProbe& probe) const {
        return Contains(*probe.GetTargetIface(), *probe.GetSourceIface());
    }

    bool TSwitchPairKey::SameTarget(const THost& targetHost) const {
        return GetTarget() == targetHost.GetSwitch();
    }

    bool TSwitchPairKey::SameTarget(const THostInterface& targetIface) const {
        return GetTarget() == targetIface.GetSwitch();
    }

    TLinePairKey TSwitchPairKey::GetParent() const {
        return {GetTarget().GetLine(), GetSource().GetLine()};
    }

    TLinePairKey::TLinePairKey(const TTopologyStorage& topologyStorage,
                               const NCommon::TLinePairKey& key)
        : TObjectPairKey(
            topologyStorage.FindQueue(key.Target()),
            topologyStorage.FindQueue(key.Source()))
    {
    }

    bool TLinePairKey::operator<(const TSwitchPairKey& rhs) const {
        return IsLess(
            GetTarget()->GetComparator(),
            rhs.GetTarget()->GetLine().GetComparator(),
            GetSource()->GetComparator(),
            rhs.GetSource()->GetLine().GetComparator()
        );
    }

    bool TLinePairKey::operator<(const TDatacenterPairKey& rhs) const {
        return IsLess(
            GetTarget()->GetDatacenter(),
            *rhs.GetTarget(),
            GetSource()->GetDatacenter(),
            *rhs.GetSource()
        );
    }

    bool TLinePairKey::Contains(const TSwitchPairKey& switchPairKey) const {
        return (
            GetTarget() == switchPairKey.GetTarget()->GetLine()
            && GetSource() == switchPairKey.GetSource()->GetLine());
    }

    bool TLinePairKey::Contains(const THost& targetHost, const THost& sourceHost) const {
        return GetTarget() == targetHost.GetLine() && GetSource() == sourceHost.GetLine();
    }

    bool TLinePairKey::Contains(const THostInterface& targetIface, const THostInterface& sourceIface) const {
        return GetTarget() == targetIface.GetLine() && GetSource() == sourceIface.GetLine();
    }

    bool TLinePairKey::Contains(const TProbe& probe) const {
        return Contains(*probe.GetTargetIface(), *probe.GetSourceIface());
    }

    bool TLinePairKey::SameTarget(const TSwitchPairKey& switchPairKey) const {
        return GetTarget() == switchPairKey.GetTarget()->GetLine();
    }

    bool TLinePairKey::SameTarget(const THost& targetHost) const {
        return GetTarget() == targetHost.GetLine();
    }

    bool TLinePairKey::SameTarget(const THostInterface& targetIface) const {
        return GetTarget() == targetIface.GetLine();
    }

    TDatacenterPairKey TLinePairKey::GetParent() const {
        return {GetTarget().GetDatacenter(), GetSource().GetDatacenter()};
    }

    TDatacenterPairKey::TDatacenterPairKey(const TTopologyStorage& topologyStorage,
                                           const NCommon::TDatacenterPairKey& key)
        : TObjectPairKey(
            topologyStorage.FindDatacenter(key.Target()),
            topologyStorage.FindDatacenter(key.Source()))
    {
    }

    bool TDatacenterPairKey::operator<(const TSwitchPairKey& rhs) const {
        return IsLess(
            *GetTarget(),
            rhs.GetTarget()->GetDatacenter(),
            *GetSource(),
            rhs.GetSource()->GetDatacenter()
        );
    }

    bool TDatacenterPairKey::operator<(const TLinePairKey& rhs) const {
        return IsLess(
            *GetTarget(),
            rhs.GetTarget()->GetDatacenter(),
            *GetSource(),
            rhs.GetSource()->GetDatacenter()
        );
    }

    bool TDatacenterPairKey::Contains(const TSwitchPairKey& switchPairKey) const {
        return (
            GetTarget() == switchPairKey.GetTarget()->GetDatacenter() && GetSource() == switchPairKey.GetSource()->GetDatacenter());
    }

    bool TDatacenterPairKey::Contains(const TLinePairKey& queuePairKey) const {
        return (
            GetTarget() == queuePairKey.GetTarget()->GetDatacenter() && GetSource() == queuePairKey.GetSource()->GetDatacenter());
    }

    bool TDatacenterPairKey::Contains(const THost& targetHost, const THost& sourceHost) const {
        return GetTarget() == targetHost.GetDatacenter() && GetSource() == sourceHost.GetDatacenter();
    }

    bool TDatacenterPairKey::Contains(const THostInterface& targetIface, const THostInterface& sourceIface) const {
        return GetTarget() == targetIface.GetDatacenter() && GetSource() == sourceIface.GetDatacenter();
    }

    bool TDatacenterPairKey::Contains(const TProbe& probe) const {
        return Contains(*probe.GetTargetIface(), *probe.GetSourceIface());
    }

    bool TDatacenterPairKey::SameTarget(const TSwitchPairKey& switchPairKey) const {
        return GetTarget() == switchPairKey.GetTarget()->GetDatacenter();
    }

    bool TDatacenterPairKey::SameTarget(const TLinePairKey& queuePairKey) const {
        return GetTarget() == queuePairKey.GetTarget()->GetDatacenter();
    }

    bool TDatacenterPairKey::SameTarget(const THost& targetHost) const {
        return GetTarget() == targetHost.GetDatacenter();
    }

    bool TDatacenterPairKey::SameTarget(const THostInterface& targetIface) const {
        return GetTarget() == targetIface.GetDatacenter();
    }
}

template <>
void Out<NNetmon::TSwitchPairKey>(IOutputStream& stream,
                                  TTypeTraits<NNetmon::TSwitchPairKey>::TFuncParam key) {
    stream << "TSwitchPairKey(source=\"" << key.GetSource()->GetName() << "\", target=\"" << key.GetTarget()->GetName() << "\")";
}

template <>
void Out<NNetmon::TLinePairKey>(IOutputStream& stream,
                                TTypeTraits<NNetmon::TLinePairKey>::TFuncParam key) {
    stream << "TLinePairKey(source=\"" << key.GetSource()->GetName() << "\", target=\"" << key.GetTarget()->GetName() << "\")";
}

template <>
void Out<NNetmon::TDatacenterPairKey>(IOutputStream& stream,
                                      TTypeTraits<NNetmon::TDatacenterPairKey>::TFuncParam key) {
    stream << "TDatacenterPairKey(source=\"" << key.GetSource()->GetName() << "\", target=\"" << key.GetTarget()->GetName() << "\")";
}
