#pragma once

#include <infra/netmon/topology/topology_storage.h>

#include <infra/netmon/idl/common.fbs.h>

#include <util/digest/multi.h>
#include <util/generic/maybe.h>

namespace NNetmon {
    class TProbe;
    class TSwitchPairKey;
    class TLinePairKey;
    class TDatacenterPairKey;

    namespace {
        template <class T, class TObjectRef, class TFlat, class TFlatKey>
        class TObjectPairKey {
        public:
            using TType = typename TObjectRef::TType;
            using TTypeRef = TObjectRef;

            inline TObjectPairKey(const TObjectRef& target, const TObjectRef& source) noexcept
                : Target(target)
                , Source(source)
            {
                static const TFlatKey empty(0);
                if (IsValid()) {
                    Flat.ConstructInPlace(Source->ToProto(), Target->ToProto());
                } else if (Source) {
                    Flat.ConstructInPlace(Source->ToProto(), empty);
                } else if (Target) {
                    Flat.ConstructInPlace(empty, Target->ToProto());
                } else {
                    Flat.ConstructInPlace(empty, empty);
                }
            }

            inline const TObjectRef& GetTarget() const noexcept {
                return Target;
            }
            inline const TObjectRef& GetSource() const noexcept {
                return Source;
            }

            inline size_t GetHash() const noexcept {
                return MultiHash(GetTarget()->GetHash(), GetSource()->GetHash());
            }

            inline bool IsValid() const noexcept {
                return Source && Target;
            }

            inline bool IsPartiallyValid() const noexcept {
                return Source || Target;
            }

            inline const TFlat& ToProto() const noexcept {
                return Flat.GetRef();
            }

            inline const TFlat* ToOptionalProto() const noexcept {
                return Flat.Get();
            }

            inline bool operator==(const T& rhs) const noexcept {
                return GetTarget() == rhs.GetTarget() && GetSource() == rhs.GetSource();
            }

            inline bool operator!=(const T& rhs) const noexcept {
                return !(*this == rhs);
            }

        private:
            const TObjectRef Target;
            const TObjectRef Source;
            TMaybe<TFlat> Flat;
        };
    }

    class TSwitchPairKey: public TObjectPairKey<
            TSwitchPairKey, TTopology::TSwitchRef,
            NCommon::TSwitchPairKey, NCommon::TSwitch> {
    public:
        using TParent = TLinePairKey;

        using TObjectPairKey::TObjectPairKey;

        inline bool operator<(const TSwitchPairKey& rhs) const noexcept {
            return IsLess(
                GetTarget()->GetComparator(),
                rhs.GetTarget()->GetComparator(),
                GetSource()->GetComparator(),
                rhs.GetSource()->GetComparator()
            );
        }

        bool operator<(const TLinePairKey& rhs) const;
        bool operator<(const TDatacenterPairKey& rhs) const;

        bool Contains(const THost& targetHost, const THost& sourceHost) const;
        bool Contains(const THostInterface& targetIface, const THostInterface& sourceIface) const;
        bool Contains(const TProbe& probe) const;

        bool SameTarget(const THost& targetHost) const;
        bool SameTarget(const THostInterface& targetIface) const;

        TLinePairKey GetParent() const;

        TSwitchPairKey(const TProbe& probe);
        TSwitchPairKey(const TTopologyStorage& topologyStorage,
                       const NCommon::TSwitchPairKey& key);
    };

    class TLinePairKey: public TObjectPairKey<
            TLinePairKey, TTopology::TLineRef,
            NCommon::TLinePairKey, NCommon::TLine> {
    public:
        using TParent = TDatacenterPairKey;

        using TObjectPairKey::TObjectPairKey;

        inline bool operator<(const TLinePairKey& rhs) const noexcept {
            return IsLess(
                GetTarget()->GetComparator(),
                rhs.GetTarget()->GetComparator(),
                GetSource()->GetComparator(),
                rhs.GetSource()->GetComparator()
            );
        }
        bool operator<(const TSwitchPairKey& rhs) const;
        bool operator<(const TDatacenterPairKey& rhs) const;

        bool Contains(const TSwitchPairKey& switchPairKey) const;
        bool Contains(const THost& targetHost, const THost& sourceHost) const;
        bool Contains(const THostInterface& targetIface, const THostInterface& sourceIface) const;
        bool Contains(const TProbe& probe) const;

        bool SameTarget(const TSwitchPairKey& switchPairKey) const;
        bool SameTarget(const THost& targetHost) const;
        bool SameTarget(const THostInterface& targetIface) const;

        TDatacenterPairKey GetParent() const;

        TLinePairKey(const TTopologyStorage& topologyStorage,
                      const NCommon::TLinePairKey& key);
    };

    class TDatacenterPairKey: public TObjectPairKey<
            TDatacenterPairKey, TTopology::TDatacenterRef,
            NCommon::TDatacenterPairKey, NCommon::TDatacenter> {
    public:
        using TObjectPairKey::TObjectPairKey;

        inline bool operator<(const TDatacenterPairKey& rhs) const noexcept {
            return IsLess(GetTarget(), rhs.GetTarget(), GetSource(), rhs.GetSource());
        }
        bool operator<(const TSwitchPairKey& rhs) const;
        bool operator<(const TLinePairKey& rhs) const;

        bool Contains(const TSwitchPairKey& switchPairKey) const;
        bool Contains(const TLinePairKey& queuePairKey) const;
        bool Contains(const THost& targetHost, const THost& sourceHost) const;
        bool Contains(const THostInterface& targetIface, const THostInterface& sourceIface) const;
        bool Contains(const TProbe& probe) const;

        bool SameTarget(const TSwitchPairKey& switchPairKey) const;
        bool SameTarget(const TLinePairKey& queuePairKey) const;
        bool SameTarget(const THost& targetHost) const;
        bool SameTarget(const THostInterface& targetIface) const;

        TDatacenterPairKey(const TTopologyStorage& topologyStorage,
                           const NCommon::TDatacenterPairKey& key);
    };

    inline TDatacenterPairKey ToPairKey(const TTopologyStorage& topologyStorage, const NCommon::TDatacenterPairKey& pairKey) noexcept {
        return TDatacenterPairKey(topologyStorage, pairKey);
    }

    inline TLinePairKey ToPairKey(const TTopologyStorage& topologyStorage, const NCommon::TLinePairKey& pairKey) noexcept {
        return TLinePairKey(topologyStorage, pairKey);
    }

    inline TSwitchPairKey ToPairKey(const TTopologyStorage& topologyStorage, const NCommon::TSwitchPairKey& pairKey) noexcept {
        return TSwitchPairKey(topologyStorage, pairKey);
    }
}

template <>
class THash<NNetmon::TSwitchPairKey> {
public:
    size_t operator()(const NNetmon::TSwitchPairKey& key) const {
        return key.GetHash();
    }
};

template <>
class THash<NNetmon::TLinePairKey> {
public:
    size_t operator()(const NNetmon::TLinePairKey& key) const {
        return key.GetHash();
    }
};

template <>
class THash<NNetmon::TDatacenterPairKey> {
public:
    size_t operator()(const NNetmon::TDatacenterPairKey& key) const {
        return key.GetHash();
    }
};
