#include <infra/netmon/state_keys.h>

#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NNetmon;

class TStateKeysTest: public TTestBase {
    UNIT_TEST_SUITE(TStateKeysTest);
    UNIT_TEST(TestSwitch)
    UNIT_TEST(TestSwitchSerialization)
    UNIT_TEST(TestQueue)
    UNIT_TEST(TestDatacenter)
    UNIT_TEST(TestValidation)
    UNIT_TEST_SUITE_END();

public:
    TStateKeysTest()
        : TopologyStorage(TGlobalTopology::GetTopologyStorage())
    {
    }

private:
    inline void TestSwitch() {
        TTopology::THostRef hostA(TopologyStorage.FindHost("vla1-1234.search.yandex.net"));
        TTopology::THostRef hostB(TopologyStorage.FindHost("jmon-test.search.yandex.net"));
        TTopology::THostRef hostC(TopologyStorage.FindHost("sas1-1234.search.yandex.net"));
        UNIT_ASSERT(hostA);
        UNIT_ASSERT(hostB);
        UNIT_ASSERT(hostC);

        TTopology::TSwitchRef switchA(hostA.GetSwitch());
        TTopology::TSwitchRef switchB(hostB.GetSwitch());
        UNIT_ASSERT(switchA);
        UNIT_ASSERT(switchB);

        TSwitchPairKey key(switchB, switchA);
        UNIT_ASSERT(key.Contains(*hostB, *hostA));
        UNIT_ASSERT(!key.Contains(*hostB, *hostC));
        UNIT_ASSERT(!key.Contains(*hostC, *hostA));
        UNIT_ASSERT(key.SameTarget(*hostB));
        UNIT_ASSERT(!key.SameTarget(*hostC));
    }

    inline void TestSwitchSerialization() {
        TTopology::THostRef hostA(TopologyStorage.FindHost("jmon-test.search.yandex.net"));
        TTopology::THostRef hostB(TopologyStorage.FindHost("vla1-1234.search.yandex.net"));
        UNIT_ASSERT(hostA);
        UNIT_ASSERT(hostB);

        // switch serialization
        TSwitchPairKey originalSwitchKey(hostB.GetSwitch(), hostA.GetSwitch());
        const NCommon::TSwitchPairKey serializedSwitchKey(originalSwitchKey.ToProto());
        TSwitchPairKey restoredSwitchKey(TopologyStorage, serializedSwitchKey);
        UNIT_ASSERT(originalSwitchKey == restoredSwitchKey);

        // queue serialization
        TLinePairKey originalQueueKey(hostB.GetLine(), hostA.GetLine());
        const NCommon::TLinePairKey serializedQueueKey(originalQueueKey.ToProto());
        TLinePairKey restoredQueueKey(TopologyStorage, serializedQueueKey);
        UNIT_ASSERT(originalQueueKey == restoredQueueKey);

        // datacenter serialization
        TDatacenterPairKey originalDatacenterKey(hostB.GetDatacenter(), hostA.GetDatacenter());
        const NCommon::TDatacenterPairKey serializedDatacenterKey(originalDatacenterKey.ToProto());
        TDatacenterPairKey restoredDatacenterKey(TopologyStorage, serializedDatacenterKey);
        UNIT_ASSERT(originalDatacenterKey == restoredDatacenterKey);
    }

    inline void TestQueue() {
        TTopology::THostRef hostA(TopologyStorage.FindHost("jmon-test.search.yandex.net"));
        TTopology::THostRef hostB(TopologyStorage.FindHost("vla1-1234.search.yandex.net"));
        TTopology::THostRef hostC(TopologyStorage.FindHost("sas1-1234.search.yandex.net"));

        UNIT_ASSERT(hostA);
        UNIT_ASSERT(hostB);
        UNIT_ASSERT(hostC);

        TTopology::TLineRef queueA(hostA.GetLine());
        TTopology::TLineRef queueB(hostB.GetLine());
        UNIT_ASSERT(queueA);
        UNIT_ASSERT(queueB);

        TLinePairKey key(queueB, queueA);
        UNIT_ASSERT(key.Contains(*hostB, *hostA));
        UNIT_ASSERT(!key.Contains(*hostB, *hostC));
        UNIT_ASSERT(!key.Contains(*hostC, *hostA));
        UNIT_ASSERT(key.SameTarget(*hostB));
        UNIT_ASSERT(!key.SameTarget(*hostC));
    }

    inline void TestDatacenter() {
        TTopology::THostRef hostA(TopologyStorage.FindHost("myt1-1826.search.yandex.net"));
        TTopology::THostRef hostB(TopologyStorage.FindHost("man1-0020.search.yandex.net"));
        TTopology::THostRef hostC(TopologyStorage.FindHost("sas1-1234.search.yandex.net"));
        UNIT_ASSERT(hostA);
        UNIT_ASSERT(hostB);
        UNIT_ASSERT(hostC);

        TTopology::TDatacenterRef datacenterA(hostA.GetDatacenter());
        TTopology::TDatacenterRef datacenterB(hostB.GetDatacenter());
        UNIT_ASSERT(datacenterA);
        UNIT_ASSERT(datacenterB);

        TDatacenterPairKey key(datacenterB, datacenterA);
        UNIT_ASSERT(key.Contains(*hostB, *hostA));
        UNIT_ASSERT(!key.Contains(*hostB, *hostC));
        UNIT_ASSERT(!key.Contains(*hostC, *hostA));
        UNIT_ASSERT(key.SameTarget(*hostB));
        UNIT_ASSERT(!key.SameTarget(*hostC));
    }

    inline void TestValidation() {
        TTopology::THostRef knownHost(TopologyStorage.FindHost("jmon-test.search.yandex.net"));
        UNIT_ASSERT(knownHost);

        TDatacenterPairKey validKey(knownHost.GetDatacenter(), knownHost.GetDatacenter());
        UNIT_ASSERT(validKey.IsValid());
        UNIT_ASSERT(validKey.IsPartiallyValid());
        UNIT_ASSERT(validKey.ToOptionalProto());
        UNIT_ASSERT(validKey.GetSource());
        UNIT_ASSERT(validKey.GetTarget());

        TDatacenterPairKey invalidKey(knownHost.GetDatacenter(), TopologyStorage.FindDatacenter("wrong"));
        UNIT_ASSERT(!invalidKey.IsValid());
        UNIT_ASSERT(invalidKey.IsPartiallyValid());
        UNIT_ASSERT(invalidKey.ToOptionalProto());
        UNIT_ASSERT(!invalidKey.GetSource());
        UNIT_ASSERT(invalidKey.GetTarget());

        TDatacenterPairKey trashKey(TopologyStorage.FindDatacenter("wrong"), TopologyStorage.FindDatacenter("wrong"));
        UNIT_ASSERT(!trashKey.IsValid());
        UNIT_ASSERT(!trashKey.IsPartiallyValid());
        // Now we create (empty, empty) proto for invalid key
        UNIT_ASSERT(trashKey.ToOptionalProto());
        UNIT_ASSERT(!trashKey.GetSource());
        UNIT_ASSERT(!trashKey.GetTarget());

        TDatacenterPairKey restoredValidKey(TopologyStorage, *validKey.ToOptionalProto());
        UNIT_ASSERT(restoredValidKey.IsValid());
        UNIT_ASSERT(restoredValidKey.IsPartiallyValid());
        UNIT_ASSERT(restoredValidKey.GetSource());
        UNIT_ASSERT(restoredValidKey.GetTarget());

        TDatacenterPairKey restoredInvalidKey(TopologyStorage, *invalidKey.ToOptionalProto());
        UNIT_ASSERT(!restoredInvalidKey.IsValid());
        UNIT_ASSERT(restoredInvalidKey.IsPartiallyValid());
        UNIT_ASSERT(!restoredInvalidKey.GetSource());
        UNIT_ASSERT(restoredInvalidKey.GetTarget());
    }

    TTopologyStorage& TopologyStorage;
};

UNIT_TEST_SUITE_REGISTRATION(TStateKeysTest);
