#include <infra/netmon/state_picker.h>
#include <infra/netmon/response_gatherer.h>
#include <infra/netmon/probe.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/library/fences.h>
#include <infra/netmon/library/boxes.h>

#include <util/random/random.h>

namespace NNetmon {
    namespace {
        TStringBuf LevelToString(const IStateRequester& requester) {
            switch (requester.GetLevel()) {
                case NApi::ELevel_LEVEL_DATACENTER:
                    return TStringBuf("datacenter");
                case NApi::ELevel_LEVEL_QUEUE:
                    return TStringBuf("queue");
                case NApi::ELevel_LEVEL_SWITCH:
                    return TStringBuf("switch");
                default:
                    Y_VERIFY(false);
            }
        }

        TString LevelToPath(const IStateRequester& requester) {
            switch (requester.GetLevel()) {
                case NApi::ELevel_LEVEL_DATACENTER:
                    return "/slicer/v1/datacenter_state";
                case NApi::ELevel_LEVEL_QUEUE:
                    return "/slicer/v1/queue_state";
                case NApi::ELevel_LEVEL_SWITCH:
                    return "/slicer/v1/switch_state";
                default:
                    Y_VERIFY(false);
            }
        }

        ENetmonSignals LevelToSignal(const IStateRequester& requester) {
            switch (requester.GetLevel()) {
                case NApi::ELevel_LEVEL_DATACENTER:
                    return ENetmonSignals::SlicerDcCollectTime;
                case NApi::ELevel_LEVEL_QUEUE:
                    return ENetmonSignals::SlicerQueueCollectTime;
                case NApi::ELevel_LEVEL_SWITCH:
                    return ENetmonSignals::SlicerSwitchCollectTime;
                default:
                    Y_VERIFY(false);
            }
        }

        template <class T>
        struct TPickerHashFunctor {
            size_t operator()(const T& x) const {
                return Default<THash<TProbeAggregatorKey>>()(x->GetKey());
            }
            size_t operator()(const TProbeAggregatorKey& x) const {
                return Default<THash<TProbeAggregatorKey>>()(x);
            }
        };

        template <class T>
        struct TPickerEqualFunctor {
            bool operator()(const T& lhs, const T& rhs) const {
                return lhs->GetKey() == rhs->GetKey();
            }
            bool operator()(const T& lhs, const TProbeAggregatorKey& rhs) const {
                return lhs->GetKey() == rhs;
            }
        };

        class TPickerState: public TAvlTreeItem<TPickerState, TCompareUsingDeadline>,
                            public TAtomicRefCount<TPickerState> {
        public:
            using TRef = TIntrusivePtr<TPickerState>;
            using TListRef = TVector<TRef>;
            using TSetRef = THashSet<TRef, TPickerHashFunctor<TRef>, TPickerEqualFunctor<TRef>>;
            using TTree = TAvlTree<TPickerState, TCompareUsingDeadline>;

            template <typename... Args>
            static inline TRef Make(Args&&... args) {
                return MakeIntrusive<TPickerState>(std::forward<Args>(args)...);
            }

            TPickerState(const TProbeAggregatorKey& key, const TDuration& interval)
                : Key(key)
                , Interval(interval)
            {
                ui64 jitter = RandomNumber<ui64>(interval.MicroSeconds());
                TInstant now(TInstant::Now());
                Deadline = TInstant::MicroSeconds(now.MicroSeconds() + jitter);
            }

            inline const TProbeAggregatorKey& GetKey() {
                return Key;
            }

            inline const TInstant& GetDeadline() const {
                return Deadline;
            }

            inline void InsertInto(TTree& tree) {
                TInstant now(TInstant::Now());
                while (Deadline < now) {
                    Deadline += Interval;
                }
                tree.Insert(this);
            }

        private:
            const TProbeAggregatorKey Key;
            const TDuration Interval;
            TInstant Deadline;
        };

        class TPickerRequest: public TIntrusiveListItem<TPickerRequest> {
        public:
            using TRef = THolder<TPickerRequest>;
            using TListType = TIntrusiveList<TPickerRequest>;

            TPickerRequest(TAtomicLockedBox<TPickerState::TTree>& treeBox)
                : TreeBox(treeBox)
            {
            }

            ~TPickerRequest() {
                Release();
            }

            inline bool Acquire(const std::size_t limit) {
                TInstant now(TInstant::Now());
                auto ownedTree(TreeBox.Own());
                while (!ownedTree->Empty() && ObtainedStates.size() < limit) {
                    TPickerState::TRef state(&(*ownedTree->First()));
                    if (state->GetDeadline() <= now) {
                        state->Unlink();
                        // state is belong to request now
                        ObtainedStates.emplace_back(std::move(state));
                    } else {
                        break;
                    }
                }
                return !ObtainedStates.empty();
            }

            inline const TPickerState::TListRef& GetKeys() const {
                return ObtainedStates;
            }
            inline TDuration GetTime() {
                return Timer.Get();
            }

        private:
            inline void Release() {
                auto ownedTree(TreeBox.Own());
                while (!ObtainedStates.empty()) {
                    TPickerState::TRef state(ObtainedStates.back());
                    ObtainedStates.pop_back();
                    state->InsertInto(*ownedTree);
                }
            }

            TAtomicLockedBox<TPickerState::TTree>& TreeBox;
            TPickerState::TListRef ObtainedStates;
            TSimpleTimer Timer;
        };
    }

    class TStatePicker::TImpl: public TNonCopyable {
    public:
        using TRequestList = TPickerRequest::TListType;
        using TStateTree = TPickerState::TTree;

        inline TImpl(IStateRequester& requester)
            : Requester(requester)
        {
        }

        inline TThreadPool::TFuture Run() {

            DEBUG_LOG << "Let's ask slicers for " << LevelToString(Requester) << " level keys" << Endl;

            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NApi::CreateTAggregatorKeysRequest(builder, NApi::ELevel_LEVEL_DATACENTER));

            TGatherer<NApi::TAggregatorKeysResponse> gatherer("/slicer/v1/aggregator_keys", builder);
            return gatherer.Gather().Apply([box = gatherer.Box(), this] (const TThreadPool::TFuture& future_){
                future_.GetValue();

                TProbeAggregatorKeySet newKeys;
                {
                    auto result(box->Own());
                    for (const auto& response : *result) {
                        for (const auto& key : *response->Keys()) {
                            newKeys.emplace(*key);
                        }
                    }
                }

                DEBUG_LOG << newKeys.size() << " keys found for " << LevelToString(Requester) << " level" << Endl;

                if (SyncStates(newKeys)) {
                    DEBUG_LOG << "There are some states on " << LevelToString(Requester)
                              << " level that we can process, let's spawn requests" << Endl;

                    ScheduleRequests();
                }
            });
        }

        inline TInstant GetDeadline() const {
            return TInstant::Now() - Requester.GetInterval() * 3;
        }

    private:
        bool SyncStates(TProbeAggregatorKeySet& newKeys) {
            auto ownedStateQueue(StateQueue.Own());
            auto ownedStates(States.Own());

            TProbeAggregatorKeySet oldKeys;
            for (const auto& state : *ownedStates) {
                oldKeys.emplace(state->GetKey());
            }

            TVector<TProbeAggregatorKey> keysToCreate;
            SetDifference(newKeys.begin(), newKeys.end(), oldKeys.begin(), oldKeys.end(),
                          std::back_inserter(keysToCreate));

            TVector<TProbeAggregatorKey> keysToDelete;
            SetDifference(oldKeys.begin(), oldKeys.end(), newKeys.begin(), newKeys.end(),
                          std::back_inserter(keysToDelete));

            for (const auto& key : keysToDelete) {
                auto it = ownedStates->find(key);
                if (!it.IsEnd()) {
                    ownedStates->erase(it);
                }
            }

            for (const auto& key : keysToCreate) {
                auto pickerState(TPickerState::Make(key, Requester.GetInterval()));
                pickerState->InsertInto(*ownedStateQueue);
                ownedStates->emplace(std::move(pickerState));
            }

            if (!keysToDelete.empty() || !keysToCreate.empty()) {
                INFO_LOG << "There are " << keysToDelete.size() << " keys to delete and"
                         << " " << keysToCreate.size() << " keys to create on"
                         << " " << LevelToString(Requester) << " level" << Endl;
            }

            return !ownedStates->empty();
        }

        TThreadPool::TFuture DoRequest(const TPickerState::TListRef& states) {
            flatbuffers::FlatBufferBuilder builder;
            std::vector<NCommon::TAggregatorKey> keys;
            keys.reserve(states.size());
            for (const auto& state : states) {
                keys.push_back(state->GetKey().ToProto());
            }
            builder.Finish(NApi::CreateTStateRequest(
                builder,
                builder.CreateVectorOfStructs(keys)
            ));
            return Requester.Collect(LevelToPath(Requester), builder);
        }

        void ScheduleRequests() {
            DEBUG_LOG << "Let's schedule requests to slicers on " << LevelToString(Requester) << " level" << Endl;

            // limit isn't strict because function may be called from multiple threads
            bool requestsCreated = false;
            bool hasSlots = false;
            {
                auto ownedRequests(InFlightRequests.Own());
                hasSlots = ownedRequests->Size() < Requester.GetRequestLimit();
            }

            while (hasSlots) {
                TPickerRequest::TRef request(MakeHolder<TPickerRequest>(StateQueue));
                if (request->Acquire(Requester.GetBatchSize())) {
                    INFO_LOG << "Starting to collect slices for " << LevelToString(Requester)
                             << " level for " << request->GetKeys().size()
                             << " keys, request #" << (size_t)(request.Get())
                             << Endl;

                    {
                        auto ownedRequests(InFlightRequests.Own());
                        ownedRequests->PushBack(request.Get());
                        hasSlots = ownedRequests->Size() < Requester.GetRequestLimit();
                    }

                    /* FIXME: Guard should be created while being in parent thread
                       for thread sync to take effect.
                       C++11 does not support move semantics in lambda and
                       TGuard is not copyable, so workaround with TIntrusivePtr
                       instead */

                    DoRequest(request->GetKeys()).Apply(
                        [request_ = request.Release(), this, guard = MakeIntrusive<TOwningGuard>(RequestFence)] (const TThreadPool::TFuture& future_) {
                            auto ownedRequests(InFlightRequests.Own());
                            TPickerRequest::TRef request(request_);

                            try {
                                future_.GetValue();
                            } catch (...) {
                                ERROR_LOG << "Request #" << (size_t)request.Get() << " failed: "
                                          << CurrentExceptionMessage() << Endl;
                                return;
                            }

                            INFO_LOG << "Slices for " << LevelToString(Requester) << " level collected, request #"
                                     << (size_t)request.Get() << ", took " << request->GetTime()
                                     << Endl;

                            TUnistat::Instance().PushSignalUnsafe(
                                LevelToSignal(Requester), request->GetTime().MilliSeconds());
                        }
                    );

                    requestsCreated = true;
                } else {
                    break;
                }
            }

            if (!hasSlots) {
                DEBUG_LOG << "There are too many requests for " << LevelToString(Requester)
                          << " level, try again later" << Endl;
            } else if (!requestsCreated) {
                DEBUG_LOG << "Seems that there no states that should be updated for "
                          << LevelToString(Requester) << " level, try again later" << Endl;
            }
        }

        IStateRequester& Requester;

        TAtomicLockedBox<TPickerState::TSetRef> States;
        TAtomicLockedBox<TStateTree> StateQueue;
        TAtomicLockedBox<TRequestList> InFlightRequests;

        TOwningFence RequestFence;
    };

    TStatePicker::TStatePicker(IStateRequester& requester)
        : Impl(MakeHolder<TImpl>(requester))
    {
    }

    TStatePicker::~TStatePicker()
    {
    }

    TThreadPool::TFuture TStatePicker::Run() {
        return Impl->Run();
    }

    TInstant TStatePicker::GetDeadline() const {
        return Impl->GetDeadline();
    }
}
