#include <infra/netmon/statistics/accumulators.h>

namespace NNetmon {
    // based on https://en.wikipedia.org/wiki/Algorithms_for_calculating_variance

    TDescriptiveStatistics::TDescriptiveStatistics(const TDescriptiveStatistics& other)
    {
        *this = other;
    }

    TDescriptiveStatistics::TDescriptiveStatistics(const NCommon::TDescriptiveStatistics& statistics)
    {
        FromProto(statistics);
    }

    void TDescriptiveStatistics::Append(double value) {
        const std::size_t prevCount = Count;
        Count++;

        const double delta = value - MeanMoment;
        const double normDelta = delta / Count;

        MeanMoment += normDelta;
        VarianceMoment += delta * normDelta * prevCount;
    }

    void TDescriptiveStatistics::Merge(const TDescriptiveStatistics& other) {
        const std::size_t prevCount = Count;
        Count += other.Count;

        const double delta = other.MeanMoment - MeanMoment;
        const double squareDelta = delta * delta;

        MeanMoment = MeanMoment * prevCount / Count + other.MeanMoment * other.Count / Count;
        VarianceMoment += other.VarianceMoment + squareDelta * prevCount * other.Count / Count;
    }

    TMaybe<double> TDescriptiveStatistics::Mean() {
        if (Count) {
            return MeanMoment;
        } else {
            return Nothing();
        }
    }

    TMaybe<double> TDescriptiveStatistics::Variance() {
        if (Count > 1) {
            return VarianceMoment / (Count - 1);
        } else {
            return Nothing();
        }
    }

    flatbuffers::Offset<NCommon::TDescriptiveStatistics> TDescriptiveStatistics::ToProto(
            flatbuffers::FlatBufferBuilder& builder) const {
        return NCommon::CreateTDescriptiveStatistics(builder, Count, MeanMoment, VarianceMoment);
    }

    void TDescriptiveStatistics::FromProto(const NCommon::TDescriptiveStatistics& statistics) {
        Count = statistics.Count();
        MeanMoment = statistics.MeanMoment();
        VarianceMoment = statistics.VarianceMoment();
    }

    TDescriptiveStatistics& TDescriptiveStatistics::operator=(const TDescriptiveStatistics& other) {
        Count = other.Count;
        MeanMoment = other.MeanMoment;
        VarianceMoment = other.VarianceMoment;
        return *this;
    }
}
