#pragma once

#include <infra/netmon/idl/series.fbs.h>
#include <infra/netmon/library/memory_pool.h>

#include <util/generic/vector.h>
#include <util/stream/zerocopy.h>
#include <util/stream/output.h>

namespace NNetmon {
    extern TAtomic MemPoolStatsFixedBlockCount;

    TSafeFixedSizeAllocator* GetBlockAllocator(size_t capacity);

    class TFixedBlock {
    private:
        void* Data_ = nullptr;
        ui32 Capacity_ = 0;

    public:
        inline ~TFixedBlock() noexcept {
            if (Data_) {
                GetBlockAllocator(Capacity_)->Release(Data_);
                AtomicDecrement(MemPoolStatsFixedBlockCount);
            }
        }

        inline void GrowUp(size_t capacity) {
            capacity = capacity ? Max(FastClp2(capacity), 64UL) : 0UL;
            if (Data_) {
                void* newData_ = GetBlockAllocator(capacity)->Allocate();
                std::memcpy(newData_, Data_, Capacity_);
                GetBlockAllocator(Capacity_)->Release(Data_);
                Data_ = newData_;
                Capacity_ = capacity;
            } else {
                Data_ = GetBlockAllocator(capacity)->Allocate();
                Capacity_ = capacity;
                AtomicIncrement(MemPoolStatsFixedBlockCount);
            }
        }

        inline char* Data() noexcept {
            return reinterpret_cast<char*>(Data_);
        }

        inline const char* Data() const noexcept {
            return reinterpret_cast<char*>(Data_);
        }

        inline size_t Size() const noexcept {
            return Capacity_;
        }
    };

    class TChunkedInputStream;

    class TChunkedOutputStream final : public IOutputStream {
    public:
        friend TChunkedInputStream;

        inline TChunkedOutputStream()
        {
        }

        inline TChunkedOutputStream(const TChunkedOutputStream& other)
            : Written_(other.Written_)
        {
            if (Written_) {
                Block_.GrowUp(Written_);
                std::memcpy(Block_.Data(), other.Block_.Data(), Written_);
            }
        }

        inline std::size_t Size() const noexcept {
            return Written_;
        }

        inline flatbuffers::Offset<NCommon::TChunkedStream> ToProto(flatbuffers::FlatBufferBuilder& builder) const {
            TVector<flatbuffers::Offset<NCommon::TOneChunk>> chunks;
            if (Block_.Size()) {
                chunks.reserve(1);
                chunks.emplace_back(NCommon::CreateTOneChunk(
                    builder,
                    builder.CreateVector(
                        reinterpret_cast<const i8*>(Block_.Data()),
                        Written_
                    )
                ));
            }
            return NCommon::CreateTChunkedStream(
                builder,
                builder.CreateVector(chunks)
            );
        }

        inline void FromProto(const NCommon::TChunkedStream& stream) {
            for (const auto& chunk : *stream.Chunks()) {
                DoWrite(chunk->Data()->data(), chunk->Data()->size());
            }
        }

    private:
        inline void DoWrite(const void* buf, size_t len) override {
            if (Written_ + len > Block_.Size()) {
                Block_.GrowUp(Written_ + len);
            }

            std::memcpy(Block_.Data() + Written_, reinterpret_cast<const char*>(buf), len);
            Written_ += len;
        }

        TFixedBlock Block_;
        size_t Written_ = 0;
    };

    class TChunkedInputStream final : public IZeroCopyInput {
    public:
        inline TChunkedInputStream(const TChunkedOutputStream* stream) noexcept
            : Block_(stream ? &stream->Block_ : nullptr)
            , Written_(stream ? stream->Written_ : 0)
        {
        }

    private:
        inline size_t DoNext(const void** ptr, size_t len) noexcept override {
            if (Block_ && Block_->Size()) {
                len = Min(Written_ - Readed_, len);
                *ptr = Block_->Data() + Readed_;
                Readed_ += len;
                return len;
            } else {
                return 0;
            }
        }

        const TFixedBlock* Block_ = nullptr;
        const size_t Written_ = 0;
        size_t Readed_ = 0;
    };
}
