#pragma once

#include <infra/netmon/statistics/chunked_stream.h>
#include <infra/netmon/statistics/varint.h>

#include <util/datetime/base.h>

namespace NNetmon {
    namespace NImpl {
        enum class EMode : ui8 {
            EMPTY,
            CLEAN,
            DIRTY
        };

        template <class T>
        inline T XorOperator(T first, T second) noexcept {
            return first ^ second;
        }

        template <>
        inline double XorOperator<double>(double first, double second) noexcept {
            auto result(XorOperator<ui64>(reinterpret_cast<ui64&>(first), reinterpret_cast<ui64&>(second)));
            return reinterpret_cast<double&>(result);
        }
    }

    class TTimestampDecoder;

    // this encoder assume that timestamps was rounded to interval value
    class TTimestampEncoder {
    public:
        friend TTimestampDecoder;

        inline TTimestampEncoder(const TDuration& interval) noexcept
            : Interval(interval.Seconds())
        {
        }

        inline TTimestampEncoder(const TTimestampEncoder& other) noexcept
            : Stream(other.Stream)
            , Interval(other.Interval)
            , First(other.First)
            , Last(other.Last)
            , Mode(other.Mode)
        {
        }

        inline TTimestampEncoder(const NCommon::TChunkedStream& stream) noexcept
        {
            Stream.FromProto(stream);
        }

        inline void Write(TInstant value) {
            using NImpl::EMode;
            switch (Mode) {
                case EMode::EMPTY: {
                    Mode = EMode::CLEAN;
                    Last = value.Seconds();
                    VarintEncode<ui64>(Stream, Interval, Last);
                    break;
                }
                case EMode::CLEAN: {
                    Mode = EMode::DIRTY;
                    First = value.Seconds() - Last - Interval;
                    Last = value.Seconds();
                    break;
                }
                case EMode::DIRTY: {
                    Mode = EMode::CLEAN;
                    // delta calculated with saturation
                    const ui32 second = (value - TInstant::Seconds(Last) - TDuration::Seconds(Interval)).Seconds();
                    Last = value.Seconds();
                    VarintEncode<ui32>(Stream, First, second);
                    break;
                }
            };
        }

        inline void Finish() {
            using NImpl::EMode;
            if (Mode == EMode::DIRTY) {
                Mode = EMode::CLEAN;
                VarintEncode<ui32>(Stream, First);
            }
        }

        inline const TChunkedOutputStream* GetStream() const {
            return &Stream;
        }

        inline std::size_t Size() const {
            return Stream.Size();
        }

        inline TDuration GetInterval() const {
            return TDuration::Seconds(Interval);
        }

    private:
        TChunkedOutputStream Stream;
        const ui32 Interval = 0;

        ui32 First = 0;
        ui32 Last = 0;
        NImpl::EMode Mode = NImpl::EMode::EMPTY;
    };

    class TTimestampDecoder {
    public:
        using TResult = std::pair<bool, TInstant>;

        inline TTimestampDecoder(const TTimestampEncoder& encoder) noexcept
            : Encoder(encoder)
            , Stream(Encoder.GetStream())
        {
        }

        inline TResult Read() {
            LastResult = DoRead();
            if (!LastResult.first && !Exhausted && Encoder.Mode == NImpl::EMode::DIRTY) {
                Exhausted = true;
                LastResult = TResult{true, TInstant::Seconds(Encoder.Last)};
            }
            return LastResult;
        }

        inline TResult Peek() {
            if (Exhausted) {
                return LastResult;
            } else if (Mode == NImpl::EMode::EMPTY) {
                return {false, TInstant::Zero()};
            } else {
                return LastResult;
            }
        }

    private:
        inline TResult DoRead() {
            using NImpl::EMode;
            switch (Mode) {
                case EMode::EMPTY: {
                    ui64 interval;
                    ui64 started;
                    if (VarintDecode<ui64>(Stream, interval, started) == EVarintStatus::END_OF_STREAM) {
                        return {false, TInstant::Zero()};
                    } else {
                        Mode = EMode::CLEAN;
                        Interval = TDuration::Seconds(interval);
                        Last = TInstant::Seconds(started);
                        return {true, Last};
                    }
                }
                case EMode::CLEAN: {
                    ui32 first;
                    auto status(VarintDecode<ui32>(Stream, first, Second));
                    if (status == EVarintStatus::END_OF_STREAM) {
                        return {false, TInstant::Zero()};
                    } else {
                        Mode = status == EVarintStatus::TWO_VALUES ? EMode::DIRTY : EMode::CLEAN;
                        Last = Last + Interval + TDuration::Seconds(first);
                        return {true, Last};
                    }
                }
                case EMode::DIRTY: {
                    Mode = EMode::CLEAN;
                    Last = Last + Interval + TDuration::Seconds(Second);
                    return {true, Last};
                }
            };
        }

        const TTimestampEncoder& Encoder;
        TChunkedInputStream Stream;

        TDuration Interval;
        TInstant Last;
        ui32 Second = 0;
        NImpl::EMode Mode = NImpl::EMode::EMPTY;
        bool Exhausted = false;

        TResult LastResult;
    };

    template <class TValue> class TValueDecoder;

    template <class TValue>
    class TValueEncoder {
    public:
        friend class TValueDecoder<TValue>;

        inline TValueEncoder() noexcept
        {
        }

        inline TValueEncoder(const TValueEncoder& other) noexcept
            : Stream(other.Stream)
            , First(other.First)
            , Last(other.Last)
            , Mode(other.Mode)
        {
        }

        inline void Write(TValue value, size_t point=0) {
            using NImpl::EMode;
            switch (Mode) {
                case EMode::EMPTY: {
                    Mode = EMode::CLEAN;
                    Last = value;
                    VarintEncode<TValue>(Stream, reinterpret_cast<TValue&>(point), value);
                    break;
                }
                case EMode::CLEAN: {
                    Mode = EMode::DIRTY;
                    First = NImpl::XorOperator(Last, value);
                    Last = value;
                    break;
                }
                case EMode::DIRTY: {
                    Mode = EMode::CLEAN;
                    const TValue second = NImpl::XorOperator(Last, value);
                    Last = value;
                    VarintEncode<TValue>(Stream, First, second);
                    break;
                }
            };
        }

        inline void Finish() {
            using NImpl::EMode;
            if (Mode == EMode::DIRTY) {
                Mode = EMode::CLEAN;
                VarintEncode<TValue>(Stream, First);
            }
        }

        inline const TChunkedOutputStream* GetStream() const {
            return &Stream;
        }

        inline void FromProto(const NCommon::TChunkedStream& stream) noexcept {
            Stream.FromProto(stream);
        }

        inline std::size_t Size() const {
            return Stream.Size();
        }

    private:
        TChunkedOutputStream Stream;
        TValue First = 0;
        TValue Last = 0;
        NImpl::EMode Mode = NImpl::EMode::EMPTY;
    };

    template <class TValue>
    class TValueDecoder {
    public:
        using TResult = std::tuple<bool, TValue, size_t>;

        inline TValueDecoder(const TValueEncoder<TValue>& encoder) noexcept
            : Encoder(encoder)
            , Stream(Encoder.GetStream())
        {
        }

        inline TResult Read() {
            LastResult = DoRead();
            if (!std::get<0>(LastResult) && !Exhausted && Encoder.Mode == NImpl::EMode::DIRTY) {
                Exhausted = true;
                LastResult = TResult{true, Encoder.Last, ++Point};
            }
            return LastResult;
        }

        inline TResult Peek() {
            if (Exhausted) {
                return LastResult;
            } else if (Mode == NImpl::EMode::EMPTY) {
                return {false, TValue(), 0};
            } else {
                return LastResult;
            }
        }

    private:
        inline TResult DoRead() {
            using NImpl::EMode;
            switch (Mode) {
                case EMode::EMPTY: {
                    if (VarintDecode<TValue>(Stream, reinterpret_cast<TValue&>(Point), Last) == EVarintStatus::END_OF_STREAM) {
                        return {false, TValue(), 0};
                    } else {
                        Mode = EMode::CLEAN;
                        return {true, Last, Point};
                    }
                }
                case EMode::CLEAN: {
                    TValue first;
                    auto status(VarintDecode<TValue>(Stream, first, Second));
                    if (status == EVarintStatus::END_OF_STREAM) {
                        return {false, TValue(), 0};
                    } else {
                        Mode = status == EVarintStatus::TWO_VALUES ? EMode::DIRTY : EMode::CLEAN;
                        Last = NImpl::XorOperator(Last, first);
                        return {true, Last, ++Point};
                    }
                }
                case EMode::DIRTY: {
                    Mode = EMode::CLEAN;
                    Last = NImpl::XorOperator(Last, Second);
                    return {true, Last, ++Point};
                }
            };
        }

        const TValueEncoder<TValue>& Encoder;
        TChunkedInputStream Stream;

        size_t Point = 0;
        TValue Second = 0;
        TValue Last = 0;
        NImpl::EMode Mode = NImpl::EMode::EMPTY;
        bool Exhausted = false;

        TResult LastResult;
    };
}
