#include <infra/netmon/statistics/codecs.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/xrange.h>
#include <util/stream/null.h>

using namespace NNetmon;

class TTimestampCodecTest: public TTestBase {
    UNIT_TEST_SUITE(TTimestampCodecTest);
    UNIT_TEST(TestEmpty)
    UNIT_TEST(TestOne)
    UNIT_TEST(TestTwo)
    UNIT_TEST(TestThree)
    UNIT_TEST(TestMultiple)
    UNIT_TEST(TestCopy)
    UNIT_TEST_SUITE_END();

private:
    inline void TestEmpty() {
        TTimestampEncoder encoder(TDuration::Seconds(5));
        TTimestampDecoder decoder(encoder);
        UNIT_ASSERT(!decoder.Read().first);
    }

    inline void TestOne() {
        TTimestampEncoder encoder(TDuration::Seconds(5));
        encoder.Write(TInstant::Seconds(100));
        TTimestampDecoder decoder(encoder);
        AssertRead(decoder, TInstant::Seconds(100));
        UNIT_ASSERT(!decoder.Read().first);
    }

    inline void TestTwo() {
        TTimestampEncoder encoder(TDuration::Seconds(5));
        encoder.Write(TInstant::Seconds(100));
        encoder.Write(TInstant::Seconds(105));
        TTimestampDecoder decoder(encoder);
        AssertRead(decoder, TInstant::Seconds(100));
        AssertRead(decoder, TInstant::Seconds(105));
        UNIT_ASSERT(!decoder.Read().first);
    }

    inline void TestThree() {
        TTimestampEncoder encoder(TDuration::Seconds(5));
        encoder.Write(TInstant::Seconds(100));
        encoder.Write(TInstant::Seconds(105));
        encoder.Write(TInstant::Seconds(115));
        TTimestampDecoder decoder(encoder);
        AssertRead(decoder, TInstant::Seconds(100));
        AssertRead(decoder, TInstant::Seconds(105));
        AssertRead(decoder, TInstant::Seconds(115));
        UNIT_ASSERT(!decoder.Read().first);
    }

    inline void TestMultiple() {
        TTimestampEncoder encoder(TDuration::Seconds(5));
        for (auto idx : xrange(1000)) {
            encoder.Write(TInstant::Seconds(10000 + idx * 5));
        }
        TTimestampDecoder decoder(encoder);
        for (auto idx : xrange(1000)) {
            AssertRead(decoder, TInstant::Seconds(10000 + idx * 5));
        }
        UNIT_ASSERT(!decoder.Read().first);

        TNullOutput output;
        TChunkedInputStream input(encoder.GetStream());
        UNIT_ASSERT(TransferData(&input, &output));
    }

    inline void TestCopy() {
        TTimestampEncoder source(TDuration::Seconds(5));
        for (auto idx : xrange(100)) {
            source.Write(TInstant::Seconds(100000 + idx * 5));
        }

        TTimestampEncoder target(source);
        target.Write(TInstant::Seconds(100000 + 100 * 5));

        TTimestampDecoder sourceDecoder(source);
        TTimestampDecoder targetDecoder(target);

        for (auto idx : xrange(100)) {
            const auto value(TInstant::Seconds(100000 + idx * 5));
            AssertRead(sourceDecoder, value);
            AssertRead(targetDecoder, value);
        }

        UNIT_ASSERT(!sourceDecoder.Read().first);
        UNIT_ASSERT(targetDecoder.Read().first);
        UNIT_ASSERT(!targetDecoder.Read().first);
    }

    inline void AssertRead(TTimestampDecoder& decoder, TInstant value) {
        CompareResult(decoder.Read(), value);
        CompareResult(decoder.Peek(), value);
    }

    inline void CompareResult(const TTimestampDecoder::TResult& result, TInstant value) {
        UNIT_ASSERT(result.first);
        UNIT_ASSERT_EQUAL(result.second, value);
    }
};

UNIT_TEST_SUITE_REGISTRATION(TTimestampCodecTest);

class TValueCodecTest: public TTestBase {
    UNIT_TEST_SUITE(TValueCodecTest);
    UNIT_TEST(TestEmpty)
    UNIT_TEST(TestOne)
    UNIT_TEST(TestTwo)
    UNIT_TEST(TestThree)
    UNIT_TEST(TestMultiple)
    UNIT_TEST(TestCopy)
    UNIT_TEST_SUITE_END();

private:
    inline void TestEmpty() {
        TValueEncoder<ui32> encoder;
        TValueDecoder<ui32> decoder(encoder);
        UNIT_ASSERT(!std::get<0>(decoder.Read()));
    }

    inline void TestOne() {
        TValueEncoder<ui64> encoder;
        encoder.Write(12345);
        TValueDecoder<ui64> decoder(encoder);
        AssertRead<ui64>(decoder, 12345, 0);
        UNIT_ASSERT(!std::get<0>(decoder.Read()));
    }

    inline void TestTwo() {
        TValueEncoder<ui64> encoder;
        encoder.Write(12345, 5);
        encoder.Write(12346);
        TValueDecoder<ui64> decoder(encoder);
        AssertRead<ui64>(decoder, 12345, 5);
        AssertRead<ui64>(decoder, 12346, 6);
        UNIT_ASSERT(!std::get<0>(decoder.Read()));
    }

    inline void TestThree() {
        TValueEncoder<ui64> encoder;
        encoder.Write(12345, 5);
        encoder.Write(12346);
        encoder.Write(123456);
        TValueDecoder<ui64> decoder(encoder);
        AssertRead<ui64>(decoder, 12345, 5);
        AssertRead<ui64>(decoder, 12346, 6);
        AssertRead<ui64>(decoder, 123456, 7);
        UNIT_ASSERT(!std::get<0>(decoder.Read()));
    }

    inline void TestMultiple() {
        TValueEncoder<ui64> encoder;
        for (auto idx : xrange(1000)) {
            encoder.Write(123456789 + idx);
        }
        TValueDecoder<ui64> decoder(encoder);
        for (auto idx : xrange(1000)) {
            AssertRead<ui64>(decoder, 123456789 + idx, idx);
        }
        UNIT_ASSERT(!std::get<0>(decoder.Read()));

        TNullOutput output;
        TChunkedInputStream input(encoder.GetStream());
        UNIT_ASSERT(TransferData(&input, &output));
    }

    inline void TestCopy() {
        TValueEncoder<ui64> source;
        for (auto idx : xrange(100)) {
            source.Write(idx);
        }

        TValueEncoder<ui64> target(source);
        target.Write(100);

        TValueDecoder<ui64> sourceDecoder(source);
        TValueDecoder<ui64> targetDecoder(target);

        for (auto idx : xrange(100)) {
            AssertRead<ui64>(sourceDecoder, idx, idx);
            AssertRead<ui64>(targetDecoder, idx, idx);
        }

        UNIT_ASSERT(!std::get<0>(sourceDecoder.Read()));
        UNIT_ASSERT(std::get<0>(targetDecoder.Read()));
        UNIT_ASSERT(!std::get<0>(targetDecoder.Read()));
    }

    template <class T>
    inline void AssertRead(TValueDecoder<T>& decoder, T value, size_t point) {
        CompareResult(decoder.Read(), value, point);
        CompareResult(decoder.Peek(), value, point);
    }

    template <class T>
    inline void CompareResult(const typename TValueDecoder<T>::TResult& result, T value, size_t point) {
        UNIT_ASSERT(std::get<0>(result));
        UNIT_ASSERT_EQUAL(std::get<1>(result), value);
        UNIT_ASSERT_EQUAL(std::get<2>(result), point);
    }
};

UNIT_TEST_SUITE_REGISTRATION(TValueCodecTest);
