#pragma once

#include <infra/netmon/idl/histograms.fbs.h>

#include <library/cpp/json/writer/json.h>

#include <util/generic/maybe.h>

#include <array>

namespace NNetmon {
    const size_t RTT_BUCKET_COUNT = 64;
    const size_t CONNECTIVITY_BUCKET_COUNT = 4;

    const TVector<double>& GetBucketWeights();

    class TSampleHistogramSeries;

    class TSampleHistogram {
    public:
        using TBuckets = std::array<ui32, RTT_BUCKET_COUNT>;

        friend TSampleHistogramSeries;

        TSampleHistogram();
        TSampleHistogram(const NCommon::TSampleHistogram& hist);

        template <class TContainer>
        TSampleHistogram(const TContainer& bucketValues) {
            Y_VERIFY(bucketValues.size() == RTT_BUCKET_COUNT);
            Copy(begin(bucketValues), end(bucketValues), begin(Buckets));
            InitMinMaxIndexFromBuckets();
        }

        void Append(double value);

        void Merge(const TSampleHistogram& hist, ui32 weight = 1);

        void FillCumulativeCount(TBuckets& cumulativeCount) const;

        TMaybe<double> GetPercentileValue(double percent, const TBuckets& cumulativeCount) const;
        TMaybe<double> GetPercentileValue(double percent) const;

        double GetPercentileValueOrNan(double percent, const TBuckets& cumulativeCount) const;
        double GetPercentileValueOrNan(double percent) const;

        void ToJson(NJsonWriter::TBuf& buf) const;
        flatbuffers::Offset<NCommon::TSampleHistogram> ToProto(
                flatbuffers::FlatBufferBuilder& builder) const;
        void FromProto(const NCommon::TSampleHistogram& hist);

        bool operator==(const TSampleHistogram& rhs) const noexcept;
        inline bool operator!=(const TSampleHistogram& rhs) const noexcept {
            return !(*this == rhs);
        }

        const TBuckets& RawBuckets() const {
            return Buckets;
        }

    private:
        void InitMinMaxIndexFromBuckets();

        TBuckets Buckets;
        ui8 MinIndex;
        ui8 MaxIndex;
    };

    class TConnectivityHistogramSeries;

    class TConnectivityHistogram {
    public:
        using TBuckets = std::array<double, CONNECTIVITY_BUCKET_COUNT>;
        using TNormalizedBuckets = std::array<double, CONNECTIVITY_BUCKET_COUNT>;
        using TSeries = TConnectivityHistogramSeries;

        friend TConnectivityHistogramSeries;

        TConnectivityHistogram();
        TConnectivityHistogram(const NCommon::TConnectivityHistogram& hist);

        void Append(double value, double weight = 1);
        void Merge(const TConnectivityHistogram& hist);
        inline void Merge(const TConnectivityHistogram& hist, ui32) {
            Merge(hist);
        }

        inline bool Empty() const noexcept {
            return WeightAccumulator == 0.0;
        }

        TMaybe<TNormalizedBuckets> GetValues() const;

        void ToJson(NJsonWriter::TBuf& buf) const;
        flatbuffers::Offset<NCommon::TConnectivityHistogram> ToProto(
                flatbuffers::FlatBufferBuilder& builder) const;
        void FromProto(const NCommon::TConnectivityHistogram& hist);

        bool operator==(const TConnectivityHistogram& rhs) const noexcept;
        inline bool operator!=(const TConnectivityHistogram& rhs) const noexcept {
            return !(*this == rhs);
        }

    private:
        // buckets for connectivity: 0.5, 0.7, 0.9, 1
        TBuckets Buckets;
        double WeightAccumulator;
    };

    class TAverageHistogramSeries;

    class TAverage {
    public:
        friend TAverageHistogramSeries;

        void Append(double value, ui32 weight = 1);

        void Merge(const TAverage& other);

        TMaybe<double> GetValue() const;

        NCommon::TAverage ToProto() const;
        void FromProto(const NCommon::TAverage& element);

        bool operator==(const TAverage& rhs) const noexcept;
        inline bool operator!=(const TAverage& rhs) const noexcept {
            return !(*this == rhs);
        }

    private:
        double ValueAccumulator = 0.0;
        ui64 WeightAccumulator = 0;
    };

    class TAverageHistogram {
    public:
        using TSeries = TAverageHistogramSeries;

        friend TAverageHistogramSeries;

        TAverageHistogram() = default;
        TAverageHistogram(const NCommon::TAverageHistogram& hist);

        void Merge(const TConnectivityHistogram& hist, ui32 weight = 1);
        void Merge(const TAverageHistogram& hist);

        TMaybe<TConnectivityHistogram::TNormalizedBuckets> GetValues() const;

        void ToJson(NJsonWriter::TBuf& buf) const;

        flatbuffers::Offset<NCommon::TAverageHistogram> ToProto(
                flatbuffers::FlatBufferBuilder& builder) const;
        void FromProto(const NCommon::TAverageHistogram& hist);

        bool operator==(const TAverageHistogram& rhs) const noexcept;
        inline bool operator!=(const TAverageHistogram& rhs) const noexcept {
            return !(*this == rhs);
        }

    private:
        // medians for connectivity: 0.5, 0.7, 0.9, 1
        using TAverageList = std::array<TAverage, CONNECTIVITY_BUCKET_COUNT>;

        TAverageList Elements;
    };
}
