#include <infra/netmon/statistics/histograms.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/gtest.h>
#include <library/cpp/json/json_reader.h>

#include <cfloat>

using namespace NNetmon;

class TConnectivityHistogramTest: public TTestBase {
    UNIT_TEST_SUITE(TConnectivityHistogramTest);
    UNIT_TEST(TestAppend)
    UNIT_TEST(TestSerialization)
    UNIT_TEST(TestEmpty)
    UNIT_TEST_SUITE_END();

private:
    inline void TestAppend() {
        TConnectivityHistogram hist;
        hist.Append(0.0);
        hist.Append(0.4);
        hist.Append(0.5);
        hist.Append(0.7);
        hist.Append(0.9);
        hist.Append(1.0);
        auto buckets = hist.GetValues();
        UNIT_ASSERT(buckets.Defined());
        UNIT_ASSERT_DOUBLES_EQUAL(buckets->at(0), 2.0 / 3.0, FLT_EPSILON);
        UNIT_ASSERT_DOUBLES_EQUAL(buckets->at(1), 0.5, FLT_EPSILON);
        UNIT_ASSERT_DOUBLES_EQUAL(buckets->at(2), 1.0 / 3.0, FLT_EPSILON);
        UNIT_ASSERT_DOUBLES_EQUAL(buckets->at(3), 1.0 / 6.0, FLT_EPSILON);
    }

    inline void TestSerialization() {
        TConnectivityHistogram hist;
        hist.Append(0.0);
        hist.Append(0.4);
        hist.Append(0.5);

        // let's check JSON output
        {
            NJsonWriter::TBuf buf;
            hist.ToJson(buf);
            NJson::TJsonValue result;
            NJson::ReadJsonFastTree(buf.Str(), &result, true);
            UNIT_ASSERT(result.IsArray());
        }

        // let's check Protobuf output
        {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(hist.ToProto(builder));
            const NCommon::TConnectivityHistogram& dumpedHist(
                *flatbuffers::GetRoot<NCommon::TConnectivityHistogram>(builder.GetBufferPointer()));
            TConnectivityHistogram loadedHist(dumpedHist);
            auto buckets = loadedHist.GetValues();
            UNIT_ASSERT(buckets.Defined());
            UNIT_ASSERT(!buckets->empty());
        }
    }

    inline void TestEmpty() {
        TConnectivityHistogram hist;
        auto buckets = hist.GetValues();
        UNIT_ASSERT(buckets.Empty());
    }
};

UNIT_TEST_SUITE_REGISTRATION(TConnectivityHistogramTest);

class TSampleHistogramTest: public TTestBase {
    UNIT_TEST_SUITE(TSampleHistogramTest);
    UNIT_TEST(TestAppend)
    UNIT_TEST(TestMerge)
    UNIT_TEST(TestSerialization)
    UNIT_TEST(TestEmpty)
    UNIT_TEST_SUITE_END();

private:
    inline void TestAppend() {
        TSampleHistogram hist;
        hist.Append(0.01);
        hist.Append(0.1);
        hist.Append(1.0);
        hist.Append(10.0);
        hist.Append(100.0);
        hist.Append(1000.0);
        UNIT_ASSERT_DOUBLES_EQUAL(hist.GetPercentileValue(0.1).GetRef(), 0.01, 0.001);
        UNIT_ASSERT_DOUBLES_EQUAL(hist.GetPercentileValue(0.3).GetRef(), 0.1, 0.006);
        UNIT_ASSERT_DOUBLES_EQUAL(hist.GetPercentileValue(0.5).GetRef(), 1.0, 0.06);
        UNIT_ASSERT_DOUBLES_EQUAL(hist.GetPercentileValue(0.6).GetRef(), 10.0, 0.26);
        UNIT_ASSERT_DOUBLES_EQUAL(hist.GetPercentileValue(0.8).GetRef(), 100.0, 6);
        UNIT_ASSERT_DOUBLES_EQUAL(hist.GetPercentileValue(0.95).GetRef(), 1000.0, 0.001);
    }

    inline void TestMerge() {
        TSampleHistogram h1, h2, h3;
        h1.Append(0.1);
        h1.Append(0.2);
        h1.Append(0.3);
        h2.Append(0.2);
        h2.Append(0.3);
        h2.Append(0.4);
        h3.Merge(h1);
        h3.Merge(h2);
        UNIT_ASSERT_DOUBLES_EQUAL(h3.GetPercentileValue(0.5).GetRef(), 0.2, 0.026);

        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(h3.ToProto(builder));
        const NCommon::TSampleHistogram& dumpedHist(
            *flatbuffers::GetRoot<NCommon::TSampleHistogram>(builder.GetBufferPointer()));
        TSampleHistogram loadedHist(dumpedHist);
        UNIT_ASSERT_DOUBLES_EQUAL(loadedHist.GetPercentileValue(0.5).GetRef(), 0.2, 0.026);
    }

    inline void TestSerialization() {
        TSampleHistogram hist;
        hist.Append(0.1);
        hist.Append(1.0);
        hist.Append(10.0);

        // let's check JSON output
        {
            NJsonWriter::TBuf buf;
            hist.ToJson(buf);
            NJson::TJsonValue result;
            NJson::ReadJsonFastTree(buf.Str(), &result, true);
            UNIT_ASSERT(result.IsArray());
        }

        // let's check Protobuf output
        {
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(hist.ToProto(builder));
            const NCommon::TSampleHistogram& dumpedHist(
                *flatbuffers::GetRoot<NCommon::TSampleHistogram>(builder.GetBufferPointer()));
            TSampleHistogram loadedHist(dumpedHist);
            auto value = loadedHist.GetPercentileValue(0.5);
            UNIT_ASSERT(value > 0.0);
        }
    }

    inline void TestEmpty() {
        TSampleHistogram hist;
        UNIT_ASSERT(hist.GetPercentileValue(0.5).Empty());
    }
};

UNIT_TEST_SUITE_REGISTRATION(TSampleHistogramTest);

class TAverageTest: public TTestBase {
    UNIT_TEST_SUITE(TAverageTest);
    UNIT_TEST(TestAppend)
    UNIT_TEST(TestEmpty)
    UNIT_TEST(TestSerialization)
    UNIT_TEST_SUITE_END();

private:
    inline void TestAppend() {
        TAverage average;
        average.Append(1.0, 1);
        UNIT_ASSERT_DOUBLES_EQUAL(average.GetValue().GetRef(), 1.0, FLT_EPSILON);
        average.Append(2.0, 2);
        UNIT_ASSERT_DOUBLES_EQUAL(average.GetValue().GetRef(), 5.0 / 3.0, FLT_EPSILON);
        average.Append(3.0, 3);
        UNIT_ASSERT_DOUBLES_EQUAL(average.GetValue().GetRef(), 14.0 / 6.0, FLT_EPSILON);
    }

    inline void TestEmpty() {
        TAverage average;
        UNIT_ASSERT(average.GetValue().Empty());
    }

    inline void TestSerialization() {
        TConnectivityHistogram connectivity;
        connectivity.Append(0.0);
        connectivity.Append(1.0);

        TAverageHistogram hist;
        hist.Merge(connectivity);

        {
            const auto values = hist.GetValues();
            UNIT_ASSERT(values.Defined());
            UNIT_ASSERT_DOUBLES_EQUAL(values->at(0), 0.5, FLT_EPSILON);
            UNIT_ASSERT_DOUBLES_EQUAL(values->at(1), 0.5, FLT_EPSILON);
            UNIT_ASSERT_DOUBLES_EQUAL(values->at(2), 0.5, FLT_EPSILON);
            UNIT_ASSERT_DOUBLES_EQUAL(values->at(3), 0.5, FLT_EPSILON);
        }

        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(hist.ToProto(builder));

        const NCommon::TAverageHistogram& dumpedHist(
            *flatbuffers::GetRoot<NCommon::TAverageHistogram>(builder.GetBufferPointer()));
        TAverageHistogram loadedHist(dumpedHist);

        {
            const auto values = loadedHist.GetValues();
            UNIT_ASSERT(values.Defined());
            UNIT_ASSERT_DOUBLES_EQUAL(values->at(0), 0.5, FLT_EPSILON);
            UNIT_ASSERT_DOUBLES_EQUAL(values->at(1), 0.5, FLT_EPSILON);
            UNIT_ASSERT_DOUBLES_EQUAL(values->at(2), 0.5, FLT_EPSILON);
            UNIT_ASSERT_DOUBLES_EQUAL(values->at(3), 0.5, FLT_EPSILON);
        }
    }
};

UNIT_TEST_SUITE_REGISTRATION(TAverageTest);

