#include <infra/netmon/statistics/series.h>

#include <util/generic/xrange.h>

namespace NNetmon {
    TTimestampSeries::TTimestampSeries(const NCommon::TTimestampSeries& series)
        : Encoder(*series.Stream())
    {
    }

    void TTimestampSeries::TIteratorState::Inc() noexcept {
        auto result(Decoder.Peek());
        if (!result.first) {
            Exhausted = true;
        } else {
            Timestamp = result.second;
            Decoder.Read();
        }
    }

    flatbuffers::Offset<NCommon::TTimestampSeries> TTimestampSeries::ToProto(flatbuffers::FlatBufferBuilder& builder) const {
        return NCommon::CreateTTimestampSeries(builder, Encoder.GetStream()->ToProto(builder));
    }

    TSampleHistogramSeries::TSampleHistogramSeries(const NCommon::TSampleHistogramSeries& series)
        : MinIndex(series.MinIndex())
        , MaxIndex(series.MaxIndex())
    {
        Encoders.resize(MaxIndex + 1);
        for (const auto idx : xrange(MinIndex, static_cast<ui8>(MaxIndex + 1))) {
            Encoders[idx].FromProto(*series.Streams()->Get(idx - MinIndex));
        }
    }

    void TSampleHistogramSeries::TIteratorState::Inc() noexcept {
        Histogram.MinIndex = RTT_BUCKET_COUNT;
        Histogram.MaxIndex = 0;
        Exhausted = true;
        for (const auto idx : xrange(Decoders.size())) {
            auto& decoder(Decoders[idx]);
            auto result(decoder.Peek());
            if (std::get<0>(result) && std::get<2>(result) >= NextPoint) {
                Exhausted = false;
                if (std::get<2>(result) == NextPoint) {
                    Histogram.Buckets[idx] = std::get<1>(result);
                    Histogram.MinIndex = Min(Histogram.MinIndex, static_cast<ui8>(idx));
                    Histogram.MaxIndex = idx;
                    decoder.Read();
                }
            } else if (!std::get<0>(result) && !Exhausted) {
                break;
            }
        }
        ++NextPoint;
    }

    void TSampleHistogramSeries::Append(const THistogram& hist) {
        MinIndex = Min(hist.MinIndex, MinIndex);
        MaxIndex = Max(hist.MaxIndex, MaxIndex);
        Encoders.resize(Min(FastClp2(static_cast<size_t>(MaxIndex + 1)), RTT_BUCKET_COUNT));
        for (const auto idx : xrange(MinIndex, static_cast<ui8>(MaxIndex + 1))) {
            Encoders[idx].Write(hist.Buckets[idx], CurrentPoint);
        }
        ++CurrentPoint;
    }

    void TSampleHistogramSeries::Finish() {
        for (auto& encoder : Encoders) {
            encoder.Finish();
        }
        if (MinIndex > MaxIndex && CurrentPoint) {
            MinIndex = 0;
            MaxIndex = 0;
            Encoders.resize(1);
            Encoders[MaxIndex].Write(0, CurrentPoint - 1);
            Encoders[MaxIndex].Finish();
        }
    }

    std::size_t TSampleHistogramSeries::Size() const {
        std::size_t size = sizeof(TSampleHistogramSeries);
        for (const auto& encoder : Encoders) {
            size += encoder.Size();
        }
        return size;
    }

    flatbuffers::Offset<NCommon::TSampleHistogramSeries> TSampleHistogramSeries::ToProto(flatbuffers::FlatBufferBuilder& builder) const {
        TVector<flatbuffers::Offset<NCommon::TChunkedStream>> streams;
        if (MinIndex <= MaxIndex) {
            streams.reserve(MaxIndex + 1 - MinIndex);
        }
        for (const auto idx : xrange(MinIndex, static_cast<ui8>(MaxIndex + 1))) {
            streams.emplace_back(Encoders[idx].GetStream()->ToProto(builder));
        }
        return NCommon::CreateTSampleHistogramSeries(
            builder,
            MinIndex,
            MaxIndex,
            builder.CreateVector(streams)
        );
    }

    TConnectivityHistogramSeries::TConnectivityHistogramSeries(const NCommon::TConnectivityHistogramSeries& series)
    {
        for (const auto idx : xrange(Encoders.size())) {
            Encoders[idx].FromProto(*series.ValueStreams()->Get(idx));
        }
        WeightEncoder.FromProto(*series.WeightStream());
    }

    void TConnectivityHistogramSeries::TIteratorState::Inc() noexcept {
        auto result(WeightDecoder.Peek());
        if (!std::get<0>(result)) {
            Exhausted = true;
            return;
        }
        Histogram.WeightAccumulator = std::get<1>(result);
        WeightDecoder.Read();

        for (const auto idx : xrange(Decoders.size())) {
            auto& decoder(Decoders[idx]);
            auto result(decoder.Peek());
            Y_ASSERT(std::get<0>(result));
            Histogram.Buckets[idx] = std::get<1>(result);
            decoder.Read();
        }
    }

    void TConnectivityHistogramSeries::Append(const THistogram& hist) {
        for (const auto idx : xrange(Encoders.size())) {
            Encoders[idx].Write(hist.Buckets[idx]);
        }
        WeightEncoder.Write(hist.WeightAccumulator);
    }

    void TConnectivityHistogramSeries::Finish() {
        for (auto& encoder : Encoders) {
            encoder.Finish();
        }
        WeightEncoder.Finish();
    }

    std::size_t TConnectivityHistogramSeries::Size() const {
        std::size_t size = sizeof(TConnectivityHistogramSeries) + WeightEncoder.Size();
        for (const auto& encoder : Encoders) {
            size += encoder.Size();
        }
        return size;
    }

    flatbuffers::Offset<NCommon::TConnectivityHistogramSeries> TConnectivityHistogramSeries::ToProto(flatbuffers::FlatBufferBuilder& builder) const {
        TVector<flatbuffers::Offset<NCommon::TChunkedStream>> streams;
        streams.reserve(Encoders.size());
        for (const auto& encoder : Encoders) {
            streams.emplace_back(encoder.GetStream()->ToProto(builder));
        }
        return NCommon::CreateTConnectivityHistogramSeries(
            builder,
            builder.CreateVector(streams),
            WeightEncoder.GetStream()->ToProto(builder)
        );
    }

    TAverageHistogramSeries::TAverageHistogramSeries(const NCommon::TAverageHistogramSeries& series)
    {
        for (const auto idx : xrange(ValueEncoders.size())) {
            ValueEncoders[idx].FromProto(*series.ValueStreams()->Get(idx));
        }
        for (const auto idx : xrange(WeightEncoders.size())) {
            WeightEncoders[idx].FromProto(*series.WeightStreams()->Get(idx));
        }
    }

    void TAverageHistogramSeries::TIteratorState::Inc() noexcept {
        for (const auto idx : xrange(CONNECTIVITY_BUCKET_COUNT)) {
            auto& weightDecoder(WeightDecoders[idx]);
            auto weightResult(weightDecoder.Peek());
            if (!std::get<0>(weightResult)) {
                Exhausted = true;
                return;
            }
            Histogram.Elements[idx].WeightAccumulator = std::get<1>(weightResult);
            weightDecoder.Read();

            auto& valueDecoder(ValueDecoders[idx]);
            auto valueResult(valueDecoder.Peek());
            Y_ASSERT(std::get<0>(valueResult));
            Histogram.Elements[idx].ValueAccumulator = std::get<1>(valueResult);
            valueDecoder.Read();
        }
    }

    void TAverageHistogramSeries::Append(const THistogram& hist) {
        for (const auto idx : xrange(CONNECTIVITY_BUCKET_COUNT)) {
            ValueEncoders[idx].Write(hist.Elements[idx].ValueAccumulator);
            WeightEncoders[idx].Write(hist.Elements[idx].WeightAccumulator);
        }
    }

    void TAverageHistogramSeries::Finish() {
        for (auto& encoder : WeightEncoders) {
            encoder.Finish();
        }
        for (auto& encoder : ValueEncoders) {
            encoder.Finish();
        }
    }

    std::size_t TAverageHistogramSeries::Size() const {
        std::size_t size = sizeof(TAverageHistogramSeries);
        for (const auto& encoder : WeightEncoders) {
            size += encoder.Size();
        }
        for (const auto& encoder : ValueEncoders) {
            size += encoder.Size();
        }
        return size;
    }

    flatbuffers::Offset<NCommon::TAverageHistogramSeries> TAverageHistogramSeries::ToProto(flatbuffers::FlatBufferBuilder& builder) const {
        TVector<flatbuffers::Offset<NCommon::TChunkedStream>> valueStreams;
        valueStreams.reserve(ValueEncoders.size());
        for (const auto& encoder : ValueEncoders) {
            valueStreams.emplace_back(encoder.GetStream()->ToProto(builder));
        }

        TVector<flatbuffers::Offset<NCommon::TChunkedStream>> weightStreams;
        weightStreams.reserve(WeightEncoders.size());
        for (const auto& encoder : WeightEncoders) {
            weightStreams.emplace_back(encoder.GetStream()->ToProto(builder));
        }

        return NCommon::CreateTAverageHistogramSeries(
            builder,
            builder.CreateVector(valueStreams),
            builder.CreateVector(weightStreams)
        );
    }
}
