#pragma once

#include <infra/netmon/statistics/histograms.h>
#include <infra/netmon/statistics/codecs.h>

namespace NNetmon {
    namespace {
        template <class TIterator, class TValue, class TState>
        class TBaseIterator {
        public:
            inline TBaseIterator() noexcept
            {
            }

            inline TBaseIterator(THolder<TState>&& state) noexcept
                : State(std::move(state))
            {
            }

            inline TBaseIterator(TIterator&& other) noexcept
                : State(std::move(other.State))
            {
            }

            inline const TValue& operator*() const noexcept {
                return State->Get();
            }

            inline const TValue* operator->() const noexcept {
                return &State->Get();
            }

            inline TIterator& operator++() noexcept {
                State->Inc();
                return static_cast<TIterator&>(*this);
            }

            bool operator==(const TIterator& other) const noexcept {
                if (State && other.State) {
                    return false;
                } else if (other.State) {
                    return other.State->IsEnd();
                } else if (State) {
                    return State->IsEnd();
                } else {
                    return true;
                }
            }

            inline bool operator!=(const TIterator& other) const noexcept {
                return !(*this == other);
            }

        private:
            THolder<TState> State;
        };
    }

    class TTimestampSeries {
    private:
        class TIteratorState {
        public:
            inline TIteratorState(const TTimestampEncoder& encoder)
                : Decoder(encoder)
                , Exhausted(false)
            {
                Decoder.Read();
                Inc();
            }

            const TInstant& Get() const noexcept {
                return Timestamp;
            }

            inline bool IsEnd() const noexcept {
                return Exhausted;
            }

            void Inc() noexcept;

        private:
            TTimestampDecoder Decoder;
            TInstant Timestamp;
            bool Exhausted;
        };

    public:
        class TIterator: public TBaseIterator<TIterator, TInstant, TIteratorState> {
        public:
            using TBaseIterator::TBaseIterator;
        };

        using const_iterator = TIterator;

        inline TTimestampSeries(const TDuration& interval) noexcept
            : Encoder(interval)
        {
        }

        inline TTimestampSeries(const TTimestampSeries& other) noexcept
            : Encoder(other.Encoder)
        {
        }

        TTimestampSeries(const NCommon::TTimestampSeries& series);

        inline TIterator begin() const {
            return {MakeHolder<TIteratorState>(Encoder)};
        }

        inline TIterator end() const noexcept {
            return {};
        }

        inline void Append(TInstant now) {
            Encoder.Write(now);
        }

        inline void Finish() {
            Encoder.Finish();
        }

        inline std::size_t Size() const {
            return sizeof(TTimestampSeries) + Encoder.Size();
        }

        inline TDuration GetInterval() const {
            return Encoder.GetInterval();
        }

        flatbuffers::Offset<NCommon::TTimestampSeries> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TTimestampEncoder Encoder;
    };

    class TSampleHistogramSeries {
    private:
        using TEncoders = TVector<TValueEncoder<ui32>>;
        using TDecoders = TVector<TValueDecoder<ui32>>;

        class TIteratorState {
        public:
            inline TIteratorState(const TEncoders& encoders)
                : NextPoint(0)
                , Exhausted(false)
            {
                Decoders.reserve(encoders.size());
                for (auto& encoder : encoders) {
                    Decoders.emplace_back(encoder).Read();
                }
                Inc();
            }

            const TSampleHistogram& Get() const noexcept {
                return Histogram;
            }

            inline bool IsEnd() const noexcept {
                return Exhausted;
            }

            void Inc() noexcept;

        private:
            TDecoders Decoders;
            TSampleHistogram Histogram;
            ui16 NextPoint;
            bool Exhausted;
        };

    public:
        using THistogram = TSampleHistogram;

        class TIterator: public TBaseIterator<TIterator, THistogram, TIteratorState> {
        public:
            using TBaseIterator::TBaseIterator;
        };

        using const_iterator = TIterator;

        inline TSampleHistogramSeries() noexcept
            : CurrentPoint(0)
            , MinIndex(RTT_BUCKET_COUNT)
            , MaxIndex(0)
        {
        }

        inline TSampleHistogramSeries(const TSampleHistogramSeries& other) noexcept
            : Encoders(other.Encoders)
            , CurrentPoint(other.CurrentPoint)
            , MinIndex(other.MinIndex)
            , MaxIndex(other.MaxIndex)
        {
        }

        TSampleHistogramSeries(const NCommon::TSampleHistogramSeries& series);

        inline TIterator begin() const {
            return {MakeHolder<TIteratorState>(Encoders)};
        }

        inline TIterator end() const noexcept {
            return {};
        }

        void Append(const THistogram& hist);
        void Finish();
        std::size_t Size() const;

        flatbuffers::Offset<NCommon::TSampleHistogramSeries> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TEncoders Encoders;
        ui16 CurrentPoint;
        ui8 MinIndex;
        ui8 MaxIndex;
    };

    class TConnectivityHistogramSeries {
    private:
        using TEncoders = std::array<TValueEncoder<double>, CONNECTIVITY_BUCKET_COUNT>;
        using TDecoders = std::array<TValueDecoder<double>, CONNECTIVITY_BUCKET_COUNT>;

        using TWeightEncoder = TValueEncoder<double>;
        using TWeightDecoder = TValueDecoder<double>;

        class TIteratorState {
        public:
            template <std::size_t... I>
            inline TIteratorState(const TWeightEncoder& weightEncoder, const TEncoders& encoders, std::index_sequence<I...>)
                : Decoders{{encoders[I]...}}
                , WeightDecoder(weightEncoder)
                , Exhausted(false)
            {
                for (auto& decoder : Decoders) {
                    decoder.Read();
                }
                WeightDecoder.Read();
                Inc();
            }

            const TConnectivityHistogram& Get() const noexcept {
                return Histogram;
            }

            inline bool IsEnd() const noexcept {
                return Exhausted;
            }

            void Inc() noexcept;

        private:
            TDecoders Decoders;
            TWeightDecoder WeightDecoder;
            TConnectivityHistogram Histogram;
            bool Exhausted;
        };

    public:
        using THistogram = TConnectivityHistogram;

        class TIterator: public TBaseIterator<TIterator, THistogram, TIteratorState> {
        public:
            using TBaseIterator::TBaseIterator;
        };

        using const_iterator = TIterator;

        inline TConnectivityHistogramSeries() noexcept
        {
        }

        inline TConnectivityHistogramSeries(const TConnectivityHistogramSeries& other) noexcept
            : Encoders(other.Encoders)
            , WeightEncoder(other.WeightEncoder)
        {
        }

        TConnectivityHistogramSeries(const NCommon::TConnectivityHistogramSeries& series);

        inline TIterator begin() const {
            return {MakeHolder<TIteratorState>(
                WeightEncoder,
                Encoders,
                std::make_index_sequence<CONNECTIVITY_BUCKET_COUNT>{}
            )};
        }

        inline TIterator end() const noexcept {
            return {};
        }

        void Append(const THistogram& hist);
        void Finish();
        std::size_t Size() const;

        flatbuffers::Offset<NCommon::TConnectivityHistogramSeries> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TEncoders Encoders;
        TWeightEncoder WeightEncoder;
    };

    class TAverageHistogramSeries {
    private:
        using TEncoders = std::array<TValueEncoder<double>, CONNECTIVITY_BUCKET_COUNT>;
        using TDecoders = std::array<TValueDecoder<double>, CONNECTIVITY_BUCKET_COUNT>;

        class TIteratorState {
        public:
            template <std::size_t... I>
            inline TIteratorState(const TEncoders& weightEncoders, const TEncoders& valueEncoders, std::index_sequence<I...>)
                : WeightDecoders{{weightEncoders[I]...}}
                , ValueDecoders{{valueEncoders[I]...}}
                , Exhausted(false)
            {
                for (auto& decoder : WeightDecoders) {
                    decoder.Read();
                }
                for (auto& decoder : ValueDecoders) {
                    decoder.Read();
                }
                Inc();
            }

            const TAverageHistogram& Get() const noexcept {
                return Histogram;
            }

            inline bool IsEnd() const noexcept {
                return Exhausted;
            }

            void Inc() noexcept;

        private:
            TDecoders WeightDecoders;
            TDecoders ValueDecoders;
            TAverageHistogram Histogram;
            bool Exhausted;
        };

    public:
        using THistogram = TAverageHistogram;

        class TIterator: public TBaseIterator<TIterator, THistogram, TIteratorState> {
        public:
            using TBaseIterator::TBaseIterator;
        };

        using const_iterator = TIterator;

        inline TAverageHistogramSeries() noexcept
        {
        }

        inline TAverageHistogramSeries(const TAverageHistogramSeries& other) noexcept
            : WeightEncoders(other.WeightEncoders)
            , ValueEncoders(other.ValueEncoders)
        {
        }

        TAverageHistogramSeries(const NCommon::TAverageHistogramSeries& series);

        inline TIterator begin() const {
            return {MakeHolder<TIteratorState>(
                WeightEncoders,
                ValueEncoders,
                std::make_index_sequence<CONNECTIVITY_BUCKET_COUNT>{}
            )};
        }

        inline TIterator end() const noexcept {
            return {};
        }

        void Append(const THistogram& hist);
        void Finish();
        std::size_t Size() const;

        flatbuffers::Offset<NCommon::TAverageHistogramSeries> ToProto(flatbuffers::FlatBufferBuilder& builder) const;

    private:
        TEncoders WeightEncoders;
        TEncoders ValueEncoders;
    };
}

template <>
struct std::iterator_traits<NNetmon::TTimestampSeries::TIterator> {
    using value_type = TInstant;
};

template <>
struct std::iterator_traits<NNetmon::TSampleHistogramSeries::TIterator> {
    using value_type = NNetmon::TSampleHistogramSeries::THistogram;
};

template <>
struct std::iterator_traits<NNetmon::TConnectivityHistogramSeries::TIterator> {
    using value_type = NNetmon::TConnectivityHistogramSeries::THistogram;
};

template <>
struct std::iterator_traits<NNetmon::TAverageHistogramSeries::TIterator> {
    using value_type = NNetmon::TAverageHistogramSeries::THistogram;
};
