#include <infra/netmon/statistics/series.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/xrange.h>
#include <util/random/easy.h>

using namespace NNetmon;

class TTimestampSeriesTest: public TTestBase {
    UNIT_TEST_SUITE(TTimestampSeriesTest);
    UNIT_TEST(TestEmpty)
    UNIT_TEST(TestIterator)
    UNIT_TEST(TestSerialization)
    UNIT_TEST(TestCopy)
    UNIT_TEST_SUITE_END();

private:
    inline void TestEmpty() {
        TDuration interval(TDuration::Seconds(5));
        TTimestampSeries series(interval);
        size_t index(0);
        for (const auto& ts : series) {
            Y_UNUSED(ts);
            ++index;
        }
        UNIT_ASSERT_EQUAL(index, 0);
    }

    using TTimestamps = std::array<TInstant, 100>;

    inline TTimestamps CreateTimestamps(TDuration interval) {
        TInstant now(TInstant::Seconds(TInstant::Now().Seconds()));
        TTimestamps timestamps;
        for (const auto idx : xrange(timestamps.size())) {
            timestamps[idx] = now + TDuration::Seconds(idx * interval.Seconds());
        }
        return timestamps;
    }

    inline TTimestampSeries CreateSeries(TDuration interval, const TTimestamps& timestamps) {
        TTimestampSeries series(interval);
        for (const auto& ts : timestamps) {
            series.Append(ts);
        }
        return series;
    }

    inline void CheckSeries(const TTimestampSeries& series, const TTimestamps& timestamps) {
        size_t index(0);
        for (const auto& ts : series) {
            UNIT_ASSERT(index < timestamps.size());
            UNIT_ASSERT_EQUAL(ts, timestamps[index]);
            ++index;
        }
        UNIT_ASSERT_EQUAL(index, timestamps.size());
    }

    inline void TestIterator() {
        const TDuration interval(TDuration::Seconds(5));
        const auto timestamps(CreateTimestamps(interval));
        const auto series(CreateSeries(interval, timestamps));
        CheckSeries(series, timestamps);
    }

    inline void TestSerialization() {
        const TDuration interval(TDuration::Seconds(5));
        const auto timestamps(CreateTimestamps(interval));
        auto series(CreateSeries(interval, timestamps));

        series.Finish();
        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(series.ToProto(builder));

        const TTimestampSeries restoredSeries(
            *flatbuffers::GetRoot<NCommon::TTimestampSeries>(builder.GetBufferPointer())
        );
        CheckSeries(restoredSeries, timestamps);
    }

    inline void TestCopy() {
        const TDuration interval(TDuration::Seconds(5));
        const auto timestamps(CreateTimestamps(interval));
        const auto source(CreateSeries(interval, timestamps));
        const TTimestampSeries target(source);
        CheckSeries(target, timestamps);
    }
};

UNIT_TEST_SUITE_REGISTRATION(TTimestampSeriesTest);

class TSampleHistogramSeriesTest: public TTestBase {
    UNIT_TEST_SUITE(TSampleHistogramSeriesTest);
    UNIT_TEST(TestEmpty)
    UNIT_TEST(TestEmptyHistograms)
    UNIT_TEST(TestEmptyBeforeNonEmptyHistograms)
    UNIT_TEST(TestIterator)
    UNIT_TEST(TestMinIndex)
    UNIT_TEST(TestSerialization)
    UNIT_TEST(TestCopy)
    UNIT_TEST_SUITE_END();

private:
    inline void TestEmpty() {
        TSampleHistogramSeries series;
        size_t index(0);
        for (const auto& hist : series) {
            Y_UNUSED(hist);
            ++index;
        }
        UNIT_ASSERT_EQUAL(index, 0);
    }

    inline void TestEmptyHistograms() {
        std::array<TSampleHistogram, 2> histograms;
        auto series(CreateSeries(histograms));
        series.Finish();
        CheckSeries(series, histograms);
    }

    inline void TestEmptyBeforeNonEmptyHistograms() {
        std::array<TSampleHistogram, 2> histograms;
        histograms.back().Append(1.0);
        auto series(CreateSeries(histograms));
        series.Finish();
        CheckSeries(series, histograms);
    }

    using THistograms = std::array<TSampleHistogram, 100>;

    inline THistograms CreateHistograms() {
        THistograms histograms;
        for (const auto idx : xrange(histograms.size())) {
            histograms[idx].Append((idx + 1) * 10.0);
        }
        return histograms;
    }

    template <class T>
    inline TSampleHistogramSeries CreateSeries(const T& histograms) {
        TSampleHistogramSeries series;
        for (const auto& hist : histograms) {
            series.Append(hist);
        }
        return series;
    }

    template <class T>
    inline void CheckSeries(const TSampleHistogramSeries& series, const T& histograms) {
        size_t index(0);
        for (const auto& hist : series) {
            UNIT_ASSERT(index < histograms.size());
            UNIT_ASSERT_EQUAL(hist, histograms[index]);
            ++index;
        }
        UNIT_ASSERT_EQUAL(index, histograms.size());
    }

    inline void TestIterator() {
        const auto histograms(CreateHistograms());
        const auto series(CreateSeries(histograms));
        CheckSeries(series, histograms);
    }

    inline void TestMinIndex() {
        std::array<TSampleHistogram, 4> histograms;
        histograms[0].Append(1000.0);
        histograms[1].Append(100.0);
        histograms[2].Append(10.0);
        histograms[3].Append(1.0);

        const auto series(CreateSeries(histograms));
        CheckSeries(series, histograms);
    }

    inline void TestSerialization() {
        const auto histograms(CreateHistograms());
        auto series(CreateSeries(histograms));

        series.Finish();
        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(series.ToProto(builder));

        const TSampleHistogramSeries restoredSeries(
            *flatbuffers::GetRoot<NCommon::TSampleHistogramSeries>(builder.GetBufferPointer())
        );
        CheckSeries(restoredSeries, histograms);
    }

    inline void TestCopy() {
        const auto histograms(CreateHistograms());
        const auto source(CreateSeries(histograms));
        const TSampleHistogramSeries target(source);
        CheckSeries(target, histograms);
    }
};

UNIT_TEST_SUITE_REGISTRATION(TSampleHistogramSeriesTest);

class TConnectivityHistogramSeriesTest: public TTestBase {
    UNIT_TEST_SUITE(TConnectivityHistogramSeriesTest);
    UNIT_TEST(TestEmpty)
    UNIT_TEST(TestIterator)
    UNIT_TEST(TestSerialization)
    UNIT_TEST(TestCopy)
    UNIT_TEST_SUITE_END();

private:
    inline void TestEmpty() {
        TConnectivityHistogramSeries series;
        size_t index(0);
        for (const auto& hist : series) {
            Y_UNUSED(hist);
            ++index;
        }
        UNIT_ASSERT_EQUAL(index, 0);
    }

    using THistograms = std::array<TConnectivityHistogram, 100>;

    inline THistograms CreateHistograms() {
        THistograms histograms;
        for (const auto idx : xrange(histograms.size())) {
            histograms[idx].Append(RandomNumber<double>(), RandomNumber<double>());
        }
        return histograms;
    }

    inline TConnectivityHistogramSeries CreateSeries(const THistograms& histograms) {
        TConnectivityHistogramSeries series;
        for (const auto& hist : histograms) {
            series.Append(hist);
        }
        return series;
    }

    inline void CheckSeries(const TConnectivityHistogramSeries& series, const THistograms& histograms) {
        size_t index(0);
        for (const auto& hist : series) {
            UNIT_ASSERT(index < histograms.size());
            UNIT_ASSERT_EQUAL(hist, histograms[index]);
            ++index;
        }

        UNIT_ASSERT_EQUAL(index, histograms.size());
    }

    inline void TestIterator() {
        const auto histograms(CreateHistograms());
        const auto series(CreateSeries(histograms));
        CheckSeries(series, histograms);
    }

    inline void TestSerialization() {
        const auto histograms(CreateHistograms());
        auto series(CreateSeries(histograms));

        series.Finish();
        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(series.ToProto(builder));

        const TConnectivityHistogramSeries restoredSeries(
            *flatbuffers::GetRoot<NCommon::TConnectivityHistogramSeries>(builder.GetBufferPointer())
        );
        CheckSeries(restoredSeries, histograms);
    }

    inline void TestCopy() {
        const auto histograms(CreateHistograms());
        const auto source(CreateSeries(histograms));
        const TConnectivityHistogramSeries target(source);
        CheckSeries(target, histograms);
    }
};

UNIT_TEST_SUITE_REGISTRATION(TConnectivityHistogramSeriesTest);

class TAverageHistogramSeriesTest: public TTestBase {
    UNIT_TEST_SUITE(TAverageHistogramSeriesTest);
    UNIT_TEST(TestEmpty)
    UNIT_TEST(TestIterator)
    UNIT_TEST(TestSerialization)
    UNIT_TEST(TestCopy)
    UNIT_TEST_SUITE_END();

private:
    inline void TestEmpty() {
        TAverageHistogramSeries series;
        size_t index(0);
        for (const auto& hist : series) {
            Y_UNUSED(hist);
            ++index;
        }
        UNIT_ASSERT_EQUAL(index, 0);
    }

    using THistograms = std::array<TAverageHistogram, 100>;

    inline THistograms CreateHistograms() {
        THistograms histograms;
        for (const auto idx : xrange(histograms.size())) {
            TConnectivityHistogram hist;
            hist.Append(RandomNumber<double>(), RandomNumber<double>());
            histograms[idx].Merge(hist, RandomNumber<double>());
        }
        return histograms;
    }

    inline TAverageHistogramSeries CreateSeries(const THistograms& histograms) {
        TAverageHistogramSeries series;
        for (const auto& hist : histograms) {
            series.Append(hist);
        }
        return series;
    }

    inline void CheckSeries(const TAverageHistogramSeries& series, const THistograms& histograms) {
        size_t index(0);
        for (const auto& hist : series) {
            UNIT_ASSERT(index < histograms.size());
            UNIT_ASSERT_EQUAL(hist, histograms[index]);
            ++index;
        }

        UNIT_ASSERT_EQUAL(index, histograms.size());
    }

    inline void TestIterator() {
        const auto histograms(CreateHistograms());
        const auto series(CreateSeries(histograms));
        CheckSeries(series, histograms);
    }

    inline void TestSerialization() {
        const auto histograms(CreateHistograms());
        auto series(CreateSeries(histograms));

        series.Finish();
        flatbuffers::FlatBufferBuilder builder;
        builder.Finish(series.ToProto(builder));

        const TAverageHistogramSeries restoredSeries(
            *flatbuffers::GetRoot<NCommon::TAverageHistogramSeries>(builder.GetBufferPointer())
        );
        CheckSeries(restoredSeries, histograms);
    }

    inline void TestCopy() {
        const auto histograms(CreateHistograms());
        const auto source(CreateSeries(histograms));
        const TAverageHistogramSeries target(source);
        CheckSeries(target, histograms);
    }
};

UNIT_TEST_SUITE_REGISTRATION(TAverageHistogramSeriesTest);
