#include <infra/netmon/switch_metrics.h>
#include <infra/netmon/probe_schedule_maintainer.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/topology/settings.h>
#include <infra/netmon/topology/topology_storage.h>

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

namespace NNetmon {
    class TSwitchSlaCounters::TImpl {
    public:
        TImpl(const TTopologyStorage& topologyStorage)
            : TopologyStorage(topologyStorage)
            , EventHub(TVoidEventHub::Make())
        {
            bool enabled = (TSettings::Get()->GetNocSlaSwitchMapCapacity() &&
                            !TSettings::Get()->GetProbeScheduleDcs().empty());
            if (enabled) {
                TopologySubscription.ConstructInPlace(topologyStorage.OnChanged().Subscribe([this]() {
                    UpdateSwitches();
                }));
                UpdateSwitches();
            }
        }

        void RegisterInterSwitchPackets(const TSwitch& switch_, bool incoming, ENetworkType network,
                                        ui64 success, ui64 failed, ui64 changed, double rtt) {
            InterSwitchMaintainer.RegisterProbe(switch_, incoming, network, success, failed, changed);
            InterSwitchRttMaintainer.RegisterProbe(switch_, incoming, network, rtt);
        }

        void RegisterLinkPollerPackets(const TSwitch& switch_, ENetworkType network,
                                       ui64 successCount, ui64 failCount) {
            LinkPollerMaintainer.RegisterProbe(switch_, network, successCount, failCount, /* changed = */ 0);
        }

        TInterSwitchCounterMapBox GetInterSwitchCounterMapBox() {
            return InterSwitchMaintainer.GetCounterMapBox();
        }

        TInterSwitchRttCounterMapBox GetInterSwitchRttCounterMapBox() {
            return InterSwitchRttMaintainer.GetCounterMapBox();
        }

        TLinkPollerCounterMapBox GetLinkPollerCounterMapBox() {
            return LinkPollerMaintainer.GetCounterMapBox();
        }

        const TVoidEventHub& OnMapChanged() const noexcept {
            return *EventHub;
        }

    private:
        template <class TCounterMap>
        class TMapMaintainer {
        public:
            TMapMaintainer()
                : CounterMapBox(TSettings::Get()->GetNocSlaSwitchMapCapacity())
            {
            }

            template <class... Args>
            void RegisterProbe(const TSwitch& switch_, Args&&... args) {
                auto counterMap = CounterMapBox.OwnRead();

                auto it = counterMap->find(switch_.GetReducedId());
                if (!it.IsEnd()) {
                    IncrementCounter(it->second, std::forward<Args>(args)...);
                }
            }

            void UpdateMap(const TVector<ui64>& newSwitches) {
                auto counterMap = CounterMapBox.OwnWrite();

                if (newSwitches.size() + counterMap->size() > TSettings::Get()->GetNocSlaSwitchMapCapacity()) {
                    ERROR_LOG << "NocSlaSwitchMapCapacity exceeded: "
                              << newSwitches.size() + counterMap->size() << " > "
                              << TSettings::Get()->GetNocSlaSwitchMapCapacity() << Endl;
                    return;
                }
                for (const auto& key : newSwitches) {
                    counterMap->emplace_noresize(std::piecewise_construct, std::forward_as_tuple(key), std::forward_as_tuple());
                }
            }

            typename TRWLockedBox<TCounterMap>::TReadOwnedBox GetCounterMapBox() {
                return CounterMapBox.OwnRead();
            }

        private:
            TRWLockedBox<TCounterMap> CounterMapBox;
        };

        TVector<ui64> GetNewSwitches() {
            TVector<ui64> newSwitches;
            const auto& knownQueuesFilter = TTopologySettings::Get()->GetKnownQueuesFilter();
            for (const auto& dcName : TSettings::Get()->GetProbeScheduleDcs()) {
                auto dc = TopologyStorage.FindDatacenter(dcName);
                if (!dc) {
                    ERROR_LOG << "Datacenter " << dcName << " not found in topology" << Endl;
                    continue;
                }

                if (!TTopologySettings::Get()->GetUsePodAsQueue()) {
                    for (const auto& queue : dc->GetLines()) {
                        if (!knownQueuesFilter.Check(queue->GetName())) {
                            continue;
                        }

                        for (const auto& switchRef : queue->GetSwitches()) {
                            auto id = switchRef->GetReducedId();
                            if (!Switches.contains(id)) {
                                Switches.emplace(id);
                                newSwitches.emplace_back(id);
                            }
                        }
                    }
                } else {
                    auto switches = TopologyStorage.GetTopologySelector()->GetSwitches(
                        TopologyStorage.DefaultExpressionId()
                    );

                    for (const auto& switchRef : switches) {
                        auto id = switchRef->GetReducedId();
                        if (*dc == switchRef->GetDatacenter() && !Switches.contains(id)) {
                            Switches.emplace(id);
                            newSwitches.emplace_back(id);
                        }
                    }
                }
            }
            return newSwitches;
        }

        void UpdateSwitches() {
            auto newSwitches = GetNewSwitches();
            if (!newSwitches.empty()) {
                INFO_LOG << "Switch set updated, " << newSwitches.size() << " new switches added, new size = " << Switches.size() << Endl;
                InterSwitchMaintainer.UpdateMap(newSwitches);
                InterSwitchRttMaintainer.UpdateMap(newSwitches);
                LinkPollerMaintainer.UpdateMap(newSwitches);
                EventHub->Notify();
            }
        }

        using TSwitchIdSet = THashSet<ui64>;
        TSwitchIdSet Switches;

        TMapMaintainer<TInterSwitchCounterMap> InterSwitchMaintainer;
        TMapMaintainer<TInterSwitchRttCounterMap> InterSwitchRttMaintainer;
        TMapMaintainer<TLinkPollerCounterMap> LinkPollerMaintainer;

        const TTopologyStorage& TopologyStorage;

        TMaybe<TVoidEventHub::TSubscriptionGuard> TopologySubscription;
        TVoidEventHub::TRef EventHub;
    };

    TSwitchSlaCounters::TSwitchSlaCounters(const TTopologyStorage& topologyStorage)
        : Impl(MakeHolder<TImpl>(topologyStorage))
    {
    }

    TSwitchSlaCounters::~TSwitchSlaCounters() {
    }

    void TSwitchSlaCounters::RegisterInterSwitchPackets(const TSwitch& switch_,
                                                        bool incoming,
                                                        ENetworkType network,
                                                        ui64 success,
                                                        ui64 failed,
                                                        ui64 changed,
                                                        double rtt) {
        return Impl->RegisterInterSwitchPackets(switch_, incoming, network, success, failed, changed, rtt);
    }

    void TSwitchSlaCounters::RegisterLinkPollerPackets(const TSwitch& switch_,
                                                       ENetworkType network,
                                                       ui64 successCount,
                                                       ui64 failCount) {
        return Impl->RegisterLinkPollerPackets(switch_, network, successCount, failCount);
    }

    TSwitchSlaCounters::TInterSwitchCounterMapBox TSwitchSlaCounters::GetInterSwitchCounterMapBox() {
        return Impl->GetInterSwitchCounterMapBox();
    }

    TSwitchSlaCounters::TInterSwitchRttCounterMapBox TSwitchSlaCounters::GetInterSwitchRttCounterMapBox() {
        return Impl->GetInterSwitchRttCounterMapBox();
    }

    TSwitchSlaCounters::TLinkPollerCounterMapBox TSwitchSlaCounters::GetLinkPollerCounterMapBox() {
        return Impl->GetLinkPollerCounterMapBox();
    }

    const TVoidEventHub& TSwitchSlaCounters::OnMapChanged() const noexcept {
        return Impl->OnMapChanged();
    }
}
