#include <infra/netmon/common_ut.h>
#include <infra/netmon/metrics.h>
#include <infra/netmon/statistics/histograms.h>
#include <infra/netmon/switch_metrics.h>
#include <infra/netmon/library/thread_pool.h>
#include <infra/netmon/metrics_updater.h>
#include <infra/netmon/probe_schedule_maintainer.h>
#include <infra/netmon/settings.h>
#include <infra/netmon/topology/common_ut.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/gtest.h>

#include <util/generic/queue.h>
#include <util/generic/xrange.h>

using namespace NNetmon;

#define STRINGIFY(x) #x
#define TO_STRING(x) STRINGIFY(x)
#define CHECK_METRICS_COMMENT "\nCheckMetrics called at " __FILE__ ":" TO_STRING(__LINE__)
#define CHECK_RTT_COMMENT "\nCheckRtt called at " __FILE__ ":" TO_STRING(__LINE__)

namespace {
    const size_t SLIDING_WINDOW_SLOTS = 45 / 5 + 1;
    const size_t COUNTERS_SIZE = 48;
    const size_t SHORT_COUNTERS_SIZE = 2;

    template <class TValue, class TMetricsMap>
    void CheckMetrics(const TMetricsMap& metricsMap,
                      const TTopology::TSwitchRef& switchRef,
                      const std::array<TValue, COUNTERS_SIZE>& expectedCounters,
                      const TStringBuf& comment) {
        UNIT_ASSERT_C(metricsMap->contains(switchRef->GetReducedId()), comment);
        auto metricsStruct = metricsMap->at(switchRef->GetReducedId());

        static_assert(COUNTERS_SIZE == sizeof(metricsStruct) / sizeof(TValue));

        // CS0 isn't used yet
        //UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS0.Success,   expectedCounters[ 0], comment);
        //UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS0.Failed,    expectedCounters[ 1], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS1.Success,   expectedCounters[ 2], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS1.Failed,    expectedCounters[ 3], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS2.Success,   expectedCounters[ 4], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS2.Failed,    expectedCounters[ 5], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS3.Success,   expectedCounters[ 6], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS3.Failed,    expectedCounters[ 7], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS4.Success,   expectedCounters[ 8], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS4.Failed,    expectedCounters[ 9], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.SparseBB.Success, expectedCounters[10], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.SparseBB.Failed,  expectedCounters[11], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.SparseFB.Success, expectedCounters[12], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.SparseFB.Failed,  expectedCounters[13], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.BB4.Success,   expectedCounters[14], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.BB4.Failed,    expectedCounters[15], comment);

        //UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS0.Success,   expectedCounters[16], comment);
        //UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS0.Failed,    expectedCounters[17], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS1.Success,   expectedCounters[18], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS1.Failed,    expectedCounters[19], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS2.Success,   expectedCounters[20], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS2.Failed,    expectedCounters[21], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS3.Success,   expectedCounters[22], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS3.Failed,    expectedCounters[23], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS4.Success,   expectedCounters[24], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS4.Failed,    expectedCounters[25], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.SparseBB.Success, expectedCounters[26], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.SparseBB.Failed,  expectedCounters[27], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.SparseFB.Success, expectedCounters[28], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.SparseFB.Failed,  expectedCounters[29], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.BB4.Success,   expectedCounters[30], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.BB4.Failed,    expectedCounters[31], comment);

        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS3.TosChanged,    expectedCounters[32], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Rx.CS4.TosChanged,    expectedCounters[33], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS3.TosChanged,    expectedCounters[34], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.Tx.CS4.TosChanged,    expectedCounters[35], comment);
    }

    template <class TValue, class TMetricsMap>
    void CheckMetricsShort(const TMetricsMap& metricsMap,
                           const TTopology::TSwitchRef& switchRef,
                           const std::array<TValue, SHORT_COUNTERS_SIZE>& expectedCounters,
                           const TStringBuf& comment) {
        UNIT_ASSERT_C(metricsMap->contains(switchRef->GetReducedId()), comment);
        auto metricsStruct = metricsMap->at(switchRef->GetReducedId());

        UNIT_ASSERT_EQUAL_C(metricsStruct.CS3.Success, expectedCounters[0], comment);
        UNIT_ASSERT_EQUAL_C(metricsStruct.CS3.Failed,  expectedCounters[1], comment);
    }

    void CheckRtt(TInterSwitchRttMetricsUpdater& rttUpdater,
                  const TTopology::TSwitchRef& switchRef,
                  const TTopologyStorage& topologyStorage,
                  double percentile,
                  double value,
                  const TStringBuf& comment) {
        auto interSwitchRttMetrics = rttUpdater.GetPastMetrics(TInstant(), TInstant::Now());

        for (const auto& item : interSwitchRttMetrics) {
            auto metricsMap = item.second;
            for (const auto& metric : *metricsMap) {
                auto metricSwitchRef = topologyStorage.FindSwitch(metric.first);
                if (switchRef->GetReducedId() == metricSwitchRef->GetReducedId()) {
                    auto counters = metric.second.Tx.CS4;
                    TSampleHistogram hist(*counters.Buckets);
                    TSampleHistogram::TBuckets cumulativeCount;
                    hist.FillCumulativeCount(cumulativeCount);

                    if (auto p = hist.GetPercentileValue(percentile, cumulativeCount)) {
                        UNIT_ASSERT_C(*p >= value && *p < value + 0.06, comment); // 0.05 step of the buckets and 0.01 delta for loss of precision
                        return;
                    }
                }
            }
        }
        UNIT_ASSERT(false);
    }
};

class TSwitchSlaCountersTest: public TTestBase {
    UNIT_TEST_SUITE(TSwitchSlaCountersTest);
    UNIT_TEST(TestInterSwitchCounters)
    UNIT_TEST(TestRttCounters)
    UNIT_TEST(TestLinkPollerCounters)
    UNIT_TEST_SUITE_END();

    void SetUp() override {
        TSettings::Get()->SetNocSlaMetricWindow(TDuration::Seconds(45));
        TSettings::Get()->SetNocSlaSwitchMapCapacity(10000);
        TSettings::Get()->SetProbeScheduleDcs({"vla"});
    }

private:
    inline void TestInterSwitchCounters() {
        const auto& topologyStorage = TGlobalTopology::GetTopologyStorage();
        TSwitchSlaCounters counters(topologyStorage);
        TInterSwitchMetricsUpdater updater(counters, false, false);

        updater.Update();

        auto switchRef = topologyStorage.FindSwitch("vla1-1s124");
        counters.RegisterInterSwitchPackets(*switchRef, true, FASTBONE_CS1, 1, 2, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, false, FASTBONE_CS2, 2, 1, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, true, BACKBONE_CS3, 1, 1, 1, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, true, BACKBONE6, 0, 2, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, false, FASTBONE_CS1, 2, 0, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, true, SPARSE_FASTBONE6, 1, 0, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, false, SPARSE_BACKBONE6, 0, 1, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, true, BACKBONE4, 3, 4, 0, 0.1);

        {
            auto counterMap = counters.GetInterSwitchCounterMapBox();
            CheckMetrics<TAtomic>(counterMap,
                                  switchRef,
                                  {{0, 0,
                                    1, 2, 0, 0,
                                    1, 3, 0, 0,
                                    0, 0, 1, 0,
                                    3, 4,

                                    0, 0,
                                    2, 0, 2, 1,
                                    0, 0, 0, 0,
                                    0, 1, 0, 0,
                                    0, 0,

                                    1, 0, 0, 0}},
                                  CHECK_METRICS_COMMENT);
        }

        updater.Update();

        counters.RegisterInterSwitchPackets(*switchRef, false, BACKBONE_CS4, 0, 1, 1, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, false, BACKBONE_CS4, 1, 1, 1, 0.5);

        {
            auto counterMap = counters.GetInterSwitchCounterMapBox();
            CheckMetrics<TAtomic>(counterMap,
                                  switchRef,
                                  {{0, 0,
                                    0, 0, 0, 0,
                                    0, 0, 0, 0,
                                    0, 0, 0, 0,
                                    0, 0,

                                    0, 0,
                                    0, 0, 0, 0,
                                    0, 0, 1, 2,
                                    0, 0, 0, 0,
                                    0, 0,

                                    0, 0, 0, 2}},
                                  CHECK_METRICS_COMMENT);
        }

        for (size_t i = 0; i < SLIDING_WINDOW_SLOTS; ++i) {
            updater.Update();
        }

        auto metrics = updater.GetPastMetrics(TInstant(), TInstant::Now());
        UNIT_ASSERT_EQUAL(metrics.size(), 2 + SLIDING_WINDOW_SLOTS);
        CheckMetrics<ui64>(metrics[0].second,
                           switchRef,
                           {{0, 0,
                             0, 0, 0, 0,
                             0, 0, 0, 0,
                             0, 0, 0, 0,
                             0, 0,

                             0, 0,
                             0, 0, 0, 0,
                             0, 0, 0, 0,
                             0, 0, 0, 0,
                             0, 0,

                             0, 0, 0, 0}},
                           CHECK_METRICS_COMMENT);
        CheckMetrics<ui64>(metrics[1].second,
                           switchRef,
                           {{0, 0,
                             1, 2, 0, 0,
                             1, 3, 0, 0,
                             0, 0, 1, 0,
                             3, 4,

                             0, 0,
                             2, 0, 2, 1,
                             0, 0, 0, 0,
                             0, 1, 0, 0,
                             0, 0,

                             1, 0, 0, 0}},
                           CHECK_METRICS_COMMENT);
        for (size_t i = 2; i <= SLIDING_WINDOW_SLOTS; ++i) {
            CheckMetrics<ui64>(metrics[i].second,
                               switchRef,
                               {{0, 0,
                                 1, 2, 0, 0,
                                 1, 3, 0, 0,
                                 0, 0, 1, 0,
                                 3, 4,

                                 0, 0,
                                 2, 0, 2, 1,
                                 0, 0, 1, 2,
                                 0, 1, 0, 0,
                                 0, 0,

                                 1, 0, 0, 2}},
                               CHECK_METRICS_COMMENT);
        }
        CheckMetrics<ui64>(metrics.back().second,
                           switchRef,
                           {{0, 0,
                             0, 0, 0, 0,
                             0, 0, 0, 0,
                             0, 0, 0, 0,
                             0, 0,

                             0, 0,
                             0, 0, 0, 0,
                             0, 0, 1, 2,
                             0, 0, 0, 0,
                             0, 0,

                             0, 0, 0, 2}},
                           CHECK_METRICS_COMMENT);
    }

    inline void TestRttCounters() {
        const auto& topologyStorage = TGlobalTopology::GetTopologyStorage();
        TSwitchSlaCounters counters(topologyStorage);
        TInterSwitchRttMetricsUpdater rttUpdater(counters, false, false);

        auto switchRef = topologyStorage.FindSwitch("vla1-1s124");
        counters.RegisterInterSwitchPackets(*switchRef, true, FASTBONE_CS1, 1, 2, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, false, FASTBONE_CS2, 2, 1, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, true, BACKBONE6, 1, 1, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, true, SPARSE_FASTBONE6, 1, 0, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, false, SPARSE_BACKBONE6, 0, 1, 0, 0.1);
        counters.RegisterInterSwitchPackets(*switchRef, false, BACKBONE_CS4, 0, 1, 0, 0.1);
        rttUpdater.Update();
        CheckRtt(rttUpdater, switchRef, topologyStorage, 1, 0.1, CHECK_RTT_COMMENT);

        counters.RegisterInterSwitchPackets(*switchRef, false, BACKBONE_CS4, 1, 1, 0, 0.5);
        rttUpdater.Update();
        CheckRtt(rttUpdater, switchRef, topologyStorage, 0.5, 0.1, CHECK_RTT_COMMENT);
        CheckRtt(rttUpdater, switchRef, topologyStorage, 1, 0.5, CHECK_RTT_COMMENT);

        counters.RegisterInterSwitchPackets(*switchRef, false, BACKBONE_CS4, 0, 0, 0, 0.3);
        rttUpdater.Update();
        CheckRtt(rttUpdater, switchRef, topologyStorage, 1, 0.5, CHECK_RTT_COMMENT);

        counters.RegisterInterSwitchPackets(*switchRef, false, BACKBONE_CS4, 0, 0, 0, 0.9);
        rttUpdater.Update();
        CheckRtt(rttUpdater, switchRef, topologyStorage, 0.5, 0.3, CHECK_RTT_COMMENT);
        CheckRtt(rttUpdater, switchRef, topologyStorage, 1, 0.9, CHECK_RTT_COMMENT);
    }

    inline void TestLinkPollerCounters() {
        const auto& topologyStorage = TGlobalTopology::GetTopologyStorage();
        TSwitchSlaCounters counters(topologyStorage);
        TLinkPollerMetricsUpdater updater(counters, false, false);
        updater.Update();

        auto switchRef = topologyStorage.FindSwitch("vla1-1s124");
        counters.RegisterLinkPollerPackets(*switchRef, FASTBONE_CS1, 1, 1);
        counters.RegisterLinkPollerPackets(*switchRef, FASTBONE_CS2, 1, 1);
        counters.RegisterLinkPollerPackets(*switchRef, BACKBONE_CS3, 1, 1);
        counters.RegisterLinkPollerPackets(*switchRef, BACKBONE_CS4, 1, 1);

        {
            auto counterMap = counters.GetLinkPollerCounterMapBox();
            UNIT_ASSERT(counterMap->contains(switchRef->GetReducedId()));
            auto metricsStruct = counterMap->at(switchRef->GetReducedId());
            UNIT_ASSERT_EQUAL(metricsStruct.CS1.Success, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS1.Failed, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS2.Success, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS2.Failed, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS3.Success, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS3.Failed, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS4.Success, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS4.Failed, 1);
        }

        updater.Update();
        auto metrics = updater.GetPastMetrics(TInstant(), TInstant::Now());
        UNIT_ASSERT_EQUAL(metrics.size(), 2);

        {
            auto oldMetricsMap = metrics.front().second;
            UNIT_ASSERT(oldMetricsMap->contains(switchRef->GetReducedId()));
            auto metricsStruct = oldMetricsMap->at(switchRef->GetReducedId());
            UNIT_ASSERT_EQUAL(metricsStruct.CS1.Success, 0);
            UNIT_ASSERT_EQUAL(metricsStruct.CS1.Failed, 0);
            UNIT_ASSERT_EQUAL(metricsStruct.CS2.Success, 0);
            UNIT_ASSERT_EQUAL(metricsStruct.CS2.Failed, 0);
            UNIT_ASSERT_EQUAL(metricsStruct.CS3.Success, 0);
            UNIT_ASSERT_EQUAL(metricsStruct.CS3.Failed, 0);
            UNIT_ASSERT_EQUAL(metricsStruct.CS4.Success, 0);
            UNIT_ASSERT_EQUAL(metricsStruct.CS4.Failed, 0);
        }

        {
            auto newMetricsMap = metrics.back().second;
            UNIT_ASSERT(newMetricsMap->contains(switchRef->GetReducedId()));
            auto metricsStruct = newMetricsMap->at(switchRef->GetReducedId());
            UNIT_ASSERT_EQUAL(metricsStruct.CS1.Success, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS1.Failed, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS2.Success, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS2.Failed, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS3.Success, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS3.Failed, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS4.Success, 1);
            UNIT_ASSERT_EQUAL(metricsStruct.CS4.Failed, 1);
        }
    }
};

UNIT_TEST_SUITE_REGISTRATION(TSwitchSlaCountersTest);
