#include <infra/netmon/tasks/enqueued_task_index.h>
#include <infra/netmon/library/api_client_helpers.h>

#include <library/cpp/containers/intrusive_hash/intrhash.h>
#include <library/cpp/containers/intrusive_rb_tree/rb_tree.h>

namespace NNetmon {
    TEnqueuedTask::TEnqueuedTask(const NClient::TEnqueuedTask& task)
    {
        CreateGuid(&Key_);
        Task_.CopyFrom(task);
        Task_.SetKey(GetGuidAsString(Key_));
    }

    const TEnqueuedTask::TKey& TEnqueuedTask::Key() const noexcept {
        return Key_;
    }

    TInstant TEnqueuedTask::Deadline() const noexcept {
        return TInstant::MicroSeconds(Task_.GetDeadline());
    }

    void TEnqueuedTask::Dump(IOutputStream& stream) const noexcept {
        ::Save(&stream, Task_);
    }

    class TEnqueuedTaskIndex::TImpl {
    private:
        struct THostComparator {
            template <class T>
            static inline bool Compare(const T& lhs, const T& rhs) noexcept {
                return (
                    lhs.Host() < rhs.Host()
                    || (
                        lhs.Host() == rhs.Host()
                        && lhs.Deadline() < rhs.Deadline()
                    )
                );
            }

            template <class T>
            static inline bool Compare(const T& lhs, const TString& rhs) noexcept {
                return lhs.Host() < rhs;
            }
        };

        class TTaskContainer: public TIntrusiveHashItem<TTaskContainer>,
                              public TRbTreeItem<TTaskContainer, THostComparator>  {
        public:
            using TRef = THolder<TTaskContainer>;

            struct TKeyOps: public ::TCommonIntrHashOps {
                static inline const IAgentTask::TKey& ExtractKey(const TTaskContainer& obj) noexcept {
                    return obj.Key;
                }
            };
            using TKeyMap = TIntrusiveHashWithAutoDelete<TTaskContainer, TKeyOps>;

            using THostTree = TRbTree<TTaskContainer, THostComparator>;

            inline TTaskContainer(const IAgentTask::TKey& key, IInputStream& stream)
                : Key(key)
            {
                ::Load(&stream, Task);
                if (Task.GetHost().empty()) {
                    ythrow yexception() << "no host specified";
                }
            }

            inline void CopyInto(NClient::TEnqueuedTask& task) const noexcept {
                task.CopyFrom(Task);
            }

            inline void Link(TImpl& impl) noexcept {
                impl.HostTree.Insert(this);
            }

            inline void Unlink(TImpl&) noexcept {
                TRbTreeItem<TTaskContainer, THostComparator>::UnLink();
            }

            inline const TString& Host() const noexcept {
                return Task.GetHost();
            }

            inline ui64 Deadline() const noexcept {
                return Task.GetDeadline();
            }

        private:
            const TGUID Key;
            NClient::TEnqueuedTask Task;
        };

    public:
        inline void Load(const IAgentTask::TKey& key, IInputStream& stream) noexcept {
            Y_VERIFY(Tasks.Find(key) == Tasks.End());

            TTaskContainer::TRef container;
            try {
                container.Reset(MakeHolder<TTaskContainer>(key, stream));
            } catch(...) {
                ERROR_LOG << "Can't load task from storage: " << CurrentExceptionMessage() << Endl;
                return;
            }
            DEBUG_LOG << "Schedule enqueued task '" << GetGuidAsString(key) << "' to host '" << container->Host() << "'" << Endl;

            container->Link(*this);
            Tasks.Push(container.Release());
        }

        inline void Delete(const IAgentTask::TKey& key) noexcept {
            auto it(Tasks.Find(key));
            if (it != Tasks.End()) {
                TTaskContainer::TRef container(Tasks.Pop(it));
                DEBUG_LOG << "Delete enqueued task '" << GetGuidAsString(key) << "' from host '" << container->Host() << "'" << Endl;
                container->Unlink(*this);
            }
        }

        inline TVector<NClient::TEnqueuedTask> FindByHost(const TString& hostname) const noexcept {
            TVector<NClient::TEnqueuedTask> result;
            TTaskContainer::THostTree::TConstIterator it(HostTree.LowerBound(hostname));
            for (; it != HostTree.End(); ++it) {
                const TTaskContainer& container(*it);
                if (container.Host() != hostname) {
                    break;
                } else {
                    container.CopyInto(result.emplace_back());
                }
            }
            return result;
        }

    private:
        TTaskContainer::TKeyMap Tasks;
        TTaskContainer::THostTree HostTree;
    };

    TEnqueuedTaskIndex::TEnqueuedTaskIndex()
        : Impl(MakeHolder<TImpl>())
    {
    }

    TEnqueuedTaskIndex::~TEnqueuedTaskIndex() {
    }

    void TEnqueuedTaskIndex::Load(const IAgentTask::TKey& key, IInputStream& stream) noexcept {
        Impl.Own()->Load(key, stream);
    }

    void TEnqueuedTaskIndex::Delete(const IAgentTask::TKey& key) noexcept {
        Impl.Own()->Delete(key);
    }

    TVector<NClient::TEnqueuedTask> TEnqueuedTaskIndex::FindByHost(const TString& hostname) const noexcept {
        return Impl.Own()->FindByHost(hostname);
    }
}
