#include <infra/netmon/tasks/finished_task_index.h>
#include <infra/netmon/library/api_client_helpers.h>

#include <library/cpp/containers/intrusive_hash/intrhash.h>

#include <util/generic/hash.h>
#include <util/generic/hash_set.h>

namespace NNetmon {
    TFinishedTask::TFinishedTask(const NClient::TTaskResult& result)
    {
        CreateGuid(&Key_);
        Result_.CopyFrom(result);
    }

    TFinishedTask::TKey TFinishedTask::ParentKey() const noexcept {
        TKey key;
        if (!Result_.GetParentKey().empty()) {
            GetGuid(Result_.GetParentKey(), key);
        }
        return key;
    }

    const TFinishedTask::TKey& TFinishedTask::Key() const noexcept {
        return Key_;
    }

    TInstant TFinishedTask::Deadline() const noexcept {
        return TInstant::MicroSeconds(Result_.GetDeadline());
    }

    void TFinishedTask::Dump(IOutputStream& stream) const noexcept {
        ::Save(&stream, Result_);
    }

    class TFinishedTaskIndex::TImpl {
    private:
        class TTaskContainer: public TIntrusiveHashItem<TTaskContainer> {
        public:
            using TRef = THolder<TTaskContainer>;

            struct TKeyOps: public ::TCommonIntrHashOps {
                static inline const IAgentTask::TKey& ExtractKey(const TTaskContainer& obj) noexcept {
                    return obj.Key;
                }
            };
            using TKeyMap = TIntrusiveHashWithAutoDelete<TTaskContainer, TKeyOps>;

            struct TParentKeyHashFunctor {
                inline size_t operator()(const TTaskContainer* x) const noexcept {
                    return operator()(x->ParentKey);
                }
                inline size_t operator()(const IAgentTask::TKey& x) const noexcept {
                    return Default<THash<IAgentTask::TKey>>()(x);
                }
            };
            struct TParentKeyEqualFunctor {
                inline bool operator()(const TTaskContainer* lhs, const TTaskContainer* rhs) const noexcept {
                    return lhs->Key == rhs->Key;
                }
                inline bool operator()(const TTaskContainer* lhs, const IAgentTask::TKey& rhs) const noexcept {
                    return lhs->ParentKey == rhs;
                }
            };
            using TParentKeySet = THashMultiSet<TTaskContainer*, TParentKeyHashFunctor, TParentKeyEqualFunctor>;

            struct TVersionHashFunctor {
                inline size_t operator()(const TTaskContainer* x) const noexcept {
                    return ComputeHash(x->Result.GetVersion().GetHost());
                }
                inline size_t operator()(const TString& x) const noexcept {
                    return ComputeHash(x);
                }
            };
            struct TVersionEqualFunctor {
                inline bool operator()(const TTaskContainer* lhs, const TTaskContainer* rhs) const noexcept {
                    return lhs->Result.GetVersion().GetHost() == rhs->Result.GetVersion().GetHost();
                }
                inline bool operator()(const TTaskContainer* lhs, const TString& rhs) const noexcept {
                    return lhs->Result.GetVersion().GetHost() == rhs;
                }
            };
            using TVersionSet = THashSet<TTaskContainer*, TVersionHashFunctor, TVersionEqualFunctor>;

            inline TTaskContainer(const IAgentTask::TKey& key, IInputStream& stream)
                : Key(key)
            {
                ::Load(&stream, Result);
                if (!Result.GetParentKey().empty() && !GetGuid(Result.GetParentKey(), ParentKey)) {
                    ythrow yexception() << "wrong parent key given";
                }
            }

            inline void CopyInto(NClient::TTaskResult& result) const noexcept {
                result.CopyFrom(Result);
            }

            inline void Link(TImpl& impl) noexcept {
                if (ParentKey) {
                    impl.ParentKeyIndex.insert(this);
                }
                if (Result.HasVersion()) {
                    auto it(impl.VersionIndex.find(this));
                    if (it.IsEnd()) {
                        impl.VersionIndex.insert(this);
                    } else if ((*it)->Result.GetGenerated() < Result.GetGenerated()) {
                        impl.VersionIndex.erase(this);
                        impl.VersionIndex.insert(this);
                    }
                }
            }

            inline void Unlink(TImpl& impl) noexcept {
                if (ParentKey) {
                    impl.ParentKeyIndex.erase(this);
                }
                if (Result.HasVersion()) {
                    auto it(impl.VersionIndex.find(this));
                    if (!it.IsEnd() && *it == this) {
                        impl.VersionIndex.erase(it);
                    }
                }
            }

            inline const NClient::TTaskResult& GetResult() const noexcept {
                return Result;
            }

            inline const IAgentTask::TKey& GetKey() const noexcept {
                return Key;
            }

            inline const IAgentTask::TKey& GetParentKey() const noexcept {
                return ParentKey;
            }

        private:
            const IAgentTask::TKey Key;
            IAgentTask::TKey ParentKey;
            NClient::TTaskResult Result;
        };

    public:
        inline void Load(const IAgentTask::TKey& key, IInputStream& stream) noexcept {
            Y_VERIFY(Tasks.Find(key) == Tasks.End());

            TTaskContainer::TRef container;
            try {
                container.Reset(MakeHolder<TTaskContainer>(key, stream));
            } catch(...) {
                ERROR_LOG << "Can't load task from storage: " << CurrentExceptionMessage() << Endl;
                return;
            }

            if (container->GetResult().HasVersion()) {
                DEBUG_LOG << "Loading task '" << GetGuidAsString(container->GetKey())
                          << "' as telemetry for host '" << container->GetResult().GetVersion().GetHost() << "'" << Endl;
            } else {
                DEBUG_LOG << "Loading task '" << GetGuidAsString(container->GetKey())
                          << "' as result for '" << GetGuidAsString(container->GetParentKey()) << "'" << Endl;
            }
            container->Link(*this);
            Tasks.Push(container.Release());
        }

        inline void Delete(const IAgentTask::TKey& key) noexcept {
            auto it(Tasks.Find(key));
            if (it != Tasks.End()) {
                TTaskContainer::TRef container(Tasks.Pop(it));
                DEBUG_LOG << "Deleting task '" << GetGuidAsString(container->GetKey()) << "'" << Endl;
                container->Unlink(*this);
            }
        }

        inline const TTaskContainer* FindByParentKey(const IAgentTask::TKey& key) const noexcept {
            // return the most recent task with same parent key
            const TTaskContainer* container = nullptr;
            const auto range(ParentKeyIndex.equal_range(key));
            for (auto it(range.first); it != range.second; ++it) {
                if (container == nullptr || container->GetResult().GetGenerated() < (*it)->GetResult().GetGenerated()) {
                    container = *it;
                }
            }
            return container;
        }

        inline bool FindByParentKey(const IAgentTask::TKey& key, NClient::TTaskResult& result) const noexcept {
            const auto* container(FindByParentKey(key));
            if (container != nullptr) {
                container->CopyInto(result);
                return true;
            } else {
                return false;
            }
        }

        TMaybe<IAgentTask::TKey> KeyByParentKey(const IAgentTask::TKey& key) const noexcept {
            const auto* container(FindByParentKey(key));
            if (container != nullptr) {
                return container->GetKey();
            } else {
                return {};
            }
        }

        inline THashMap<TString, int> VersionHistogram() const noexcept {
            THashMap<TString, int> hist;
            for (const auto& task : VersionIndex) {
                hist[task->GetResult().GetVersion().GetVersion()]++;
            }
            return hist;
        }

        inline TString GetVersion(const TString& hostname) const noexcept {
            auto it(VersionIndex.find(hostname));
            return !it.IsEnd() ? (*it)->GetResult().GetVersion().GetVersion() : "";
        }

        TVector<TString> FindByVersion(const TString& version) const noexcept {
            TVector<TString> result;
            for (const auto& task : VersionIndex) {
                const auto& info(task->GetResult().GetVersion());
                if (info.GetVersion() == version) {
                    result.emplace_back(info.GetHost());
                }
            }
            return result;
        }

        template <typename F>
        void ForEachVersion(F&& func) const {
            for (const auto& task : VersionIndex) {
                func(task->GetResult().GetVersion(), TInstant::MicroSeconds(task->GetResult().GetGenerated()));
            }
        }

    private:
        TTaskContainer::TKeyMap Tasks;
        TTaskContainer::TParentKeySet ParentKeyIndex;
        TTaskContainer::TVersionSet VersionIndex;
    };

    TFinishedTaskIndex::TFinishedTaskIndex()
        : Impl(MakeHolder<TImpl>())
    {
    }

    TFinishedTaskIndex::~TFinishedTaskIndex() {
    }

    void TFinishedTaskIndex::Load(const IAgentTask::TKey& key, IInputStream& stream) noexcept {
        Impl.Own()->Load(key, stream);
    }

    void TFinishedTaskIndex::Delete(const IAgentTask::TKey& key) noexcept {
        Impl.Own()->Delete(key);
    }

    bool TFinishedTaskIndex::FindByParentKey(const IAgentTask::TKey& key, NClient::TTaskResult& result) const noexcept {
        return Impl.Own()->FindByParentKey(key, result);
    }

    TMaybe<IAgentTask::TKey> TFinishedTaskIndex::KeyByParentKey(const IAgentTask::TKey& key) const noexcept {
        return Impl.Own()->KeyByParentKey(key);
    }

    THashMap<TString, int> TFinishedTaskIndex::VersionHistogram() const noexcept {
        return Impl.Own()->VersionHistogram();
    }

    TString TFinishedTaskIndex::GetVersion(const TString& hostname) const noexcept {
        return Impl.Own()->GetVersion(hostname);
    }

    TVector<TString> TFinishedTaskIndex::FindByVersion(const TString& version) const noexcept {
        return Impl.Own()->FindByVersion(version);
    }

    void TFinishedTaskIndex::ForEachVersion(std::function<void(const NClient::TVersionResult&, TInstant)> func) const {
        Impl.Own()->ForEachVersion(std::move(func));
    }
}
