#include <infra/netmon/tasks/finished_task_index.h>
#include <infra/netmon/library/api_client_helpers.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/gtest.h>

using namespace NNetmon;

class TFinishedTaskIndexTest: public TTestBase {
    UNIT_TEST_SUITE(TFinishedTaskIndexTest);
    UNIT_TEST(TestFindByParentKey)
    UNIT_TEST(TestVersionHistogram)
    UNIT_TEST_SUITE_END();

private:
    inline void TestFindByParentKey() {
        IAgentTask::TKey key, parentKey;
        CreateGuid(&key);
        CreateGuid(&parentKey);

        NClient::TTaskResult reference;
        reference.SetParentKey(GetGuidAsString(parentKey));
        Load(key, reference);

        NClient::TTaskResult copy;
        UNIT_ASSERT(!Index.FindByParentKey(key, copy));
        UNIT_ASSERT(Index.FindByParentKey(parentKey, copy));
        UNIT_ASSERT_EQUAL(copy.GetParentKey(), GetGuidAsString(parentKey));

        Index.Delete(key);
        UNIT_ASSERT(!Index.FindByParentKey(parentKey, copy));
    }

    inline void TestVersionHistogram() {
        IAgentTask::TKey firstKey, secondKey, thirdKey;
        CreateGuid(&firstKey);
        CreateGuid(&secondKey);
        CreateGuid(&thirdKey);

        NClient::TTaskResult firstResult, secondResult, thirdResult;

        firstResult.MutableVersion()->SetHost("beta");
        firstResult.MutableVersion()->SetVersion("0.1.0");
        firstResult.SetGenerated(1);

        secondResult.MutableVersion()->SetHost("gamma");
        secondResult.MutableVersion()->SetVersion("0.2.0");
        secondResult.SetGenerated(3);

        thirdResult.MutableVersion()->SetHost("gamma");
        thirdResult.MutableVersion()->SetVersion("0.1.1");
        thirdResult.SetGenerated(2);

        Load(firstKey, firstResult);
        Load(secondKey, secondResult);
        Load(thirdKey, thirdResult);

        auto histogram(Index.VersionHistogram());
        UNIT_ASSERT_EQUAL(histogram.size(), 2);
        UNIT_ASSERT_EQUAL(histogram["0.1.0"], 1);
        UNIT_ASSERT_EQUAL(histogram["0.2.0"], 1);

        UNIT_ASSERT_EQUAL(Index.GetVersion("gamma"), "0.2.0");
        UNIT_ASSERT_EQUAL(Index.GetVersion("notfound"), "");
        UNIT_ASSERT_EQUAL(Index.FindByVersion("0.2.0").size(), 1);

        Index.Delete(thirdKey);
        Index.Delete(secondKey);
        Index.Delete(firstKey);
        UNIT_ASSERT(Index.VersionHistogram().empty());
        UNIT_ASSERT_EQUAL(Index.GetVersion("gamma"), "");
        UNIT_ASSERT_EQUAL(Index.FindByVersion("0.2.0").size(), 0);
    }

    inline void Load(const IAgentTask::TKey& key, const NClient::TTaskResult& obj) {
        TBufferStream stream;
        ::Save(&stream, obj);
        Index.Load(key, stream);
    }

    TFinishedTaskIndex Index;
};

UNIT_TEST_SUITE_REGISTRATION(TFinishedTaskIndexTest);
