#include <infra/netmon/tasks/storage.h>
#include <infra/netmon/library/api_client_helpers.h>
#include <infra/netmon/library/fences.h>
#include <infra/netmon/library/zookeeper.h>
#include <infra/netmon/library/settings.h>

#include <infra/netmon/idl/task_queue.fbs.h>

#include <library/cpp/containers/intrusive_hash/intrhash.h>

#include <util/generic/set.h>
#include <util/generic/xrange.h>
#include <util/folder/path.h>
#include <util/string/builder.h>

namespace NNetmon {
    namespace {
        const ui64 SHARD_COUNT = 1024;
        const TAtomic IN_FLIGHT_LIMIT = 16;

        inline TString Dump(const IAgentTask& task) {
            TString body;
            TStringOutput stream(body);
            task.Dump(stream);
            stream.Finish();
            return body;
        }

        inline TString Dump(const flatbuffers::FlatBufferBuilder& builder) {
            TString body;
            TStringOutput stream(body);
            WriteFlatBuffer(&stream, builder);
            stream.Finish();
            return body;
        }

        inline TString Encapsulate(const IAgentTask& task) {
            TString encapsulatedBody(Dump(task));
            flatbuffers::FlatBufferBuilder builder;
            builder.Finish(NTaskQueue::CreateTDumpedTask(
                builder,
                builder.CreateString(encapsulatedBody.data(), encapsulatedBody.size()),
                task.Deadline().Seconds()
            ));
            return Dump(builder);
        }

        inline TMaybe<IAgentTask::TKey> ExtractKey(const TString& path) noexcept {
            TFsPath fs(path);
            IAgentTask::TKey key;
            if (GetGuid(fs.GetName(), key)) {
                return key;
            } else {
                return Nothing();
            }
        }

        class TTaskHeader: public TIntrusiveHashItem<TTaskHeader>,
                           public TAvlTreeItem<TTaskHeader, TCompareUsingDeadline>,
                           public TIntrusiveListItem<TTaskHeader> {
        public:
            using TRef = THolder<TTaskHeader>;

            struct TOps: public ::TCommonIntrHashOps {
                static inline const IAgentTask::TKey& ExtractKey(const TTaskHeader& obj) noexcept {
                    return obj.Key;
                }
            };
            using TMapType = TIntrusiveHash<TTaskHeader, TOps>;
            using TTree = TAvlTree<TTaskHeader, TCompareUsingDeadline>;
            using TListType = TIntrusiveListWithAutoDelete<TTaskHeader, TDelete>;

            TTaskHeader(const TGUID& key, const TString& path, const TInstant& deadline)
                : Key(key)
                , Path(path)
                , Deadline(deadline)
            {
            }

            const IAgentTask::TKey& GetKey() const noexcept {
                return Key;
            }
            const TString& GetPath() const noexcept {
                return Path;
            }
            const TInstant& GetDeadline() const noexcept {
                return Deadline;
            }

        private:
            const TGUID Key;
            const TString Path;
            const TInstant Deadline;
        };
    }

    class TAgentTaskStorage::TImpl: public NAsyncZookeeper::IWatcher, public TScheduledTask, public TAtomicRefCount<TImpl> {
    public:
        friend TAgentTaskMover;

        using TRef = TIntrusivePtr<TImpl>;

        inline TImpl(const TString& prefix, IAgentTaskWatcher& watcher)
            : TScheduledTask(TDuration::Seconds(5))
            , Prefix(prefix)
            , ZookeeperClient(*NAsyncZookeeper::TClient::Get())
            , Watcher(watcher)
        {
            for (const auto idx : xrange(SHARD_COUNT)) {
                ZookeeperClient.AddWatcher(Join(idx), this);
            }
        }

        inline ~TImpl() {
            for (const auto idx : xrange(SHARD_COUNT)) {
                ZookeeperClient.RemoveWatcher(Join(idx), this);
            }
        }

    private:
        class TInitializer: public TNonCopyable, public TAtomicRefCount<TInitializer> {
        public:
            using TRef = TIntrusivePtr<TInitializer>;

            inline TInitializer(TImpl::TRef impl)
                : Impl(impl)
            {
            }

            NThreading::TFuture<bool> Execute() noexcept {
                WARNING_LOG << "Task storage not initialized, let's initialize it" << Endl;

                Fence.Retain(Impl->ZookeeperClient.CreateRecursive(Impl->Prefix).Subscribe(
                    Y_MEMBER_REF_TO_CB(&TInitializer::OnPrefixCreated, this)));

                TRef self(this);
                return Fence.Wait().Apply([self](const NThreading::TFuture<void>&) {
                    return AtomicSwap(&self->ShardLoaded, 0) == SHARD_COUNT;
                });
            }

        private:
            void OnPrefixCreated(const NAsyncZookeeper::TVoidResult& result) noexcept {
                if (result.ResultCode() == NAsyncZookeeper::OK) {
                    DEBUG_LOG << "Prefix '" << Impl->Prefix << "' created" << Endl;
                    for (const auto idx : xrange(SHARD_COUNT)) {
                        const auto path(Impl->Join(idx));
                        Fence.Retain(Impl->ZookeeperClient.Create(path, "").Subscribe(
                            Y_MEMBER_REF_TO_CB(&TInitializer::OnShardCreated, this, path)));
                    }
                } else {
                    ERROR_LOG << "Can't create prefix '" << Impl->Prefix << "'" << Endl;
                }
            }

            void OnShardCreated(const NAsyncZookeeper::TPathResult& result, const TString& path) noexcept {
                if (result.ResultCode() != NAsyncZookeeper::OK && result.ResultCode() != NAsyncZookeeper::NODE_EXISTS) {
                    ERROR_LOG << "Can't create shard '" << path << "'" << Endl;
                } else {
                    Fence.Retain(Impl->ZookeeperClient.GetChildrenData(path, true).Subscribe(
                        Y_MEMBER_REF_TO_CB(&TInitializer::OnShardData, this, path)));
                }
            }

            void OnShardData(const NAsyncZookeeper::TChildrenDataResult& result, const TString& path) noexcept {
                if (result.ResultCode() != NAsyncZookeeper::OK) {
                    ERROR_LOG << "Can't load shard '" << path << "'" << Endl;
                } else {
                    TSet<TString> actualChildren;
                    for (const auto& pair : result.Result()) {
                        actualChildren.insert(pair.first);
                        auto key(ExtractKey(pair.first));
                        if (key.Defined()) {
                            Impl->CreateNewNode(key.GetRef(), NAsyncZookeeper::Join(path, pair.first), pair.second.Data());
                        }
                    }
                    Impl->Tasks.Own()->RemoveObsolete(path, actualChildren, Impl->Watcher);

                    if (result.Result().size()) {
                        DEBUG_LOG << "Finish loading shard '" << path << "' nodes, " << result.Result().size() << " found" << Endl;
                    }
                    AtomicIncrement(ShardLoaded);
                }
            }

            TImpl::TRef Impl;
            TFutureFence Fence;
            TAtomic ShardLoaded = 0;
        };

        class TSynchronizer: public TNonCopyable, public TAtomicRefCount<TSynchronizer> {
        public:
            using TRef = TIntrusivePtr<TSynchronizer>;

            inline TSynchronizer(TImpl::TRef impl)
                : Impl(impl)
            {
            }

            NThreading::TFuture<void> Execute() noexcept {
                {
                    auto box(Impl->ShardsToReload.Own());
                    for (const auto& path : *box) {
                        Fence.Retain(Impl->ZookeeperClient.GetChildren(path, true).Subscribe(
                            Y_MEMBER_REF_TO_CB(&TSynchronizer::OnShardChildren, this, path)));
                    }
                    box->clear();
                }

                {
                    auto box(Impl->Tasks.Own());
                    box->Cleanup(Impl->Watcher);
                    while (!box->GetTombstones().Empty()) {
                        TTaskHeader::TRef task(box->GetTombstones().Front());
                        task->TIntrusiveListItem::Unlink();
                        auto unguard = Unguard(box.GetGuard()); // because callback can be executed earlier
                        Fence.Retain(Impl->ZookeeperClient.Delete(task->GetPath()).Subscribe(
                            Y_MEMBER_REF_TO_CB(&TSynchronizer::OnNodeDeleted, this, task.Release())));
                    }
                }

                return Fence.Wait();
            }

        private:
            void OnShardChildren(const NAsyncZookeeper::TChildrenResult& result, const TString& prefix) noexcept {
                if (result.Empty()) {
                    Impl->ForceInitialization();
                    DEBUG_LOG << "Can't load shard '" << prefix << "', because of " << result.ResultCode() << Endl;
                    return;
                }

                TSet<TString> actualChildren;
                auto box(Impl->Tasks.Own());
                for (const auto& name : result.Result()) {
                    actualChildren.insert(name);
                    TString fullPath(NAsyncZookeeper::Join(prefix, name));
                    auto key(ExtractKey(fullPath));
                    if (key.Defined() && !box->Exists(key.GetRef())) {
                        auto unguard = Unguard(box.GetGuard()); // because callback can be executed earlier
                        DEBUG_LOG << "Let's load node '" << fullPath << "'" << Endl;
                        Fence.Retain(Impl->ZookeeperClient.GetData(fullPath).Subscribe(
                            Y_MEMBER_REF_TO_CB(&TSynchronizer::OnNodeData, this, key.GetRef(), fullPath)));
                    }
                }

                box->RemoveObsolete(prefix, actualChildren, Impl->Watcher);

                DEBUG_LOG << "Shard '" << prefix << "' reloaded" << Endl;
            }

            void OnNodeData(const NAsyncZookeeper::TDataResult& result, const IAgentTask::TKey& key, const TString& path) noexcept {
                if (!result.Empty()) {
                    DEBUG_LOG << "Node '" << path << "' loaded" << Endl;
                    Impl->CreateNewNode(key, path, result.Result().Data());
                } else {
                    DEBUG_LOG << "Can't load node '" << path << "', because of " << result.ResultCode() << Endl;
                }
            }

            void OnNodeDeleted(const NAsyncZookeeper::TVoidResult& result, TTaskHeader* task_) noexcept {
                TTaskHeader::TRef task(task_);
                if (result.ResultCode() != NAsyncZookeeper::OK && result.ResultCode() != NAsyncZookeeper::NODE_NOT_EXISTS) {
                    DEBUG_LOG << "Can't delete node '" << task->GetPath() << "', because of " << result.ResultCode() << Endl;
                    Impl->Tasks.Own()->GetTombstones().PushBack(task.Release());
                } else {
                    DEBUG_LOG << "Node '" << task->GetPath() << "' deleted" << Endl;
                }
            }

            TImpl::TRef Impl;
            TFutureFence Fence;
        };

    public:
        TThreadPool::TFuture Run() override {
            auto synchronizer(MakeIntrusive<TSynchronizer>(this));
            if (!AtomicGet(Initialized)) {
                auto initializer(MakeIntrusive<TInitializer>(this));
                Fence.Retain(initializer->Execute().Subscribe([this, synchronizer](const NThreading::TFuture<bool>& future) {
                    if (future.GetValue()) {
                        AtomicSet(Initialized, true);
                        Fence.Retain(synchronizer->Execute());
                        INFO_LOG << "All tasks from '" << Prefix << "' loaded" << Endl;
                    } else {
                        ERROR_LOG << "Can't load tasks from '" << Prefix << "'" << Endl;
                    }
                }));
            } else {
                Fence.Retain(synchronizer->Execute());
            }
            return Fence.Wait();
        }

        void OnEvent(const NAsyncZookeeper::TEvent& event) override {
            if (event.Path.StartsWith(Prefix) && event.EventType == NAsyncZookeeper::EEventType::ET_NODE_CHILDREN_CHANGED) {
                DEBUG_LOG << "Shard '" << event.Path << "' changed, schedule reloading" << Endl;
                ShardsToReload.Own()->insert(event.Path);
            }
        }

        void OnExpired() override {
            WARNING_LOG << "Connection to zookeeper expired, let's schedule task storage initialization" << Endl;
            ForceInitialization();
        }

        inline TString KeyToPath(const IAgentTask::TKey& key) const noexcept {
            const auto shardIndex = Default<THash<IAgentTask::TKey>>()(key) % SHARD_COUNT;
            return Join(shardIndex, GetGuidAsString(key));
        }

        inline NThreading::TFuture<TAgentTaskStorage::EResultCode> DumpNewNode(const IAgentTask::TKey& key, const TString& body) noexcept {
            if (AtomicGet(InFlightRequests) > IN_FLIGHT_LIMIT) {
                return NThreading::MakeFuture<TAgentTaskStorage::EResultCode>(TAgentTaskStorage::TOO_MANY_REQUESTS);
            } else {
                AtomicIncrement(InFlightRequests);
                const auto path(KeyToPath(key));
                return ZookeeperClient.Create(path, body).Apply(
                    Y_MEMBER_REF_TO_CB(&TImpl::OnNodeCreated, this, key, path, body));
            }
        }

    private:
        template <typename... Args>
        inline TString Join(Args&&... paths) const noexcept {
            return NAsyncZookeeper::Join(Prefix, std::forward<Args>(paths)...);
        }

        void CreateNewNode(const IAgentTask::TKey& key, const TString& path, const TString& body) noexcept {
            TFlatObject<NTaskQueue::TDumpedTask> task;
            ReadFlatBuffer(body, task);
            if (!task.Verify()) {
                return;
            }

            auto box(Tasks.Own());
            if (box->Create(MakeHolder<TTaskHeader>(key, path, TInstant::Seconds(task->Deadline())))) {
                TMemoryInput stream(task->Body()->data(), task->Body()->size());
                Watcher.Load(key, stream);
                DEBUG_LOG << "Node '" << path << "' created" << Endl;
            }
        }

        TAgentTaskStorage::EResultCode OnNodeCreated(const NAsyncZookeeper::TPathResult& result,
                                                     const IAgentTask::TKey& key, const TString& path, const TString& body) noexcept {
            AtomicDecrement(InFlightRequests);
            if (result.ResultCode() == NAsyncZookeeper::OK) {
                CreateNewNode(key, path, body);
                return TAgentTaskStorage::OK;
            } else {
                return TAgentTaskStorage::FAILED;
            }
        }

        void ForceInitialization() {
            AtomicSet(Initialized, false);
        }

        const TString Prefix;
        NAsyncZookeeper::TClient& ZookeeperClient;
        IAgentTaskWatcher& Watcher;

        class TTaskBox {
        public:
            inline bool Create(TTaskHeader::TRef task) noexcept {
                if (Exists(task->GetKey())) {
                    return false;
                } else {
                    Expiring.Insert(task.Get());
                    Existing.Push(task.Get());
                    Living.PushBack(task.Release());
                    return true;
                }
            }

            inline bool Exists(const IAgentTask::TKey& key) noexcept {
                return Existing.Find(key) != Existing.End();
            }

            inline void Cleanup(IAgentTaskWatcher& watcher) noexcept {
                auto now(TInstant::Now());
                while (!Expiring.Empty()) {
                    TTaskHeader::TRef task(&(*Expiring.First()));
                    if (task->GetDeadline() > now) {
                        Y_UNUSED(task.Release());
                        break;
                    }
                    DEBUG_LOG << "Node '" << task->GetPath() << "' expired, scheduling deletion" << Endl;
                    UnlinkTask(*task, watcher);
                    Tombstones.PushBack(task.Release());
                }
            }

            inline void RemoveObsolete(const TString& prefix, TSet<TString>& actual, IAgentTaskWatcher& watcher) noexcept {
                TVector<TString> obsolete;

                auto& stored(Nodes[prefix]);
                SetDifference(stored.cbegin(), stored.cend(),
                              actual.cbegin(), actual.cend(),
                              std::back_inserter(obsolete));
                stored.swap(actual);

                for (const auto& name : obsolete) {
                    IAgentTask::TKey key;
                    if (GetGuid(name, key)) {
                        auto it(Existing.Find(key));
                        if (it != Existing.End()) {
                            TTaskHeader::TRef task(&(*it));
                            UnlinkTask(*task, watcher);
                        }
                    }
                }
            }

            inline TTaskHeader::TListType& GetTombstones() noexcept {
                return Tombstones;
            }

        private:
            inline void UnlinkTask(TTaskHeader& task, IAgentTaskWatcher& watcher) noexcept {
                DEBUG_LOG << "Unlinking node '" << task.GetPath() << "'" << Endl;
                watcher.Delete(task.GetKey());
                task.TAvlTreeItem::Unlink();
                Existing.Pop(task.GetKey());
                task.TIntrusiveListItem::Unlink();
            }

            TTaskHeader::TListType Living;
            TTaskHeader::TListType Tombstones;
            TTaskHeader::TMapType Existing;
            TTaskHeader::TTree Expiring;
            THashMap<TString, TSet<TString>> Nodes;
        };
        TPlainLockedBox<TTaskBox> Tasks;

        TPlainLockedBox<TSet<TString>> ShardsToReload;

        TFutureFence Fence;

        TAtomic Initialized = false;
        TAtomic InFlightRequests = 0;
    };

    TAgentTaskStorage::TAgentTaskStorage(const TString& prefix, IAgentTaskWatcher& watcher)
        : TAgentTaskStorage(prefix, watcher, !TLibrarySettings::Get()->GetZookeeperUri().empty())
    {
    }

    TAgentTaskStorage::TAgentTaskStorage(const TString& prefix, IAgentTaskWatcher& watcher, bool schedule)
        : Impl(MakeIntrusive<TImpl>(prefix, watcher))
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TAgentTaskStorage::~TAgentTaskStorage() {
    }

    NThreading::TFuture<TAgentTaskStorage::EResultCode> TAgentTaskStorage::Put(const IAgentTask& task) noexcept {
        return Impl->DumpNewNode(task.Key(), Encapsulate(task));
    }

    class TAgentTaskMover::TImpl {
    public:
        inline TImpl(TAgentTaskStorage& sourceQueue, TAgentTaskStorage& targetQueue,
                     const IAgentTask::TKey& sourceKey, const IAgentTask& targetTask)
            : SourcePath(sourceQueue.Impl->KeyToPath(sourceKey))
            , TargetPath(targetQueue.Impl->KeyToPath(targetTask.Key()))
            , Data(Encapsulate(targetTask))
            , TargetQueue(targetQueue)
        {
        }

        inline NThreading::TFuture<bool> Execute() const noexcept {
            NAsyncZookeeper::IAsyncOperation::TVectorType operations;
            operations.PushBack(new NAsyncZookeeper::TDeleteOperation(SourcePath));
            operations.PushBack(new NAsyncZookeeper::TCreateOperation(TargetPath, Data));
            // don't rate limit tasks finishing because user waits for them
            return TargetQueue.Impl->ZookeeperClient.Mutate(operations).Apply([](const NAsyncZookeeper::TVoidResult::TFuture& future) {
                return future.GetValue().ResultCode() == NAsyncZookeeper::OK;
            });
        }

    private:
        const TString SourcePath;
        const TString TargetPath;
        const TString Data;
        TAgentTaskStorage& TargetQueue;
    };

    TAgentTaskMover::TAgentTaskMover(TAgentTaskStorage& sourceQueue, TAgentTaskStorage& targetQueue,
                                     const IAgentTask::TKey& sourceKey, const IAgentTask& targetTask)
        : Impl(MakeHolder<TImpl>(sourceQueue, targetQueue, sourceKey, targetTask))
    {
    }

    TAgentTaskMover::~TAgentTaskMover() {
    }

    NThreading::TFuture<bool> TAgentTaskMover::Execute() const {
        return Impl->Execute();
    }
}
