#pragma once

#include <infra/netmon/library/scheduler.h>

#include <util/generic/guid.h>

namespace NNetmon {
    class IAgentTask {
    public:
        using TKey = TGUID;

        virtual ~IAgentTask() = default;

        virtual const TKey& Key() const noexcept = 0;
        virtual TInstant Deadline() const noexcept = 0;

        // used to dump data into zookeeper
        virtual void Dump(IOutputStream& stream) const noexcept = 0;
    };

    class IAgentTaskWatcher {
    public:
        virtual ~IAgentTaskWatcher() = default;

        virtual void Load(const IAgentTask::TKey& key, IInputStream& stream) noexcept = 0;
        virtual void Delete(const IAgentTask::TKey& key) noexcept = 0;
    };

    class TAgentTaskMover;

    class TAgentTaskStorage: public TNonCopyable {
    public:
        friend TAgentTaskMover;

        enum EResultCode {
            OK,
            FAILED,
            TOO_MANY_REQUESTS
        };

        TAgentTaskStorage(const TString& prefix, IAgentTaskWatcher& watcher);
        ~TAgentTaskStorage();

        NThreading::TFuture<EResultCode> Put(const IAgentTask& task) noexcept;

    private:
        TAgentTaskStorage(const TString& prefix, IAgentTaskWatcher& watcher, bool schedule);

        class TImpl;
        TIntrusivePtr<TImpl> Impl;

        TScheduledTask::TTaskGuard SchedulerGuard;
    };

    class TAgentTaskMover: public TNonCopyable {
    public:
        TAgentTaskMover(TAgentTaskStorage& sourceQueue, TAgentTaskStorage& targetQueue,
                        const IAgentTask::TKey& sourceKey, const IAgentTask& targetTask);

        ~TAgentTaskMover();

        NThreading::TFuture<bool> Execute() const;

    private:
        class TImpl;
        THolder<TImpl> Impl;
    };
}
