#include <infra/netmon/terminated_hosts.h>
#include <infra/netmon/response_gatherer.h>
#include <infra/netmon/idl/api.fbs.h>

namespace NNetmon {
    class TTerminatedHostsUpdater::TImpl: public TScheduledTask {
    public:
        TImpl()
            : TScheduledTask(TSettings::Get()->GetTerminatedHostsInterval())
        {
        }

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                DoSnapshot();
            });
        }

        inline void AddHost(const TTopology::THostRef host) {
            Storage.Add(host);
        }

        inline void RemoveHost(const TTopology::THostRef host) {
            Storage.Remove(host);
        }

        inline THostStorage::TState::TConstValueRef GetState() const {
            return TerminatedHosts.Get();
        }

    private:
        void DoSnapshot() {
            const auto changeSet(Storage.ResetChangeSet());
            if (!changeSet->Empty()) {
                Success = false;

                auto terminatedHosts(THostStorage::TState::Make(*TerminatedHosts.Own()));
                terminatedHosts->Apply(changeSet);
                TerminatedHosts.Swap(terminatedHosts);

                UpdateHosts();

                return;
            }

            // changeset is empty but previous update was failed
            if (!Success) {
                UpdateHosts();
            }
        }

        using TGatherer = TGatherer<NApi::TUpdateTerminatedHostsResponse>;

        void UpdateHosts() {
            Success = false;

            auto state(GetState());
            const auto& terminatedHosts(state->GetHosts());

            flatbuffers::FlatBufferBuilder builder;
            std::vector<NCommon::THost> convertedHosts;
            convertedHosts.reserve(terminatedHosts.size());
            for (auto hostId : terminatedHosts) {
                convertedHosts.emplace_back(hostId);
            }
            builder.Finish(NApi::CreateTUpdateTerminatedHostsRequest(builder, builder.CreateVectorOfStructs(convertedHosts)));

            TGatherer::Collect("/slicer/v1/update_terminated_hosts", builder).Subscribe([this](const NThreading::TFuture<TGatherer::TResponseList>& future) {
                try {
                    future.GetValue();
                    Success = true;
                } catch (...) {
                    ERROR_LOG << CurrentExceptionMessage() << Endl;
                }
            });
        }

        THostStorage Storage;
        THostStorage::TState::TBox TerminatedHosts;
        bool Success = true;
    };

    TTerminatedHostsUpdater::TTerminatedHostsUpdater()
        : Impl(MakeHolder<TImpl>())
        , SchedulerGuard(Impl->Schedule())
    {
    }

    TTerminatedHostsUpdater::~TTerminatedHostsUpdater() = default;

    void TTerminatedHostsUpdater::AddHost(const TTopology::THostRef host) {
        Impl->AddHost(host);
    }

    void TTerminatedHostsUpdater::RemoveHost(const TTopology::THostRef host) {
        Impl->RemoveHost(host);
    }

    THostStorage::TState::TConstValueRef TTerminatedHostsUpdater::GetState() const {
        return Impl->GetState();
    }

    class TTerminatedHostsMaintainer::TImpl {
    public:
        TImpl(const TTopologyStorage& topologyStorage)
            : TopologyStorage(topologyStorage)
        {
        }

        void UpdateHosts(const flatbuffers::Vector<const NCommon::THost*>& hosts) {
            auto newTerminatedHosts(MakeAtomicShared<TTopologyStorage::THostSet>(hosts.size()));
            for (const auto& host : hosts) {
                auto foundHost = TopologyStorage.FindHost(*host);
                if (foundHost && !newTerminatedHosts->contains(foundHost)) {
                    newTerminatedHosts->emplace(foundHost);
                }
            }
            TerminatedHosts.Swap(newTerminatedHosts);
        }

        TTopologyStorage::THostSetBox::TConstValueRef GetHosts() const {
            return TerminatedHosts.Get();
        }

    private:
        const TTopologyStorage& TopologyStorage;
        TTopologyStorage::THostSetBox TerminatedHosts;
    };

    TTerminatedHostsMaintainer::TTerminatedHostsMaintainer(const TTopologyStorage& topologyStorage)
        : Impl(MakeHolder<TImpl>(topologyStorage))
    {
    }

    TTerminatedHostsMaintainer::~TTerminatedHostsMaintainer() = default;

    void TTerminatedHostsMaintainer::UpdateHosts(const flatbuffers::Vector<const NCommon::THost*>& hosts) {
        Impl->UpdateHosts(hosts);
    }

    TTopologyStorage::THostSetBox::TConstValueRef TTerminatedHostsMaintainer::GetHosts() const {
        return Impl->GetHosts();
    }
}
