#include <infra/netmon/topology/clients/groups.h>
#include <infra/netmon/topology/settings.h>
#include <infra/netmon/library/requester.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/containers/intrusive_avl_tree/avltree.h>

#include <util/generic/hash_set.h>
#include <util/generic/set.h>
#include <util/random/random.h>
#include <util/string/builder.h>
#include <library/cpp/cgiparam/cgiparam.h>

namespace NNetmon {
    namespace {
        const size_t MAX_UPDATING_GROUPS = 3;
        const TDuration GROUP_REFRESH_INTERVAL = TDuration::Seconds(60 * 5);
        const TDuration GROUP_EMPTY_REFRESH_INTERVAL = TDuration::Seconds(5);
        const TDuration GROUP_LIFETIME = TDuration::Seconds(60 * 60);
        const TDuration GROUP_REFRESH_JITTER = TDuration::Seconds(5);

        template <class T>
        struct TGroupHashFunctor {
            size_t operator()(const typename T::TRef& x) const {
                return Hasher(x->GetKey());
            }
            size_t operator()(const TGroupKey& x) const {
                return Hasher(x);
            }

            THash<TGroupKey> Hasher;
        };

        template <class T>
        struct TGroupEqualFunctor {
            bool operator()(const typename T::TRef& lhs, const typename T::TRef& rhs) const {
                return lhs->GetKey() == rhs->GetKey();
            }
            bool operator()(const typename T::TRef& lhs, const TGroupKey& rhs) const {
                return lhs->GetKey() == rhs;
            }
        };

        class TGroup : public TAvlTreeItem<TGroup, TCompareUsingDeadline>,
                       public TIntrusiveListItem<TGroup>,
                       public TAtomicRefCount<TGroup> {
        public:
            using TRef = TIntrusivePtr<TGroup>;

            using TSetType = THashSet<TRef, TGroupHashFunctor<TGroup>, TGroupEqualFunctor<TGroup>>;
            using TListType = TIntrusiveList<TGroup>;
            using TTree = TAvlTree<TGroup, TCompareUsingDeadline>;

            using THostsBox = TGroupStorage::THostsBox;

            inline TGroup(const TGroupKey& key)
                : Key(key)
                , Created(TInstant::Now())
            {
            }

            inline const TGroupKey& GetKey() const noexcept {
                return Key;
            }
            inline const TInstant& GetDeadline() const noexcept {
                return State.Own()->Deadline;
            }

            inline bool SetHosts(THostsBox::TValueRef& hosts) noexcept {
                bool changed = false;
                Hosts.Apply([&] (const THostsBox::TConstValueRef& oldHosts) {
                    changed = *oldHosts != *hosts;
                    return hosts;
                });

                auto state(State.Own());
                state->Ready = true;
                state->Error.clear();
                state->Tries = 0;
                if (changed) {
                    state->Changed = TInstant::Now();
                }
                return changed;
            }
            inline void SetError(const TString& message) noexcept {
                auto state(State.Own());
                state->Ready = true;
                state->Error = message;
                state->Tries++;
            }

            inline bool IsExpired() const {
                return State.Own()->TouchTime < TInstant::Now();
            }

            void Touch() {
                State.Own()->TouchTime = GROUP_LIFETIME.ToDeadLine();
            }

            void Add(TListType& running) {
                TAvlTreeItem<TGroup, TCompareUsingDeadline>::Unlink();
                running.PushBack(this);
            }

            void Add(TTree& scheduled, bool now=false) {
                TIntrusiveListItem<TGroup>::Unlink();

                auto state(State.Own());
                if (now) {
                    state->Deadline = TInstant::Zero();
                } else {
                    TDuration interval = GROUP_REFRESH_INTERVAL;
                    if (!state->Ready) {
                        interval = Min(GROUP_EMPTY_REFRESH_INTERVAL + TDuration::Seconds(2 * state->Tries), interval);
                        interval += TDuration::MilliSeconds(RandomNumber<ui64>(GROUP_REFRESH_JITTER.MilliSeconds()));
                    } else {
                        // don't update loaded groups at same time across all instances
                        interval += TDuration::MilliSeconds(RandomNumber<ui64>((interval.MilliSeconds() / 2)));
                    }

                    state->Deadline = interval.ToDeadLine();
                }

                if (!state->Deleted) {
                    auto unguard = Unguard(state.GetGuard());
                    scheduled.Insert(this);
                }
            }

            NThreading::TFuture<void> Remove() {
                auto state(State.Own());
                Y_VERIFY(!state->Deleted);
                state->Deleted = true;
                TIntrusiveListItem<TGroup>::Unlink();
                TAvlTreeItem<TGroup, TCompareUsingDeadline>::Unlink();
                return state->Future.IgnoreResult();
            }

            NThreading::TFuture<bool> Update();

            inline TGroupStorage::TResult ToResult() const {
                auto state(State.Own());
                return {
                    state->Changed,
                    state->Ready,
                    state->Error,
                    Hosts.Get(),
                    Created
                };
            }

        private:
            struct TState {
                TState()
                    : TouchTime(GROUP_LIFETIME.ToDeadLine())
                    , Deadline(TInstant::Zero())
                    , Changed(TInstant::Zero())
                    , Ready(false)
                    , Deleted(false)
                    , Tries(0)
                    , Future(NThreading::MakeFuture(false))
                {
                }

                TInstant TouchTime;
                TInstant Deadline;
                TInstant Changed;
                bool Ready;
                bool Deleted;
                TString Error;
                ui64 Tries;

                NThreading::TFuture<bool> Future;
            };

            const TGroupKey Key;
            const TInstant Created;
            TPlainLockedBox<TState> State;
            THostsBox Hosts;
        };

        class TGroupUpdater: public TAtomicRefCount<TGroupUpdater> {
        public:
            using TRef = TIntrusivePtr<TGroupUpdater>;

            static NThreading::TFuture<bool> Start(const TGroup::TRef group) {
                DEBUG_LOG << "Start to update group '" << group->GetKey() << "'" << Endl;
                TRef ptr(MakeIntrusive<TGroupUpdater>(group));
                ptr->MakeRequest().Subscribe([ptr](const THttpRequester::TFuture& future_) {
                    try {
                        ptr->ParseResponse(future_.GetValue()->Data);
                    } catch (...) {
                        ptr->Group->SetError(CurrentExceptionMessage());
                        ERROR_LOG << "Unable to get hosts for group '" << ptr->Group->GetKey()
                                  << "': " << CurrentExceptionMessage() << Endl;
                    }
                    ptr->Promise.SetValue(ptr->Changed);
                });
                return ptr->Promise.GetFuture();
            }

            inline TGroupUpdater(const TGroup::TRef& group)
                : Group(group)
                , Promise(NThreading::NewPromise<bool>())
                , Changed(false)
            {
            }

        private:
            THttpRequester::TFuture MakeRequest() {
                TCgiParameters params;
                params.InsertEscaped("do", TStringBuf("1"));
                params.InsertEscaped("group_type", Group->GetKey().Type);
                params.InsertEscaped("group_name", Group->GetKey().Name);
                return THttpRequester::Get()->MakeRequest(
                    TStringBuilder() << TTopologySettings::Get()->GetJugglerUrl() << "/api/groups/request_group?" << params.Print()
                );
            }

            void ParseResponse(const TString& data) {
                NJson::TJsonValue rootObj;
                NJson::ReadJsonFastTree(data, &rootObj, true);
                if (!rootObj["success"].GetBooleanSafe()) {
                    if (rootObj["error"].GetStringSafe() == TStringBuf("group_not_cached")) {
                        // group not ready yet, do nothing
                        INFO_LOG << "Group '" << Group->GetKey() << "' isn't ready, let's wait" << Endl;
                        return;
                    } else {
                        ythrow yexception() << rootObj["error"].GetStringSafe();
                    }
                }

                TGroupStorage::THostsBox::TValueRef hosts(MakeAtomicShared<TGroupStorage::THosts>());
                for (const auto& rowObj : rootObj["group"]["instances_list"].GetArraySafe()) {
                    hosts->insert(rowObj[0].GetStringSafe());
                }

                INFO_LOG << hosts->size() << " hosts found for group '" << Group->GetKey() << "'" << Endl;

                Changed = Group->SetHosts(hosts);
            }

            TGroup::TRef Group;
            NThreading::TPromise<bool> Promise;
            bool Changed;
        };

        NThreading::TFuture<bool> TGroup::Update() {
            auto state(State.Own());
            if (!state->Deleted) {
                state->Future = TGroupUpdater::Start(this);
                return state->Future;
            } else {
                return NThreading::MakeFuture(false);
            }
        }
    }

    class TGroupStorage::TImpl : public TScheduledTask {
    public:
        TImpl()
            : TScheduledTask(TTopologySettings::Get()->GetTagsInterval())
            , EventHub(TVoidEventHub::Make())
        {
        }

        inline ~TImpl() {
            TVector<TThreadPool::TFuture> futuresToWait;
            {
                auto storage(Storage.Own());
                futuresToWait.reserve(storage->Running.Size());
                while (!storage->Running.Empty()) {
                    futuresToWait.emplace_back(storage->Running.PopBack()->Remove());
                }
            }
            WaitExceptionOrAll(futuresToWait).Wait();
        }

        TThreadPool::TFuture Run() override {
            return TThreadPool::Get()->Add([this]() {
                auto storage(Storage.Own());

                TInstant now(TInstant::Now());
                while (!storage->Scheduled.Empty() && storage->Running.Size() < MAX_UPDATING_GROUPS) {
                    if (ShouldStop()) {
                        ythrow yexception() << "Group request was interrupted";
                    }

                    TGroup::TRef group(&(*storage->Scheduled.First()));
                    if (group->GetDeadline() <= now) {
                        group->Add(storage->Running);
                        {
                            // future can execute our callback right away if result is ready, so let's
                            // release lock for some time
                            auto unguard = Unguard(storage.GetGuard());
                            group->Update().Subscribe([group, this](const NThreading::TFuture<bool>& future) {
                                if (future.GetValue()) {
                                    EventHub->Notify();
                                }
                                auto storage(Storage.Own());
                                group->Add(storage->Scheduled);
                            });
                        }
                    } else {
                        break;
                    }
                }

                for (auto it = storage->Groups.begin(); it != storage->Groups.end();) {
                    TGroup::TRef group(*it);
                    if (group->IsExpired()) {
                        storage->Groups.erase(it++);
                        group->Remove();
                        INFO_LOG << "Group '" << group->GetKey() << "' was removed" << Endl;
                    } else {
                        ++it;
                    }
                }
            }, false);
        }

        inline TResult FetchGroup(const TGroupKey& key) {
            auto storage(Storage.Own());

            TGroup::TRef group;
            const auto it(storage->Groups.find(key));
            if (it.IsEnd()) {
                group.Reset(MakeIntrusive<TGroup>(key));
                storage->Groups.emplace(group);
                group->Add(storage->Scheduled, true);
                Spin();

                INFO_LOG << "Group '" << group->GetKey() << "' created" << Endl;
            } else {
                group.Reset(*it);
                group->Touch();
            }

            return group->ToResult();
        }

        TExistingHosts ExistingGroups() const {
            auto storage(Storage.Own());

            TExistingHosts result;
            result.reserve(storage->Groups.size());

            for (const auto& group : storage->Groups) {
                const auto state(group->ToResult());
                result.emplace_back(group->GetKey(), state.Hosts);
            }

            return result;
        }

        const TVoidEventHub& OnChanged() const {
            return *EventHub;
        }

    private:
        struct TStorage {
            TGroup::TSetType Groups;
            TGroup::TTree Scheduled;
            TGroup::TListType Running;
        };
        TPlainLockedBox<TStorage> Storage;
        TVoidEventHub::TRef EventHub;
    };

    TGroupStorage::TGroupStorage(bool schedule)
        : Impl(MakeHolder<TImpl>())
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TGroupStorage::~TGroupStorage()
    {
    }

    TGroupStorage::TResult TGroupStorage::FetchGroup(const TGroupKey& key) const {
        return Impl->FetchGroup(key);
    }

    TGroupStorage::TExistingHosts TGroupStorage::ExistingGroups() const {
        return Impl->ExistingGroups();
    }

    const TVoidEventHub& TGroupStorage::OnChanged() const {
        return Impl->OnChanged();
    }
}

template <>
void Out<NNetmon::TGroupKey>(IOutputStream& stream,
                             TTypeTraits<NNetmon::TGroupKey>::TFuncParam key) {
    stream << key.Type << TStringBuf("%") << key.Name;
}
