#pragma once

#include <infra/netmon/library/helpers.h>
#include <infra/netmon/library/boxes.h>
#include <infra/netmon/library/scheduler.h>
#include <infra/netmon/library/event_hub.h>

#include <util/digest/multi.h>

namespace NNetmon {
    const TString SKYNET_GROUP_TYPE = "SKYLH";
    const TString NANNY_GROUP_TYPE = "NANNY";
    const TString GENCFG_GROUP_TYPE = "GENCFG";

    struct TGroupKey {
        static inline TGroupKey Skynet(const TString& name) {
            return TGroupKey(SKYNET_GROUP_TYPE, name);
        }

        static inline TGroupKey Nanny(const TString& name) {
            return TGroupKey(NANNY_GROUP_TYPE, name);
        }

        static inline TGroupKey Gencfg(const TString& name) {
            return TGroupKey(GENCFG_GROUP_TYPE, name);
        }

        TGroupKey(const TString& type, const TString& name)
            : Type(type)
            , Name(name)
        {
        }

        TGroupKey(const TGroupKey& key)
            : Type(key.Type)
            , Name(key.Name)
        {
        }

        bool operator<(const TGroupKey& other) const noexcept {
            return std::tie(Type, Name) < std::tie(other.Type, other.Name);
        }

        bool operator==(const TGroupKey& other) const noexcept {
            return Type == other.Type && Name == other.Name;
        }

        const TString Type;
        const TString Name;
    };

    class TGroupStorage : public TNonCopyable {
    public:
        using THosts = TSet<TString>;
        using THostsBox = TAtomicLockedBox<THosts>;
        using TExistingHosts = TVector<std::pair<TGroupKey, THostsBox::TConstValueRef>>;

        struct TResult {
            TInstant Changed;
            bool Ready;
            TString Error;
            THostsBox::TConstValueRef Hosts;
            TInstant Created;
        };

        TGroupStorage(bool schedule=true);
        ~TGroupStorage();

        TResult FetchGroup(const TGroupKey& key) const;
        TExistingHosts ExistingGroups() const;
        const TVoidEventHub& OnChanged() const;

    private:
        class TImpl;
        THolder<TImpl> Impl;

        TScheduledTask::TTaskGuard SchedulerGuard;
    };
}

template <>
class THash<NNetmon::TGroupKey> {
public:
    size_t operator()(const NNetmon::TGroupKey& key) const {
        return MultiHash(key.Type, key.Name);
    }
};
