#include <infra/netmon/topology/clients/staff.h>
#include <infra/netmon/topology/settings.h>
#include <infra/netmon/library/requester.h>

#include <library/cpp/json/json_reader.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <util/string/builder.h>
#include <util/generic/hash_set.h>
#include <util/generic/set.h>

namespace NNetmon {
    namespace {
        using TGroupIndexBox = TAtomicLockedBox<THashMultiMap<TString, TString>>;

        class TStaffClient: public TNonCopyable {
        public:
            TStaffClient(TAtomic& shouldStop)
                : GroupIndex(MakeAtomicShared<TGroupIndexBox::TValue>())
                , Promise(NThreading::NewPromise<void>())
                , ShouldStop(shouldStop)
            {
                MakeRequest().Subscribe([this](const THttpRequester::TFuture& future_) {
                    OnReady(future_);
                });
            }

            NThreading::TFuture<void> GetFuture() {
                return Promise.GetFuture();
            }

            void Swap(TGroupIndexBox& box) {
                box.Swap(GroupIndex);
            };

        private:
            THttpRequester::TFuture MakeRequest() {
                TCgiParameters params;
                params.InsertEscaped("_fields", TStringBuf("id,login,groups.group.url,department_group.ancestors.url,department_group.url"));
                params.InsertEscaped("_limit", ToString(Limit));
                params.InsertEscaped("_sort", TStringBuf("id"));
                if (OffsetId) {
                    params.InsertEscaped("_query", "id>" + ToString(OffsetId));
                }

                NHttp::TFetchOptions options;
                if (TTopologySettings::Get()->GetStaffToken()) {
                    options.SetOAuthToken(TTopologySettings::Get()->GetStaffToken());
                }
                return THttpRequester::Get()->MakeRequest(
                    TStringBuilder() << TTopologySettings::Get()->GetStaffUrl() << "/v3/persons?" << params.Print(),
                    {},
                    options
                );
            }

            bool ParseResponse(const TString& data) {
                NJson::TJsonValue root;
                NJson::ReadJsonFastTree(data, &root, true);

                const auto& result = root["result"].GetArraySafe();
                if (result.empty()) {
                    return false;
                }

                for (const auto& person : result) {
                    THashSet<TString> groupSet;
                    // groups field may be missed, so don't use GetArraySafe() here
                    for (const auto& group : person["groups"].GetArray()) {
                        if (group["group"].Has("url")) {
                            groupSet.emplace(group["group"]["url"].GetStringSafe());
                        }
                    }
                    // same for ancestors
                    for (const auto& department : person["department_group"]["ancestors"].GetArray()) {
                        groupSet.emplace(department["url"].GetStringSafe());
                    }
                    groupSet.emplace(person["department_group"]["url"].GetStringSafe());
                    for (const auto& group : groupSet) {
                        GroupIndex->emplace(group, person["login"].GetStringSafe());
                    }

                    OffsetId = Max(OffsetId, ui64(person["id"].GetUIntegerSafe()));
                }

                return true;
            }

            void OnReady(const THttpRequester::TFuture& future_) {
                if (ShouldStop) {
                    Promise.SetException("Staff request was interrupted");
                    return;
                }

                bool needMore = false;
                try {
                    needMore = ParseResponse(future_.GetValue()->Data);
                } catch (...) {
                    Promise.SetException(CurrentExceptionMessage());
                    return;
                }
                if (needMore) {
                    MakeRequest().Subscribe([this](const THttpRequester::TFuture& future_) {
                        OnReady(future_);
                    });
                } else {
                    DEBUG_LOG << "Staff index fetching took " << timer.Get() << Endl;
                    INFO_LOG << GroupIndex->size() << " persons found" << Endl;
                    Promise.SetValue();
                }
            }

            TGroupIndexBox::TValueRef GroupIndex;
            NThreading::TPromise<void> Promise;

            const size_t Limit = 1000;
            ui64 OffsetId = 0;

            TSimpleTimer timer;

            TAtomic& ShouldStop;
        };

        template <class T>
        inline TSet<TString> ExpandLoginsAndGroupsImpl(const TStaffStorage* storage, const T& groupsAndLogins) noexcept {
            TSet<TString> result;
            for (const auto& something : groupsAndLogins) {
                if (something.StartsWith('@')) {
                    TString groupName(TStringBuf(something).Skip(1));
                    for (const auto& login : storage->ExpandGroup(groupName)) {
                        result.emplace(login);
                    }
                } else {
                    result.emplace(something);
                }
            }
            return result;
        }
    }

    class TStaffStorage::TImpl : public TScheduledTask {
    public:
        TImpl()
            : TScheduledTask(TTopologySettings::Get()->GetStaffInterval(),
                             true,
                             TTopologySettings::Get()->GetStaffRetryInterval())
        {
        }

        TThreadPool::TFuture Run() override {
            auto client(MakeAtomicShared<TStaffClient>(ShouldStop()));
            return client->GetFuture().Apply([this, client](const NThreading::TFuture<void>& future_) mutable {
                future_.GetValue();
                client->Swap(GroupIndex);
            });
        }

        inline TVector<TString> ExpandGroup(const TString& groupName) noexcept {
            TVector<TString> result;
            auto groupIndex(GroupIndex.Own());
            const auto range(groupIndex->equal_range(groupName));
            for (auto it(range.first); it != range.second; ++it) {
                result.emplace_back(it->second);
            }
            return result;
        }

    private:
        TGroupIndexBox GroupIndex;
    };

    TStaffStorage::TStaffStorage()
        : TStaffStorage(!TTopologySettings::Get()->GetStaffUrl().empty())
    {
    }

    TStaffStorage::TStaffStorage(bool schedule)
        : Impl(MakeHolder<TImpl>())
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TStaffStorage::~TStaffStorage() {
    }

    TVector<TString> TStaffStorage::ExpandGroup(const TString& groupName) const noexcept {
        return Impl->ExpandGroup(groupName);
    }

    TSet<TString> TStaffStorage::ExpandLoginsAndGroups(const TVector<TString>& groupsAndLogins) const noexcept {
        return ExpandLoginsAndGroupsImpl(this, groupsAndLogins);
    }

    TSet<TString> TStaffStorage::ExpandLoginsAndGroups(const TSet<TString>& groupsAndLogins) const noexcept {
        return ExpandLoginsAndGroupsImpl(this, groupsAndLogins);
    }

    TThreadPool::TFuture TStaffStorage::SpinAndWait() noexcept {
        return Impl->SpinAndWait();
    }
}
