#include <infra/netmon/topology/clients/walle.h>
#include <infra/netmon/topology/metrics.h>
#include <infra/netmon/topology/settings.h>
#include <infra/netmon/library/requester.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/cgiparam/cgiparam.h>

#include <util/string/builder.h>

namespace NNetmon {
    namespace {
        template <class TCallback>
        NThreading::TFuture<void> OnResponse(const THttpRequester::TFuture& future,
                                             const TString& path,
                                             const TAtomic& shouldStop,
                                             TCallback callback,
                                             const TCgiParameters& params) {
            if (AtomicGet(shouldStop)) {
                return NThreading::MakeErrorFuture<void>(std::make_exception_ptr(
                    yexception() << "Walle request was interrupted"
                ));
            }

            try {
                NJson::TJsonValue rootObj;
                NJson::ReadJsonFastTree(future.GetValue()->Data, &rootObj, true);

                callback(rootObj["result"]);

                auto cursor = rootObj["next_cursor"];
                if (cursor.IsDefined() && !cursor.IsNull()) {
                    return MakePagedRequest(path, shouldStop, callback, params, cursor.GetStringRobust());
                } else {
                    return NThreading::MakeFuture();
                }
            } catch (...) {
                return NThreading::MakeErrorFuture<void>(std::current_exception());
            }
        }

        // partialResponseCallback is called for each response page.
        template <class TCallback>
        NThreading::TFuture<void> MakePagedRequest(const TString& path,
                                                   const TAtomic& shouldStop,
                                                   TCallback partialResponseCallback,
                                                   const TCgiParameters& params = {},
                                                   const TStringBuf cursor = TStringBuf("0")) {
            TCgiParameters paramsWithCursor(params);
            paramsWithCursor.InsertEscaped("limit", ::ToString(TWalleUpdater::RequestPageSize));
            paramsWithCursor.InsertEscaped("cursor", cursor);

            TString url = TTopologySettings::Get()->GetWalleUrl() + path + "?" + paramsWithCursor.Print();
            DEBUG_LOG << "Making a request to " << url << Endl;
            return THttpRequester::Get()->MakeRequest(url)
                .Apply([path, partialResponseCallback, shouldStop, params](const THttpRequester::TFuture& future) {
                    return OnResponse(future, path, shouldStop, partialResponseCallback, params);
                });
        }

        class TWalleClient: public TNonCopyable {
        public:
            TWalleClient(const TTopologyStorage& topologyStorage, TAtomic& shouldStop)
                : ShouldStop(shouldStop)
                , TopologyStorage(topologyStorage)
            {
            }

            NThreading::TFuture<void> UpdateHosts() {
                DeadHosts = MakeAtomicShared<TTopologyStorage::THostSet>();
                MaintenanceHosts = MakeAtomicShared<TTopologyStorage::THostSet>();
                TSimpleTimer blockedHostNamesTimer;

                return MakePagedRequest(
                    "/v1/dns/blocked-host-names",
                    ShouldStop,
                    std::bind(&TWalleClient::ParseBlockedHostNamesResponse, this, std::placeholders::_1)
                ).Apply([this, blockedHostNamesTimer](const NThreading::TFuture<void>& future) {
                    future.GetValue();
                    TUnistat::Instance().PushSignalUnsafe(ETopologySignals::WalleBlockedHostNamesFetchTime,
                                                          blockedHostNamesTimer.Get().MilliSeconds());

                    /*
                        We fetch all hosts to fix output of /v1/dns/blocked-host-names
                        because host may be in blocked-host-names, but has ready status in wall-e
                    */
                    TSimpleTimer hostsTimer;
                    return MakePagedRequest(
                        "/v1/hosts",
                        ShouldStop,
                        std::bind(&TWalleClient::ParseHostsResponse, this, std::placeholders::_1),
                        {{"strict", "true"},
                         {"fields", "name,status,state,inv"}}
                    ).Subscribe([this, hostsTimer](const NThreading::TFuture<void>& future) {
                        if (future.HasException()) {
                            return;
                        }

                        TUnistat::Instance().PushSignalUnsafe(ETopologySignals::WalleHostsFetchTime, hostsTimer.Get().MilliSeconds());
                        DEBUG_LOG << "Fetched hosts in " << hostsTimer.Get() << Endl;
                        INFO_LOG << DeadHosts->size() << " dead hosts found" << Endl;
                        INFO_LOG << MaintenanceHosts->size() << " hosts under maintenance found" << Endl;
                    });
                });
            }

            void SwapDeadHosts(TTopologyStorage::THostSetBox& box) {
                box.Swap(DeadHosts);
            };

            void SwapMaintenanceHosts(TTopologyStorage::THostSetBox& box) {
                box.Swap(MaintenanceHosts);
            };

        private:
            void ParseBlockedHostNamesResponse(NJson::TJsonValue& result) {
                std::size_t filtered = 0;
                std::size_t found = 0;

                for (const auto& hostObj : result.GetArraySafe()) {
                    found++;
                    const auto hostRef = TopologyStorage.FindHost(hostObj.GetStringSafe());
                    if (hostRef) {
                        DeadHosts->insert(hostRef);
                    } else {
                        filtered++;
                    }
                }
                DEBUG_LOG << found << " blocked host names found, " << filtered << " filtered" << Endl;
            }

            void ParseHostsResponse(NJson::TJsonValue& result) {
                std::size_t totalCount = 0;
                std::size_t notReadyCount = 0;
                std::size_t notReadyFiltered = 0;
                std::size_t maintenanceCount = 0;
                std::size_t maintenanceFiltered = 0;

                for (const auto& hostObj : result.GetArraySafe()) {
                    ++totalCount;
                    if (!hostObj.Has(TStringBuf("name"))) {
                        continue;
                    }
                    bool ready(hostObj["status"].GetStringSafe() == TStringBuf("ready") &&
                               hostObj["state"].GetStringSafe() != TStringBuf("probation"));
                    bool maintenance(hostObj["state"].GetStringSafe() == TStringBuf("maintenance"));
                    const auto hostRef = TopologyStorage.FindHost(hostObj["name"].GetStringSafe());
                    if (!hostRef) {
                        notReadyFiltered += !ready;
                        maintenanceFiltered += maintenance;
                        continue;
                    }

                    if (ready) {
                        DeadHosts->erase(hostRef);
                    } else {
                        DeadHosts->insert(hostRef);
                        ++notReadyCount;
                    }

                    if (maintenance) {
                        MaintenanceHosts->insert(hostRef);
                        ++maintenanceCount;
                    }
                }

                DEBUG_LOG << totalCount << " hosts found" << Endl;
                DEBUG_LOG << notReadyCount << " not ready hosts found, " << notReadyFiltered << " filtered" << Endl;
                DEBUG_LOG << maintenanceCount << " hosts under maintenance found, " << maintenanceFiltered << " filtered" << Endl;
            }

            const TAtomic& ShouldStop;
            const TTopologyStorage& TopologyStorage;
            TTopologyStorage::THostSetBox::TValueRef DeadHosts;
            TTopologyStorage::THostSetBox::TValueRef MaintenanceHosts;
        };
    }

    class TWalleUpdater::TImpl : public TScheduledTask {
    public:
        TImpl(const TTopologyStorage& topologyStorage,
              const TDuration& timeOffset,
              bool schedule)
            : TScheduledTask(TScheduledTask::TRoundToIntervalTag{},
                             TTopologySettings::Get()->GetWalleInterval(),
                             timeOffset,
                             true)
            , TopologyStorage(topologyStorage)
            , WalleClient(TopologyStorage, ShouldStop())
            , Ready(!schedule)
            , EventHub(TVoidEventHub::Make())
        {
        }

        TThreadPool::TFuture Run() override {
            return WalleClient.UpdateHosts().Subscribe([this](const NThreading::TFuture<void>& future) mutable {
                if (future.HasException()) {
                    return;
                }

                WalleClient.SwapDeadHosts(DeadHosts);

                TTopologyStorage::THostSetBox maintenanceHosts;
                WalleClient.SwapMaintenanceHosts(maintenanceHosts);
                auto maintenanceMap = MakeMaintenanceCounterMap(*maintenanceHosts.Get());
                MaintenanceCounterBox.Swap(maintenanceMap);

                AtomicSet(Ready, true);
                EventHub->Notify();
            });
        }

        inline TTopologyStorage::THostSetBox::TConstValueRef GetDeadHosts() const {
            return DeadHosts.Get();
        }

        void ReplaceDeadHosts(const TTopologyStorage::THostSet& hosts) {
            auto hostsPtr = MakeAtomicShared<TTopologyStorage::THostSet>(hosts);
            DeadHosts.Swap(hostsPtr);
        }

        inline THostCounter::TConstRef GetHostCountUnderMaintenance(TExpressionId expressionId) const {
            auto counterMap = MaintenanceCounterBox.Get();
            auto it = counterMap->find(expressionId);
            if (it.IsEnd()) {
                return MakeAtomicShared<THostCounter>();
            } else {
                return it->second;
            }
        }

        inline bool IsReady() const {
            return AtomicGet(Ready);
        }

        const TVoidEventHub& OnUpdated() const {
            return *EventHub;
        }

    private:
        using TMaintenanceCounterMap = THashMap<TExpressionId, THostCounter::TRef>;
        using TMaintenanceCounterBox = TAtomicLockedBox<TMaintenanceCounterMap>;

        TMaintenanceCounterBox::TValueRef MakeMaintenanceCounterMap(const TTopologyStorage::THostSet& hosts) {
            auto counterMap = MakeAtomicShared<TMaintenanceCounterMap>();
            auto selector(TopologyStorage.GetTopologySelector());

            for (const auto& host : hosts) {
                const auto* expressions = selector->FindSourceExpressions(host);
                if (!expressions) {
                    continue;
                }
                for (auto expressionId : *expressions) {
                    auto& counter = (*counterMap)[expressionId];
                    if (!counter) {
                        counter.Reset(MakeAtomicShared<THostCounter>());
                    }
                    ++counter->Total;
                    ++counter->ByDc[host.GetDatacenter()];
                    ++counter->ByQueue[host.GetLine()];
                }
            }
            return counterMap;
        }

        const TTopologyStorage& TopologyStorage;
        TWalleClient WalleClient;
        TTopologyStorage::THostSetBox DeadHosts;
        TMaintenanceCounterBox MaintenanceCounterBox;
        TAtomic Ready;

        TVoidEventHub::TRef EventHub;
    };

    TWalleUpdater::TWalleUpdater(const TTopologyStorage& topologyStorage,
                                 const TDuration& timeOffset)
        : TWalleUpdater(topologyStorage, timeOffset, !TTopologySettings::Get()->GetWalleUrl().empty())
    {
    }

    TWalleUpdater::TWalleUpdater(const TTopologyStorage& topologyStorage,
                                 const TDuration& timeOffset,
                                 bool schedule)
        : Impl(MakeHolder<TImpl>(topologyStorage, timeOffset, schedule))
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TWalleUpdater::~TWalleUpdater() {
    }

    TTopologyStorage::THostSetBox::TConstValueRef TWalleUpdater::GetDeadHosts() const {
        return Impl->GetDeadHosts();
    }

    void TWalleUpdater::ReplaceDeadHosts(const TTopologyStorage::THostSet& hosts) {
        return Impl->ReplaceDeadHosts(hosts);
    }

    TWalleUpdater::THostCounter::TConstRef TWalleUpdater::GetHostCountUnderMaintenance(TExpressionId expressionId) const {
        return Impl->GetHostCountUnderMaintenance(expressionId);
    }

    bool TWalleUpdater::IsReady() const {
        return Impl->IsReady();
    }

    const TVoidEventHub& TWalleUpdater::OnUpdated() const {
        return Impl->OnUpdated();
    }

    TThreadPool::TFuture TWalleUpdater::SpinAndWait() noexcept {
        return Impl->SpinAndWait();
    }
}
