#include <infra/netmon/library/api_handler_helpers.h>
#include <infra/netmon/topology/clients/walle.h>
#include <infra/netmon/topology/common_ut.h>
#include <infra/netmon/topology/settings.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>
#include <util/string/cast.h>

using namespace NNetmon;

namespace {
    struct TWalleContext : public TNonCopyable {
        ui64 LastInv = 0;
        TVector<ui64> RequestCursors;
    };

    class TWalleReply : public THttpReply<THttpDispatcher, TWalleContext> {
    public:
        TWalleReply(TWalleContext& serverContext, const TServiceRequest::TRef clientContext)
            : THttpReply<THttpDispatcher, TWalleContext>(serverContext, clientContext)
        {
        }

        void PreprocessRequest(THttpInput&) {
        }
        void ParseRequest(THttpInput&) {
        }

        void WriteResponse(THttpOutput& output) {
            output << THttpResponse()
                .SetContentType(TStringBuf("application/json"))
                .SetContent(Response.Str());
            output.Finish();
        }

    protected:
        NJsonWriter::TBuf Response;
    };

    class TBlockedHostNamesReply : public TWalleReply {
    public:
        using TWalleReply::TWalleReply;

        void Process() {
            Response
                .BeginObject()
                    .WriteKey("result")
                    .BeginList()
                        .WriteString("sas1-1234.search.yandex.net")
                        .WriteString("man1-1234.search.yandex.net")
                    .EndList()
                .EndObject();
        }
    };

    class THostsReply : public TWalleReply {
    public:
        using TWalleReply::TWalleReply;

        void Process() {
            Response
                .BeginObject()
                    .WriteKey("result")
                    .BeginList()
                        .BeginObject()
                        .WriteKey("inv").WriteInt(1)
                        .WriteKey("name")
                        .WriteString("man1-1234.search.yandex.net")
                        .WriteKey("status").WriteString("ready")
                        .WriteKey("state").WriteString("assigned")
                        .EndObject()

                        .BeginObject()
                        .WriteKey("inv").WriteInt(2)
                        .WriteKey("name")
                        .WriteString("vla1-1234.search.yandex.net")
                        .WriteKey("status").WriteString("manual")
                        .WriteKey("state").WriteString("maintenance")
                        .EndObject()
                    .EndList()
                .EndObject();
        }
    };

    class TErrorHostsReply : public TWalleReply {
    public:
        using TWalleReply::TWalleReply;

        void Process() {
        }
        void WriteResponse(THttpOutput& output) {
            output << THttpResponse()
                .SetHttpCode(HttpCodes::HTTP_INTERNAL_SERVER_ERROR);
            output.Finish();
        }
    };

    class TPagedHostsReply : public TWalleReply {
    public:
        using TWalleReply::TWalleReply;

        static constexpr ui64 PageSize = TWalleUpdater::RequestPageSize;
        static constexpr ui64 FullPageCount = 2;
        static constexpr ui64 LastPageSize = 1;

        void Process() {
            auto& context = GetServerContext();

            auto line = GetClientContext()->GetInput().FirstLine();
            auto params = TCgiParameters(TStringBuf(line).After('?').Before(' '));
            auto it = params.Find("cursor");
            auto cursor = (it == params.end() ? 0 : FromString<ui64>(it->second));
            context.RequestCursors.emplace_back(cursor);

            bool lastPage = (context.LastInv == PageSize * FullPageCount);
            auto iterStart = context.LastInv + 1;
            auto iterEnd = iterStart + (lastPage ? LastPageSize : PageSize);
            Response.BeginObject().WriteKey("result").BeginList();

            for (auto inv = iterStart; inv < iterEnd; ++inv) {
                Response
                    .BeginObject()
                        .WriteKey("inv").WriteInt(inv)
                        .WriteKey("name")
                        .WriteString("nonexistent-host.search.yandex.net")
                        .WriteKey("status").WriteString("dead")
                        .WriteKey("state").WriteString("free")
                    .EndObject();
            }

            Response.EndList();
            if (!lastPage) {
                Response.WriteKey("next_cursor").WriteULongLong(iterEnd);
            }
            Response.EndObject();

            context.LastInv = iterEnd - 1;
        }
    };
}

class TWalleTest: public TTestBase {
    UNIT_TEST_SUITE(TWalleTest);
    UNIT_TEST(TestResponseParsing)
    UNIT_TEST(TestResponseError)
    UNIT_TEST(TestPaging)
    UNIT_TEST_SUITE_END();

public:
    TWalleTest() {
    }

    void SetUp() override {
        Host = "localhost";
        Port = PortManager.GetPort(12345);
        TTopologySettings::Get()->SetWalleUrl(TStringBuilder() << "http://" << Host << ":" << Port);
    }

private:
    inline void TestResponseParsing() {
        TWalleContext context;
        THttpHandler<TBlockedHostNamesReply> blockedHostNamesService(context);
        THttpHandler<THostsReply> hostsService(context);

        auto server = TWebServer(THttpServerOptions().SetHost(Host).SetPort(Port).SetThreads(1));
        server.Add("/v1/dns/blocked-host-names", blockedHostNamesService);
        server.Add("/v1/hosts", hostsService);
        TWebServer::TThreadGuard serverGuard(server);

        auto& topologyStorage = TGlobalTopology::GetTopologyStorage();

        TWalleUpdater walle(topologyStorage, {});

        TAtomic updateCount = 0;
        auto guard = walle.OnUpdated().Subscribe([&updateCount]() {
            AtomicIncrement(updateCount);
        });

        auto future = walle.SpinAndWait();
        future.Wait();
        UNIT_ASSERT(!future.HasException());
        UNIT_ASSERT(walle.IsReady());
        UNIT_ASSERT_EQUAL(walle.GetDeadHosts()->size(), 2);
        UNIT_ASSERT(walle.GetDeadHosts()->contains(
                        topologyStorage.FindHost("sas1-1234.search.yandex.net")
                   ));
        UNIT_ASSERT(walle.GetDeadHosts()->contains(
                        topologyStorage.FindHost("vla1-1234.search.yandex.net")
                   ));
        auto counter = walle.GetHostCountUnderMaintenance(topologyStorage.DefaultExpressionId());
        UNIT_ASSERT_EQUAL(counter->Total, 1);
        UNIT_ASSERT_EQUAL(counter->ByDc.at(topologyStorage.FindDatacenter("vla")), 1);
        UNIT_ASSERT_EQUAL(counter->ByQueue.at(topologyStorage.FindQueue("vla-01")), 1);
        UNIT_ASSERT_EQUAL(updateCount, 1);

        // test ReplaceDeadHosts as well
        auto host = topologyStorage.FindHost("vla1-1234.search.yandex.net");
        walle.ReplaceDeadHosts({host});
        UNIT_ASSERT_EQUAL(walle.GetDeadHosts()->size(), 1);
        UNIT_ASSERT(walle.GetDeadHosts()->contains(host));
        UNIT_ASSERT_EQUAL(updateCount, 1); // updateCount hasn't changed
    }

    inline void TestResponseError() {
        TWalleContext context;
        THttpHandler<TBlockedHostNamesReply> blockedHostNamesService(context);
        THttpHandler<TErrorHostsReply> errorHostsService(context);

        auto server = TWebServer(THttpServerOptions().SetHost(Host).SetPort(Port).SetThreads(1));
        server.Add("/v1/dns/blocked-host-names", blockedHostNamesService);
        server.Add("/v1/hosts", errorHostsService);
        TWebServer::TThreadGuard serverGuard(server);

        auto& topologyStorage = TGlobalTopology::GetTopologyStorage();

        TWalleUpdater walle(topologyStorage, {});

        TAtomic updateCount = 0;
        auto guard = walle.OnUpdated().Subscribe([&updateCount]() {
            AtomicIncrement(updateCount);
        });

        auto future = walle.SpinAndWait();
        future.Wait();
        UNIT_ASSERT(future.HasException());
        UNIT_ASSERT(!walle.IsReady());
        UNIT_ASSERT(walle.GetDeadHosts()->empty());
        UNIT_ASSERT_EQUAL(
            walle.GetHostCountUnderMaintenance(topologyStorage.DefaultExpressionId())->Total, 0);
        UNIT_ASSERT_EQUAL(updateCount, 0);
    }

    inline void TestPaging() {
        TWalleContext context;
        THttpHandler<TBlockedHostNamesReply> blockedHostNamesService(context);
        THttpHandler<TPagedHostsReply> pagedHostsService(context);

        auto server = TWebServer(THttpServerOptions().SetHost(Host).SetPort(Port).SetThreads(1));
        server.Add("/v1/dns/blocked-host-names", blockedHostNamesService);
        server.Add("/v1/hosts", pagedHostsService);
        TWebServer::TThreadGuard serverGuard(server);

        auto& topologyStorage = TGlobalTopology::GetTopologyStorage();

        TWalleUpdater walle(topologyStorage, {});

        TAtomic updateCount = 0;
        auto guard = walle.OnUpdated().Subscribe([&updateCount]() {
            AtomicIncrement(updateCount);
        });

        auto future = walle.SpinAndWait();
        future.Wait();
        UNIT_ASSERT(!future.HasException());
        UNIT_ASSERT(walle.IsReady());

        TVector<ui64> expectedCursors;
        expectedCursors.emplace_back(0);
        for (ui64 index = 1; index <= TPagedHostsReply::FullPageCount; ++index) {
            expectedCursors.emplace_back(index * TPagedHostsReply::PageSize + 1);
        }
        UNIT_ASSERT_EQUAL(context.RequestCursors, expectedCursors);
        UNIT_ASSERT_EQUAL(updateCount, 1);
    }

    TPortManager PortManager;
    TString Host;
    ui16 Port;
};

UNIT_TEST_SUITE_REGISTRATION(TWalleTest);
