#include <infra/netmon/topology/expression_storage.h>
#include <infra/netmon/topology/settings.h>
#include <infra/netmon/library/api_client_helpers.h>
#include <infra/netmon/library/event_hub.h>
#include <infra/netmon/library/fences.h>
#include <infra/netmon/library/zookeeper.h>

#include <infra/netmon/idl/expression.fbs.h>

#include <util/generic/hash_set.h>
#include <util/generic/hash.h>
#include <util/generic/guid.h>
#include <util/random/random.h>

namespace NNetmon {
    namespace {
        template <class T>
        struct TIdHashFunctor {
            inline size_t operator()(const typename T::TRef& x) const noexcept {
                return ComputeHash(x->GetId());
            }
            inline size_t operator()(const TString& x) const noexcept {
                return ComputeHash(x);
            }
        };

        template <class T>
        struct TIdEqualFunctor {
            inline bool operator()(const typename T::TRef& lhs, const typename T::TRef& rhs) const noexcept {
                return lhs->GetId() == rhs->GetId();
            }
            inline bool operator()(const typename T::TRef& lhs, const TString& rhs) const noexcept {
                return lhs->GetId() == rhs;
            }
        };

        template <class T>
        struct TExpressionHashFunctor {
            inline size_t operator()(const typename T::TRef& x) const noexcept {
                return x->GetNormalForm().hash();
            }
            inline size_t operator()(const TExpressionDnf& x) const {
                return x.hash();
            }
            inline size_t operator()(const TExpressionId& x) const {
                return x;
            }
        };

        template <class T>
        struct TExpressionEqualFunctor {
            inline bool operator()(const typename T::TRef& lhs, const typename T::TRef& rhs) const noexcept {
                return lhs->GetNormalForm() == rhs->GetNormalForm();
            }
            inline bool operator()(const typename T::TRef& lhs, const TExpressionDnf& rhs) const noexcept {
                return lhs->GetNormalForm() == rhs;
            }
            inline bool operator()(const typename T::TRef& lhs, const TExpressionId& rhs) const noexcept {
                return lhs->GetNormalForm().hash() == rhs;
            }
        };

        inline TSet<TString> FromFlatBuffer(const flatbuffers::Vector<flatbuffers::Offset<flatbuffers::String>>* incoming) {
            TSet<TString> result;
            if (incoming) {
                for (const auto& owner : *incoming) {
                    result.emplace(owner->data(), owner->size());
                }
            }
            return result;
        }

        class TExpressionState : public TAtomicRefCount<TExpressionState> {
        public:
            using TRef = TIntrusivePtr<TExpressionState>;
            using TIdSet = THashSet<TRef, TIdHashFunctor<TExpressionState>, TIdEqualFunctor<TExpressionState>>;
            using TExpressionSet = THashMultiSet<TRef, TExpressionHashFunctor<TExpressionState>, TExpressionEqualFunctor<TExpressionState>>;

            TExpressionState(const TExpressionState& state)
                : Id(state.Id)
                , Expression(state.Expression)
                , NormalForm(state.NormalForm)
                , Owners(state.Owners)
                , Created(state.Created)
                , Modified(state.Modified)
                , Version(state.Version)
            {
            }

            TExpressionState(const TString& id, const TExpressionStorage::TChangeSet& changeSet)
                : Id(id)
                , Expression(changeSet.Expression.Defined() ? changeSet.Expression.GetRef() : Default<TString>())
                , NormalForm(ParseExpression(Expression))
                , Owners(changeSet.Owners.GetOrElse(Default<TSet<TString>>()))
                , Created(TInstant::Now())
                , Modified(TInstant::Now())
                , Version(0)
            {
            }

            TExpressionState(const TExpressionState& state, const TExpressionStorage::TChangeSet& changeSet)
                : Id(changeSet.Id.Defined() ? changeSet.Id.GetRef() : state.Id)
                , Expression(changeSet.Expression.Defined() ? changeSet.Expression.GetRef() : state.Expression)
                , NormalForm(changeSet.Expression.Defined() ? ParseExpression(Expression) : state.NormalForm)
                , Owners(changeSet.Owners.GetOrElse(state.Owners))
                , Created(state.Created)
                , Modified(TInstant::Now())
                , Version((changeSet.Id.Defined() && changeSet.Id.GetRef() != state.Id) ? 0 : state.Version)
            {
            }

            TExpressionState(const NExpression::TAliasedExpression& state, int version)
                : Id(state.Id()->data(), state.Id()->size())
                , Expression(state.Expression()->data(), state.Expression()->size())
                , NormalForm(ParseExpression(Expression))
                , Owners(FromFlatBuffer(state.Owners()))
                , Created(TInstant::Seconds(state.Created()))
                , Modified(TInstant::Seconds(state.Modified()))
                , Version(version)
            {
            }

            inline const TString& GetId() const noexcept {
                return Id;
            }

            inline const TExpressionDnf& GetNormalForm() const noexcept {
                return NormalForm;
            }

            inline int GetVersion() const noexcept {
                return Version;
            }

            inline flatbuffers::Offset<NExpression::TAliasedExpression> ToProto(
                    flatbuffers::FlatBufferBuilder& builder) const {
                std::vector<flatbuffers::Offset<flatbuffers::String>> owners;
                owners.reserve(Owners.size());
                for (const auto& owner : Owners) {
                    owners.emplace_back(builder.CreateString(owner));
                }
                return NExpression::CreateTAliasedExpression(
                    builder,
                    builder.CreateString(Id.data(), Id.size()),
                    builder.CreateString(Expression.data(), Expression.size()),
                    builder.CreateVector(owners),
                    Created.Seconds(),
                    Modified.Seconds(),
                    0,
                    false
                );
            }

            inline TString ToString() const {
                TString body;
                TStringOutput stream(body);
                flatbuffers::FlatBufferBuilder builder;
                builder.Finish(ToProto(builder));
                WriteFlatBuffer(&stream, builder);
                stream.Finish();
                return body;
            }

            inline TExpressionStorage::TResult ToResult() const {
                return {
                    GetId(),
                    Expression,
                    NormalForm.hash(),
                    Owners
                };
            }

        private:
            const TString Id;

            TString Expression;
            TExpressionDnf NormalForm;
            TSet<TString> Owners;

            const TInstant Created;
            TInstant Modified;

            const int Version;
        };
    }

    class TExpressionStorage::TImpl : public TScheduledTask, public NAsyncZookeeper::IWatcher, public TAtomicRefCount<TImpl> {
    public:
        using TRef = TIntrusivePtr<TImpl>;

        class TCreateOperation : public TNonCopyable {
        public:
            inline TCreateOperation(TImpl* parent, const TString& id, const TChangeSet& changeSet) noexcept
                : Parent(parent)
                , Id(id.empty() ? CreateGuidAsString() : id)
                , ChangeSet(changeSet)
                , Promise(NThreading::NewPromise<TResult>())
            {
            }

            NThreading::TFuture<TResult> Dispatch(TAtomicSharedPtr<TCreateOperation> self) {
                TExpressionState state(Id, ChangeSet);
                Parent->ZookeeperClient.Create(Parent->Join(Id), state.ToString()).Subscribe(
                    [self](const NAsyncZookeeper::TPathResult::TFuture& future) {
                        const auto rc(future.GetValue().ResultCode());
                        if (rc == NAsyncZookeeper::OK) {
                            self->Parent->SpinAndWait().Subscribe([self](const NThreading::TFuture<void>&) {
                                const auto expression(self->Parent->GetExpression(self->Id));
                                if (expression.Defined()) {
                                    self->Promise.SetValue(expression.GetRef());
                                } else {
                                    self->Promise.SetException(TStringBuilder() << "Can't find expression '" << self->Id << "'");
                                }
                            });
                        } else {
                            self->Promise.SetException(TStringBuilder() << "Can't create expression '" << self->Id << "', " << rc);
                        }
                    }
                );
                return Promise.GetFuture();
            }

        private:
            TImpl* Parent;
            const TString Id;
            const TChangeSet ChangeSet;
            NThreading::TPromise<TResult> Promise;
        };

        class TModifyOperation : public TNonCopyable {
        public:
            inline TModifyOperation(TImpl* parent, const TString& id, const TChangeSet& changeSet) noexcept
                : Parent(parent)
                , Id(id)
                , ChangeSet(changeSet)
                , Promise(NThreading::NewPromise<TResult>())
                , CanRetry(true)
            {
            }

            NThreading::TFuture<TResult> Dispatch(TAtomicSharedPtr<TModifyOperation> self) {
                const auto state(Parent->Storage.Own()->GetState(Id));
                if (state) {
                    TExpressionState modifiedState(*state, ChangeSet);
                    Parent->ZookeeperClient.SetData(Parent->Join(Id), modifiedState.ToString(), modifiedState.GetVersion()).Subscribe(
                        [self](const NAsyncZookeeper::TStatResult::TFuture& future) {
                            const auto rc(future.GetValue().ResultCode());
                            if (rc == NAsyncZookeeper::NODE_BAD_VERSION && self->CanRetry) {
                                self->CanRetry = false;
                                self->Parent->SpinAndWait().Subscribe([self](const NThreading::TFuture<void>&) {
                                    self->Dispatch(self);
                                });
                            } else if (rc == NAsyncZookeeper::OK) {
                                self->Parent->SpinAndWait().Subscribe([self](const NThreading::TFuture<void>&) {
                                    const auto expression(self->Parent->GetExpression(self->Id));
                                    if (expression.Defined()) {
                                        self->Promise.SetValue(expression.GetRef());
                                    } else {
                                        self->Promise.SetException(TStringBuilder() << "Can't find expression '" << self->Id << "'");
                                    }
                                });
                            } else {
                                self->Promise.SetException(TStringBuilder() << "Can't modify expression '" << self->Id << "', " << rc);
                            }
                        }
                    );
                } else {
                    Promise.SetException(TStringBuilder() << "Can't find expression '" << Id << "'");
                }
                return Promise.GetFuture();
            }

        private:
            TImpl* Parent;
            const TString Id;
            const TChangeSet ChangeSet;
            NThreading::TPromise<TResult> Promise;
            bool CanRetry;
        };

        class TRenameOperation : public TNonCopyable {
        public:
            inline TRenameOperation(TImpl* parent, const TString& oldId, const TString& newId, const TChangeSet& changeSet) noexcept
                : Parent(parent)
                , OldId(oldId)
                , NewId(newId)
                , ChangeSet(changeSet)
                , Promise(NThreading::NewPromise<TResult>())
                , CanRetry(true)
            {
            }

            NThreading::TFuture<TResult> Dispatch(TAtomicSharedPtr<TRenameOperation> self) {
                const auto state(Parent->Storage.Own()->GetState(OldId));
                if (state) {
                    TExpressionState modifiedState(*state, ChangeSet);
                    NAsyncZookeeper::IAsyncOperation::TVectorType operations;
                    operations.PushBack(new NAsyncZookeeper::TCreateOperation(Parent->Join(NewId), modifiedState.ToString()));
                    operations.PushBack(new NAsyncZookeeper::TDeleteOperation(Parent->Join(OldId), state->GetVersion()));
                    Parent->ZookeeperClient.Mutate(operations).Subscribe(
                        [self](const NAsyncZookeeper::TVoidResult::TFuture& future) {
                            const auto rc(future.GetValue().ResultCode());
                            if (rc == NAsyncZookeeper::NODE_BAD_VERSION && self->CanRetry) {
                                self->CanRetry = false;
                                self->Parent->SpinAndWait().Subscribe([self](const NThreading::TFuture<void>&) {
                                    self->Dispatch(self);
                                });
                            } else if (rc == NAsyncZookeeper::OK) {
                                self->Parent->SpinAndWait().Subscribe([self](const NThreading::TFuture<void>&) {
                                    const auto expression(self->Parent->GetExpression(self->NewId));
                                    if (expression.Defined()) {
                                        self->Promise.SetValue(expression.GetRef());
                                    } else {
                                        self->Promise.SetException(TStringBuilder() << "Can't find expression '" << self->NewId);
                                    }
                                });
                            } else {
                                self->Promise.SetException(TStringBuilder() << "Can't modify expression '" << self->OldId << "', " << rc);
                            }
                        }
                    );
                } else {
                    Promise.SetException(TStringBuilder() << "Can't find expression '" << OldId << "'");
                }
                return Promise.GetFuture();
            }

        private:
            TImpl* Parent;
            const TString OldId;
            const TString NewId;
            const TChangeSet ChangeSet;
            NThreading::TPromise<TResult> Promise;
            bool CanRetry;
        };

        class TDeleteOperation : public TNonCopyable {
        public:
            inline TDeleteOperation(TImpl* parent, const TString& id) noexcept
                : Parent(parent)
                , Id(id)
                , Promise(NThreading::NewPromise<void>())
                , CanRetry(true)
            {
            }

            NThreading::TFuture<void> Dispatch(TAtomicSharedPtr<TDeleteOperation> self) {
                const auto state(Parent->Storage.Own()->GetState(Id));
                if (state) {
                    Parent->ZookeeperClient.Delete(Parent->Join(Id), state->GetVersion()).Subscribe(
                        [self](const NAsyncZookeeper::TVoidResult::TFuture& future) {
                            const auto rc(future.GetValue().ResultCode());
                            if (rc == NAsyncZookeeper::NODE_BAD_VERSION && self->CanRetry) {
                                self->CanRetry = false;
                                self->Parent->SpinAndWait().Subscribe([self](const NThreading::TFuture<void>&) {
                                    self->Dispatch(self);
                                });
                            } else if (rc == NAsyncZookeeper::OK) {
                                self->Parent->SpinAndWait().Subscribe([self](const NThreading::TFuture<void>&) {
                                    self->Promise.SetValue();
                                });
                            } else {
                                self->Promise.SetException(
                                    TStringBuilder() << "Can't delete expression '" << self->Id << "', " << rc);
                            }
                        }
                    );
                } else {
                    Promise.SetException(TStringBuilder() << "Can't find expression '" << Id << "'");
                }
                return Promise.GetFuture();
            }

        private:
            TImpl* Parent;
            const TString Id;
            NThreading::TPromise<void> Promise;
            bool CanRetry;
        };

        TImpl(bool schedule)
            : TScheduledTask(TTopologySettings::Get()->GetTagsInterval(), true)
            , ZookeeperClient(*NAsyncZookeeper::TClient::Get())
            , Prefix("/expressions")
            , Ready(schedule ? false : true)
            , WatcherGuard(ZookeeperClient.CreateWatcher(Prefix, this))
            , EventHub(TVoidEventHub::Make())
        {
        }

        TThreadPool::TFuture Run() override {
            if (!AtomicGet(Initialized)) {
                Fence.Retain(ZookeeperClient.CreateRecursive(Prefix).Subscribe(
                    Y_MEMBER_REF_TO_CB(&TImpl::OnInitialized, this)
                ));
            } else if (AtomicGet(Changed)) {
                FetchAllChildrenData();
                AtomicSet(Changed, false);
            } else {
                FetchChangedChildren();
            }
            return Fence.Wait().Subscribe([this](const TThreadPool::TFuture&) {
                AtomicSet(Ready, true);
            });
        }

        void OnEvent(const NAsyncZookeeper::TEvent& event) override {
            using NAsyncZookeeper::EEventType;
            if (event.EventType == EEventType::ET_NODE_CHILDREN_CHANGED || event.EventType == EEventType::ET_NODE_DELETED) {
                AtomicSet(Changed, true);
                Spin();
            } else if (event.EventType == EEventType::ET_NODE_DATA_CHANGED) {
                TFsPath path(event.Path);
                Storage.Own()->ReloadExpression(path.GetName());
                Spin();
            }
        }

        void OnExpired() override {
            AtomicSet(Initialized, false);
            Spin();
        }

        inline bool IsReady() const {
            return AtomicGet(Ready);
        }

        inline TVector<TExpressionDnf> GetExpressionDnfs() const {
            TVector<TExpressionDnf> result;
            auto storage(Storage.Own());
            for (const auto& pair : storage->Expressions) {
                result.emplace_back(pair.first);
            }
            return result;
        }

        TMaybe<TExpressionDnf> GetExpressionDnf(TExpressionId expressionId) const {
            return Storage.Own()->GetDnf(expressionId);
        }

        const TVoidEventHub& OnChanged() const {
            return *EventHub;
        }

        inline NThreading::TFuture<void> DeleteExpression(const TString& id) {
            return ExecuteOperation<TDeleteOperation, void>(id);
        }

        inline NThreading::TFuture<TResult> UpsertExpression(const TString& id, const TChangeSet& changeSet) {
            if (changeSet.Id.Defined() && changeSet.Id.GetRef() != id) {
                return ExecuteOperation<TRenameOperation, TResult>(id, changeSet.Id.GetRef(), changeSet);
            } else if (id.empty() || !Storage.Own()->GetState(id)) {
                return ExecuteOperation<TCreateOperation, TResult>(id, changeSet);
            } else {
                return ExecuteOperation<TModifyOperation, TResult>(id, changeSet);
            }
        }

        TMaybe<TExpressionStorage::TResult> GetExpression(const TString& id) {
            const auto state(Storage.Own()->GetState(id));
            if (state) {
                return state->ToResult();
            } else {
                return {};
            }
        }

        TVector<TExpressionStorage::TResult> FindExpressions(TExpressionId expressionId) {
            TVector<TExpressionStorage::TResult> result;
            for (const auto& state : Storage.Own()->FindStates(expressionId)) {
                result.emplace_back(state->ToResult());
            }
            return result;
        }

        TVector<TExpressionStorage::TResult> ListExpressions() {
            const auto states(Storage.Own()->ListStates());
            TVector<TExpressionStorage::TResult> result;
            result.reserve(states.size());
            for (const auto& state : states) {
                result.emplace_back(state->ToResult());
            }
            return result;
        }

    private:
        template <class T, typename R, typename... Args>
        inline NThreading::TFuture<R> ExecuteOperation(Args&&... args) {
            auto self(MakeAtomicShared<T>(this, std::forward<Args>(args)...));
            return self->Dispatch(self);
        }

        inline TString Join(const TString& nodeName) const noexcept {
            return NAsyncZookeeper::Join(Prefix, nodeName);
        }

        void OnInitialized(const NAsyncZookeeper::TVoidResult& result) noexcept {
            if (result.ResultCode() == NAsyncZookeeper::OK) {
                AtomicSet(Initialized, true);
                FetchAllChildrenData();
                INFO_LOG << "'" << Prefix << "' is watched now" << Endl;
            } else {
                ERROR_LOG << "Can't watch on '" << Prefix << "', " << result.ResultCode() << Endl;
            }
        }

        void FetchAllChildrenData() noexcept {
            Fence.Retain(ZookeeperClient.GetChildrenData(Prefix, true, true).Subscribe(
                Y_MEMBER_REF_TO_CB(&TImpl::OnChildrenData, this)
            ));
        }

        void FetchChangedChildren() noexcept {
            const auto changedIds(Storage.Own()->PopChangedExpressions());
            for (const auto& id : changedIds) {
                TString path(Join(id));
                Fence.Retain(ZookeeperClient.GetData(path, true).Subscribe(
                    Y_MEMBER_REF_TO_CB(&TImpl::OnChildData, this, id)
                ));
            }
        }

        void OnChildData(const NAsyncZookeeper::TDataResult& result, const TString id) noexcept {
            if (result.ResultCode() == NAsyncZookeeper::OK) {
                DEBUG_LOG << "Reading expression '" << id << "' from zookeeper, version "
                          << result.Result().Stat().version << Endl;
                SaveExpression(id, result.Result());
            } else {
                ERROR_LOG << "Can't get child data on '" << Prefix << "', " << result.ResultCode() << Endl;
            }
        }

        void OnChildrenData(const NAsyncZookeeper::TChildrenDataResult& result) noexcept {
            if (result.ResultCode() == NAsyncZookeeper::OK) {
                SynchronizeExpressions(result.Result());
            } else {
                ERROR_LOG << "Can't get children data on '" << Prefix << "', " << result.ResultCode() << Endl;
            }
        }

        TExpressionState::TRef CreateExpression(const NAsyncZookeeper::TDataTuple& data) noexcept {
            TFlatObject<NExpression::TAliasedExpression> state;
            ReadFlatBuffer(data.Data(), state);
            if (!state.Verify()) {
                return nullptr;
            }

            TExpressionState::TRef expression(MakeIntrusive<TExpressionState>(
                *state.Get(),
                data.Stat().version
            ));

            return expression;
        }

        void SaveExpression(const TString& id, const NAsyncZookeeper::TDataTuple& data) {
            auto expression(CreateExpression(data));

            if (!expression) {
                WARNING_LOG << "Expression '" << id << "' is invalid" << Endl;
                return;
            }

            if (Storage.Own()->UpsertState(id, expression)) {
                DEBUG_LOG << "Expression '" << id << "' updated" << Endl;
                EventHub->Notify();
            }
        }

        void SynchronizeExpressions(const THashMap<TString, NAsyncZookeeper::TDataTuple>& nodes) noexcept {
            bool changed = false;

            // single point of truth is zookeeper, so sync with it
            // all modifications should be done in zookeeper, then replicated into memory
            auto storage(Storage.Own());

            TExpressionState::TIdSet existing(storage->StatesById);
            for (const auto& pair : nodes) {
                DEBUG_LOG << "Reading expression '" << pair.first << "' from zookeeper, version "
                          << pair.second.Stat().version << Endl;

                auto recentExpression(CreateExpression(pair.second));

                if (!recentExpression) {
                    WARNING_LOG << "Expression '" << pair.first << "' is invalid" << Endl;
                    continue;
                }

                if (storage->UpsertState(pair.first, recentExpression)) {
                    DEBUG_LOG << "Expression '" << pair.first << "' updated" << Endl;
                    changed = true;
                }

                existing.erase(recentExpression);
            }

            // remove expressions that missed in zookeeper
            for (const auto& expression : existing) {
                // do not remove default expression
                if (storage->DefaultExpressionId == expression->GetNormalForm().hash()) {
                    continue;
                }

                DEBUG_LOG << "Removing expression '" << expression->GetId() << "' from memory" << Endl;
                storage->RemoveState(expression);
                changed = true;
            }

            if (changed) {
                EventHub->Notify();
            }
        }

        struct TStorage {
            THashSet<TString> ExpressionsToReload;

            const TExpressionState::TRef DefaultState = MakeIntrusive<TExpressionState>("all", TChangeSet{Nothing(), "virtual=false", Nothing()});
            const TExpressionDnf DefaultExpression = ParseExpression("virtual=false");
            const TExpressionId DefaultExpressionId = DefaultExpression.hash();

            TExpressionState::TIdSet StatesById = {
                DefaultState
            };
            TExpressionState::TExpressionSet StatesByExpression {
                DefaultState
            };
            THashMap<TExpressionDnf, size_t> Expressions = {
                std::make_pair(DefaultExpression, 1),
            };

            void ReloadExpression(const TString& id) {
                ExpressionsToReload.emplace(id);
            }

            TVector<TString> PopChangedExpressions() {
                TVector<TString> result;
                result.insert(result.begin(), ExpressionsToReload.cbegin(), ExpressionsToReload.cend());
                ExpressionsToReload.clear();
                return result;
            }

            TMaybe<TExpressionDnf> GetDnf(TExpressionId expressionId) {
                const auto range(StatesByExpression.equal_range(expressionId));
                const auto it(range.first);
                if (!it.IsEnd()) {
                    return (*it)->GetNormalForm();
                } else {
                    return {};
                }
            }

            TExpressionState::TRef GetState(const TString& id) {
                const auto it(StatesById.find(id));
                return !it.IsEnd() ? it->Get() : nullptr;
            }

            TVector<TExpressionState::TRef> FindStates(TExpressionId expressionId) {
                TVector<TExpressionState::TRef> result;
                const auto range(StatesByExpression.equal_range(expressionId));
                for (auto it(range.first); it != range.second; ++it) {
                    result.emplace_back(*it);
                }
                return result;
            }

            TVector<TExpressionState::TRef> ListStates() {
                TVector<TExpressionState::TRef> result;
                result.reserve(StatesById.size());
                for (const auto& ptr : StatesById) {
                    result.emplace_back(ptr);
                }
                return result;
            }

            bool UpsertState(const TString& id, TExpressionState::TRef state) {
                auto it(StatesById.find(id));
                if (it.IsEnd()) {
                    // expression not exists in memory
                    InsertState(state);
                    return true;
                } else if (it->Get()->GetVersion() != state->GetVersion()) {
                    // expression in memory is outdated
                    RemoveState(*it);
                    InsertState(state);
                    return true;
                } else {
                    return false;
                }
            }

            void InsertState(TExpressionState::TRef state) {
                auto result(StatesById.emplace(state));
                Y_VERIFY(result.second);

                StatesByExpression.insert(state);
                Expressions[state->GetNormalForm()]++;
            }

            void RemoveStateByExpression(TExpressionState::TRef state) {
                const auto range(StatesByExpression.equal_range(state));
                for (auto it(range.first); it != range.second; ++it) {
                    if (it->Get()->GetId() == state->GetId()) {
                        StatesByExpression.erase(it);
                        return;
                    }
                }
                Y_VERIFY(false);
            }

            void RemoveFromExpressions(TExpressionState::TRef state) {
                auto it(Expressions.find(state->GetNormalForm()));
                Y_VERIFY(!it.IsEnd() && it->second > 0);
                it->second--;
                if (!it->second) {
                    Expressions.erase(it);
                }
            }

            void RemoveState(TExpressionState::TRef state) {
                auto removed(StatesById.erase(state));
                Y_VERIFY(removed > 0);

                RemoveStateByExpression(state);
                RemoveFromExpressions(state);
            }
        };

        NAsyncZookeeper::TClient& ZookeeperClient;
        const TString Prefix;

        TPlainLockedBox<TStorage> Storage;

        TFutureFence Fence;
        TAtomic Initialized = false;
        TAtomic Changed = false;
        TAtomic Ready;

        NAsyncZookeeper::TClient::TWatcherGuard WatcherGuard;
        TVoidEventHub::TRef EventHub;
    };

    TExpressionStorage::TExpressionStorage(bool schedule)
        : Impl(MakeIntrusive<TImpl>(schedule))
        , SchedulerGuard(schedule ? Impl->Schedule() : nullptr)
    {
    }

    TExpressionStorage::~TExpressionStorage() {
    }

    TVector<TExpressionDnf> TExpressionStorage::GetExpressionDnfs() const {
        return Impl->GetExpressionDnfs();
    }

    TMaybe<TExpressionDnf> TExpressionStorage::GetExpressionDnf(TExpressionId expressionId) const {
        return Impl->GetExpressionDnf(expressionId);
    }

    const TVoidEventHub& TExpressionStorage::OnChanged() const {
        return Impl->OnChanged();
    }

    TVector<TExpressionStorage::TResult> TExpressionStorage::List() const {
        return Impl->ListExpressions();
    }

    TMaybe<TExpressionStorage::TResult> TExpressionStorage::Get(const TString& id) const {
        return Impl->GetExpression(id);
    }

    TVector<TExpressionStorage::TResult> TExpressionStorage::Find(TExpressionId expressionId) const {
        return Impl->FindExpressions(expressionId);
    }

    NThreading::TFuture<TExpressionStorage::TResult> TExpressionStorage::Upsert(const TString& id, const TChangeSet& changeSet) const {
        return Impl->UpsertExpression(id, changeSet);
    }

    NThreading::TFuture<void> TExpressionStorage::Delete(const TString& id) const {
        return Impl->DeleteExpression(id);
    }

    bool TExpressionStorage::IsReady() const {
        return Impl->IsReady();
    }
}
