#include <infra/netmon/topology/settings.h>

#include <library/cpp/regex/pire/pire.h>

#include <util/generic/maybe.h>

namespace NNetmon {

    class TKnownQueuesFilter::TImpl {
    public:
        TImpl(const THashSet<TString>& knownQueues,
              const TString& knownQueuesRegex)
            : KnownQueues(knownQueues)
        {
            if (knownQueuesRegex) {
                Scanner = NPire::TLexer(knownQueuesRegex)
                    .Parse()
                    .Compile<NPire::TScanner>();
            }
        }

        bool Check(const TString& queue) const {
            if (TTopologySettings::Get()->GetUsePodAsQueue()) {
                return true;
            }
            if (Scanner && NPire::Matches(*Scanner, queue)) {
                return true;
            }
            return KnownQueues.contains(queue);
        }

    private:
        const THashSet<TString>& KnownQueues;
        TMaybe<Pire::Scanner> Scanner;
    };

    TKnownQueuesFilter::TKnownQueuesFilter(const THashSet<TString>& knownQueues,
                                           const TString& knownQueuesRegex)
        : Impl(MakeHolder<TImpl>(knownQueues, knownQueuesRegex))
    {
    }

    TKnownQueuesFilter::~TKnownQueuesFilter() = default;
    TKnownQueuesFilter& TKnownQueuesFilter::operator=(TKnownQueuesFilter&& other) = default;

    bool TKnownQueuesFilter::Check(const TString& queue) const {
        return Impl->Check(queue);
    }

    TTopologySettings::TTopologySettings()
        : TopologyFile("topology.msgpack.gz")
        , SandboxTopologyUrl("http://sandbox.yandex-team.ru/api/v1.0/resource?type=NETMON_TOPOLOGY&state=READY&limit=1&order=-id")
        , SandboxToken("")
        , WalleUrl("https://api.wall-e.yandex-team.ru")
        , JugglerUrl("http://juggler.search.yandex.net:8998")
        , StaffUrl("https://staff-api.yandex-team.ru")
        , StaffToken("")

        , ConstantSwitchWeights(false)

        , WalleInterval(TDuration::Seconds(300))
        , StaffInterval(TDuration::Hours(4))
        , StaffRetryInterval(TDuration::Minutes(10))
        , TagsInterval(TDuration::Seconds(5))
        , TopologyRebuildInterval(TDuration::Minutes(1))
        , TopologyUpdateInterval(TDuration::Minutes(5))

        , KnownDatacenters({"sas", "vla", "man", "iva", "myt"})

        , QueueMapping({
            {"sas1.1.1", "sas1.1"}, {"sas1.1.2", "sas1.1"}, {"sas1.1.3", "sas1.1"}, {"sas1.1.4", "sas1.1"},
            {"sas1.2.1", "sas1.2"}, {"sas1.2.2", "sas1.2"},
            {"sas1.3.1", "sas1.3"}, {"sas1.3.2", "sas1.3"}, {"sas1.3.3", "sas1.3"}, {"sas1.3.4", "sas1.3"},
            {"sas1.4.1", "sas1.4"}, {"sas1.4.2", "sas1.4"}, {"sas1.4.3", "sas1.4"}, {"sas1.4.4", "sas1.4"},
            {"sas2.1.1", "sas2.1"}, {"sas2.1.2", "sas2.1"}, {"sas2.1.3", "sas2.1"}, {"sas2.1.4", "sas2.1"},
            {"sas2.3.1", "sas2.3"}, {"sas2.3.2", "sas2.3"}, {"sas2.3.3", "sas2.3"}, {"sas2.3.4", "sas2.3"},
            {"sas2.4.1", "sas2.4"}, {"sas2.4.2", "sas2.4"}, {"sas2.4.3", "sas2.4"}, {"sas2.4.4", "sas2.4"}
        })
        , KnownQueues({
            // iva
            "iva1", "iva2", "iva3", "iva4", "iva5", "iva6", "iva7", "iva8",

            // myt
            "myt1", "myt2", "myt3", "myt4", "myt5", "myt6",

            // sas
            "sas1.1", "sas1.2", "sas1.3", "sas1.4", "sas2.1", "sas2.3", "sas2.4",
            "sas-08", "sas-09",

            // man
            "man1", "man2", "man3", "man4", "man5",

            // vla
            "vla-01", "vla-02", "vla-03", "vla-04", "vla-05"
            })
        , KnownQueuesFilter(KnownQueues, KnownQueuesRegex)
#ifdef NOC_SLA_BUILD
        , UsePodAsQueue(true)
#else
        , UsePodAsQueue(false)
#endif
    {
    }

    void TTopologySettings::Load(const NConfig::TConfig& config) {
        SandboxTopologyUrl = config["sandbox_topology_url"].As<TString>(SandboxTopologyUrl);
        SandboxToken = config["sandbox_token"].As<TString>(SandboxToken);
        S3TopologyUrl = config["s3_topology_url"].As<TString>(S3TopologyUrl);
        WalleUrl = config["walle_url"].As<TString>(WalleUrl);
        JugglerUrl = config["juggler_url"].As<TString>(JugglerUrl);
        StaffUrl = config["staff_url"].As<TString>(StaffUrl);
        StaffToken = config["staff_token"].As<TString>(StaffToken);
        if (!config["staff_interval"].IsNull()) {
            TDuration::TryParse(config["staff_interval"].As<TString>(), StaffInterval);
        }
        if (!config["staff_retry_interval"].IsNull()) {
            TDuration::TryParse(config["staff_retry_interval"].As<TString>(), StaffRetryInterval);
        }

        TDuration::TryParse(config["topology_update_interval"].As<TString>(), TopologyUpdateInterval);

        ConstantSwitchWeights = config["constant_switch_weights"].As<bool>(ConstantSwitchWeights);

        if (!config["known_dcs"].IsNull()) {
            // clear default datacenters
            KnownDatacenters.clear();

            for (const auto& element : config["known_dcs"].Get<NConfig::TArray>()) {
                KnownDatacenters.emplace(element.As<TString>());
            }
        }
        Y_VERIFY(!KnownDatacenters.empty());

        if (!config["queue_mapping"].IsNull()) {
            // clear default mapping
            QueueMapping.clear();

            for (const auto& rule : config["queue_mapping"].Get<NConfig::TDict>()) {
                const auto& newQueue(rule.first);
                if (!newQueue.empty()) {
                    for (const auto& queue : rule.second.Get<NConfig::TArray>()) {
                        const auto oldQueue(queue.As<TString>());
                        if (!oldQueue.empty()) {
                            QueueMapping[oldQueue] = newQueue;
                        }
                    }
                }
            }
        }

        if (!config["known_queues"].IsNull()) {
            // clear default queues
            KnownQueues.clear();

            for (const auto& element : config["known_queues"].Get<NConfig::TArray>()) {
                KnownQueues.emplace(element.As<TString>());
            }
        }

        KnownQueuesRegex = config["known_queues_regex"].As<TString>(KnownQueuesRegex);

        Y_VERIFY(!KnownQueues.empty() || !KnownQueuesRegex.empty());
        KnownQueuesFilter = TKnownQueuesFilter(KnownQueues, KnownQueuesRegex);

        SetUsePodAsQueue(config["use_pod_as_queue"].As<bool>(UsePodAsQueue));
    }
}
