#pragma once

#include <infra/netmon/library/settings.h>

#include <util/datetime/base.h>
#include <util/generic/singleton.h>
#include <util/generic/string.h>
#include <util/generic/hash_set.h>

class IInputStream;

namespace NNetmon {
    class TKnownQueuesFilter {
    public:
        TKnownQueuesFilter(const THashSet<TString>& knownQueues,
                           const TString& knownQueuesRegex);
        ~TKnownQueuesFilter();
        TKnownQueuesFilter& operator=(TKnownQueuesFilter&& other);

        bool Check(const TString& queue) const;

    private:
        class TImpl;
        THolder<TImpl> Impl;
    };

    class TTopologySettings {
        Y_DECLARE_SINGLETON_FRIEND()
    public:
        const TString& GetTopologyFile() const {
            return TopologyFile;
        }
        void SetTopologyFile(const TString& path) {
            TopologyFile = path;
        }

        const TString& GetSandboxTopologyUrl() const {
            return SandboxTopologyUrl;
        }
        const TString& GetSandboxToken() const {
            return SandboxToken;
        }
        const TString& GetS3TopologyUrl() const {
            return S3TopologyUrl;
        }
        const TString& GetWalleUrl() const {
            return WalleUrl;
        }
        void SetWalleUrl(const TString& url) {
            WalleUrl = url;
        }
        const TString& GetJugglerUrl() const {
            return JugglerUrl;
        }
        const TString& GetStaffUrl() const {
            return StaffUrl;
        }
        void SetStaffUrl(const TString& url) {
            StaffUrl = url;
        }
        const TString& GetStaffToken() const {
            return StaffToken;
        }

        bool GetConstantSwitchWeights() const {
            return ConstantSwitchWeights;
        }

        const TDuration& GetWalleInterval() const {
            return WalleInterval;
        }
        const TDuration& GetStaffInterval() const {
            return StaffInterval;
        }
        const TDuration& GetStaffRetryInterval() const {
            return StaffRetryInterval;
        }
        const TDuration& GetTagsInterval() const {
            return TagsInterval;
        }
        const TDuration& GetTopologyRebuildInterval() const {
            return TopologyRebuildInterval;
        }
        const TDuration& GetTopologyUpdateInterval() const {
            return TopologyUpdateInterval;
        }

        const THashSet<TString>& GetKnownDatacenters() const {
            return KnownDatacenters;
        }

        const auto& GetQueueMapping() const {
            return QueueMapping;
        }

        const TKnownQueuesFilter& GetKnownQueuesFilter() const {
            return KnownQueuesFilter;
        }
        void SetKnownQueues(const THashSet<TString>& value) {
            KnownQueues = value;
            // no need to recreate KnownQueuesFilter here
        }
        void SetKnownQueuesRegex(const TString& value) {
            KnownQueuesRegex = value;
            KnownQueuesFilter = TKnownQueuesFilter(KnownQueues, KnownQueuesRegex);
        }

        bool GetUsePodAsQueue() const {
            return UsePodAsQueue;
        }
        void SetUsePodAsQueue(bool value) {
            UsePodAsQueue = value;

            DcMapping.clear();
            if (UsePodAsQueue) {
                // https://st.yandex-team.ru/NETMON-448
                DcMapping["sas"] = {"sas1", "sas2"};
                // https://st.yandex-team.ru/NETMON-482
                // https://st.yandex-team.ru/NETMON-609
                DcMapping["vla"] = {"vla", "vlx", "noclab"};
            }
        }

        inline const TStringBuf GetRawDc(const TStringBuf& dc) const {
            // https://st.yandex-team.ru/NETMON-448
            if (dc == "sas1" || dc == "sas2") {
                return "sas";
            }
            // https://st.yandex-team.ru/NETMON-482
            if (dc == "vla" || dc == "vlx" || dc == "noclab") {
                return "vla";
            }
            return dc;
        }

        const TVector<TString>& GetMappedDcs(const TString& dc) const {
            auto& mapping = DcMapping[dc];
            if (mapping.empty()) {
                mapping = {dc};
            }
            return mapping;
        }

        void Load(const NConfig::TConfig& config);

        static TTopologySettings* Get() {
            return SingletonWithPriority<TTopologySettings, 100000>();
        }

    private:
        TTopologySettings();

        TString TopologyFile;
        TString SandboxTopologyUrl;
        TString SandboxToken;
        TString S3TopologyUrl;
        TString WalleUrl;
        TString JugglerUrl;
        TString StaffUrl;
        TString StaffToken;

        bool ConstantSwitchWeights;

        TDuration WalleInterval;
        TDuration StaffInterval;
        TDuration StaffRetryInterval;
        TDuration TagsInterval;
        TDuration TopologyRebuildInterval;
        TDuration TopologyUpdateInterval;

        THashSet<TString> KnownDatacenters;

        THashMap<TString, TString> QueueMapping;

        THashSet<TString> KnownQueues;
        TString KnownQueuesRegex;
        TKnownQueuesFilter KnownQueuesFilter;

        bool UsePodAsQueue;
        mutable THashMap<TString, TVector<TString>> DcMapping;
    };
}
